package com.bringspring.ai.controller;

import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.bringspring.ai.enmu.DifyCategoryEnum;
import com.bringspring.ai.service.DifyDatasetService;
import com.bringspring.ai.vo.DifyDataSetConfigVO;
import com.bringspring.ai.vo.DifyDataSetSynchronousVO;
import com.bringspring.common.annotation.NotCheckLogin;
import com.bringspring.common.base.ActionResult;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.system.base.entity.SysConfigEntity;
import com.bringspring.system.base.service.SysConfigService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.*;

/**
 * Dify知识库
 */
@Slf4j
@RestController
@RequestMapping("/ai/difyconfig/dataset")
public class DifyDataSetConfigController {

    @Autowired
    private SysConfigService sysConfigService;

    @Autowired
    private DifyDatasetService difyDatasetService;

    /**
     * 查询列表
     * @return
     */
    @NotCheckLogin
    @GetMapping
    public ActionResult list(){
        List<SysConfigEntity> list = sysConfigService.list(Wrappers.lambdaQuery(SysConfigEntity.class).eq(SysConfigEntity::getCategory, DifyCategoryEnum.KNOWLEDGE_BASE.getCode()));
        List<DifyDataSetConfigVO> result = new ArrayList<>();
        for (SysConfigEntity sys : list) {
            result = JsonUtil.getJsonToList(sys.getKeyValue(), DifyDataSetConfigVO.class);
        }

        return ActionResult.success(result);
    }

    /**
     * 根据知识库id查询配置
     * @param datasetId 数据集Id
     * @return
     */
    @NotCheckLogin
    @GetMapping("/{datasetId}")
    public ActionResult<DifyDataSetConfigVO> get(@PathVariable("datasetId") String datasetId){
        DifyDataSetConfigVO rs = new DifyDataSetConfigVO();
        List<SysConfigEntity> list = sysConfigService.list(Wrappers.lambdaQuery(SysConfigEntity.class).eq(SysConfigEntity::getCategory, DifyCategoryEnum.KNOWLEDGE_BASE.getCode()));
        if(CollectionUtil.isNotEmpty(list)){
            SysConfigEntity entity = list.get(0);
            List<DifyDataSetConfigVO> result = JsonUtil.getJsonToList(entity.getKeyValue(), DifyDataSetConfigVO.class);
            Optional<DifyDataSetConfigVO> first = result.stream().filter(item -> datasetId.equals(item.getDataSetId())).findFirst();
            if (first.isPresent()) {
                rs = first.get();
            }
        }
        return ActionResult.success(rs);
    }

    /**
     * 创建知识库及空文档
     * @param difyDataSetConfigVO 知识库参数集合
     * @return
     */
    @NotCheckLogin
    @PostMapping
    public ActionResult create(@RequestBody DifyDataSetConfigVO difyDataSetConfigVO){
        List<SysConfigEntity> list = sysConfigService.list(Wrappers.lambdaQuery(SysConfigEntity.class).eq(SysConfigEntity::getCategory, DifyCategoryEnum.KNOWLEDGE_BASE.getCode()));
        if(CollectionUtil.isNotEmpty(list)){
            SysConfigEntity entity = list.get(0);
            List<DifyDataSetConfigVO> result = JsonUtil.getJsonToList(entity.getKeyValue(), DifyDataSetConfigVO.class);
            result.add(difyDataSetConfigVO);


            // 调用Dify
            String datasetId = difyDatasetService.createDataset(difyDataSetConfigVO.getDataSetName(), difyDataSetConfigVO.getDescription());
            String documentId = difyDatasetService.createDocument(datasetId, difyDataSetConfigVO.getDocumentName());

            difyDataSetConfigVO.setDataSetId(datasetId);
            difyDataSetConfigVO.setDocumentId(documentId);

            String jsonObj = JsonUtil.getObjectToString(result);
            SysConfigEntity saveEntity = new SysConfigEntity();
            saveEntity.setCategory(DifyCategoryEnum.KNOWLEDGE_BASE.getCode());
            saveEntity.setName(DifyCategoryEnum.KNOWLEDGE_BASE.getDescribe());
            saveEntity.setKeyName(DifyCategoryEnum.KNOWLEDGE_BASE.getCode());
            saveEntity.setKeyValue(jsonObj);
            sysConfigService.create(saveEntity);
        }

        return ActionResult.success(MsgCode.SU005.get());
    }

    /**
     * 修改知识库配置
     * @param difyDataSetConfigVO
     * @return
     */
    @NotCheckLogin
    @PutMapping
    public ActionResult update(@RequestBody DifyDataSetConfigVO difyDataSetConfigVO){
        List<SysConfigEntity> list = sysConfigService.list(Wrappers.lambdaQuery(SysConfigEntity.class).eq(SysConfigEntity::getCategory, DifyCategoryEnum.KNOWLEDGE_BASE.getCode()));
        if(CollectionUtil.isNotEmpty(list)){
            SysConfigEntity entity = list.get(0);
            List<DifyDataSetConfigVO> result = JsonUtil.getJsonToList(entity.getKeyValue(), DifyDataSetConfigVO.class);
            result.removeIf(item -> difyDataSetConfigVO.getDataSetId().equals(item.getDataSetId()));
            result.add(difyDataSetConfigVO);

            String jsonObj = JsonUtil.getObjectToString(result);
            SysConfigEntity saveEntity = new SysConfigEntity();
            saveEntity.setCategory(DifyCategoryEnum.KNOWLEDGE_BASE.getCode());
            saveEntity.setName(DifyCategoryEnum.KNOWLEDGE_BASE.getDescribe());
            saveEntity.setKeyName(DifyCategoryEnum.KNOWLEDGE_BASE.getCode());
            saveEntity.setKeyValue(jsonObj);
            sysConfigService.create(saveEntity);
        }

        return ActionResult.success(MsgCode.SU005.get());
    }

    /**
     * 根据知识库ID删除
     * @param datasetId
     * @return
     */
    @DeleteMapping("{datasetId}")
    public ActionResult delete(@PathVariable("datasetId") String datasetId){
        List<SysConfigEntity> list = sysConfigService.list(Wrappers.lambdaQuery(SysConfigEntity.class).eq(SysConfigEntity::getCategory, DifyCategoryEnum.KNOWLEDGE_BASE.getCode()));
        if(CollectionUtil.isNotEmpty(list)){
            SysConfigEntity entity = list.get(0);
            List<DifyDataSetConfigVO> result = JsonUtil.getJsonToList(entity.getKeyValue(), DifyDataSetConfigVO.class);
            result.removeIf(item -> datasetId.equals(item.getDataSetId()));

            String jsonObj = JsonUtil.getObjectToString(result);
            SysConfigEntity saveEntity = new SysConfigEntity();
            saveEntity.setCategory(DifyCategoryEnum.KNOWLEDGE_BASE.getCode());
            saveEntity.setName(DifyCategoryEnum.KNOWLEDGE_BASE.getDescribe());
            saveEntity.setKeyName(DifyCategoryEnum.KNOWLEDGE_BASE.getCode());
            saveEntity.setKeyValue(jsonObj);
            sysConfigService.create(saveEntity);
        }

        return ActionResult.success(MsgCode.SU005.get());
    }

    /**
     * 同步数据
     * @param data
     * @return
     */
    @NotCheckLogin
    @PostMapping("/synchronous")
    public ActionResult synchronous(@RequestBody DifyDataSetSynchronousVO data){
        String jsonData = JsonUtil.getObjectToString(data.getData());
        difyDatasetService.invokeDatasetSynchronous(data.getDataSetId(),data.getDocumentId(),jsonData);
        return ActionResult.success(MsgCode.SU005.get());
    }
}
