package com.bringspring.ai.controller;

import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.bringspring.ai.configuration.DifyConfig;
import com.bringspring.ai.enmu.DifyCategoryEnum;
import com.bringspring.ai.service.DifyDatasetService;
import com.bringspring.ai.vo.DifyBaseConfigVO;
import com.bringspring.ai.vo.DifyDataSetConfigVO;
import com.bringspring.ai.vo.DifyDataSetSynchronousVO;
import com.bringspring.common.annotation.NotCheckLogin;
import com.bringspring.common.base.ActionResult;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.system.base.entity.SysConfigEntity;
import com.bringspring.system.base.service.SysConfigService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

/**
 * Dify基础配置
 */
@Slf4j
@RestController
@RequestMapping("/ai/difyconfig/base")
public class DifyBaseConfigController {

    @Autowired
    private SysConfigService sysConfigService;

    @Autowired
    private DifyConfig difyConfig;

    /**
     * 查询列表
     * @return
     */
    @NotCheckLogin
    @GetMapping("/list")
    public ActionResult list(){
        List<SysConfigEntity> list = sysConfigService.list(Wrappers.lambdaQuery(SysConfigEntity.class).eq(SysConfigEntity::getCategory, DifyCategoryEnum.BASE_CONFIG.getCode()));
        List<DifyBaseConfigVO> result = new ArrayList<>();
        for (SysConfigEntity sys : list) {
            DifyBaseConfigVO vo = JsonUtil.getJsonToBean(sys.getKeyValue(), DifyBaseConfigVO.class);
            result.add(vo);
        }

        return ActionResult.success(result);
    }

    /**
     * 根据keyName查询配置
     * @param keyName 键
     * @return
     */
    @NotCheckLogin
    @GetMapping("/{keyName}")
    public ActionResult<DifyBaseConfigVO> get(@PathVariable("keyName") String keyName){
        DifyBaseConfigVO result = null;
        List<SysConfigEntity> list = sysConfigService.list(
                Wrappers.lambdaQuery(SysConfigEntity.class)
                        .eq(SysConfigEntity::getCategory, DifyCategoryEnum.BASE_CONFIG.getCode())
                        .eq(SysConfigEntity::getKeyName, keyName)
        );
        if(CollectionUtil.isNotEmpty(list)){
            SysConfigEntity entity = list.get(0);
            result = JsonUtil.getJsonToBean(entity.getKeyValue(), DifyBaseConfigVO.class);
        }
        return ActionResult.success(result);
    }

    /**
     * 创建基础配置
     * @param baseConfigVO 基础配置参数集合
     * @return
     */
    @NotCheckLogin
    @PostMapping
    public ActionResult create(@RequestBody DifyBaseConfigVO baseConfigVO){
        List<SysConfigEntity> list = sysConfigService.list(
                Wrappers.lambdaQuery(SysConfigEntity.class)
                        .eq(SysConfigEntity::getCategory, DifyCategoryEnum.BASE_CONFIG.getCode())
                        .eq(SysConfigEntity::getKeyName, baseConfigVO.getKeyName())
        );

        if(CollectionUtil.isNotEmpty(list)){
            return ActionResult.fail(MsgCode.EXIST002.get());
        }

        String jsonObj = JsonUtil.getObjectToString(baseConfigVO);
        SysConfigEntity saveEntity = new SysConfigEntity();
        saveEntity.setCategory(DifyCategoryEnum.BASE_CONFIG.getCode());
        saveEntity.setName(baseConfigVO.getName());
        saveEntity.setKeyName(baseConfigVO.getKeyName());
        saveEntity.setKeyValue(jsonObj);
        sysConfigService.create(saveEntity);

        return ActionResult.success(MsgCode.SU005.get());
    }

    /**
     * 修改基础配置
     * @param baseConfigVO
     * @return
     */
    @NotCheckLogin
    @PutMapping
    public ActionResult update(@RequestBody DifyBaseConfigVO baseConfigVO){
        List<SysConfigEntity> list = sysConfigService.list(
                Wrappers.lambdaQuery(SysConfigEntity.class)
                        .eq(SysConfigEntity::getCategory, DifyCategoryEnum.BASE_CONFIG.getCode())
                        .eq(SysConfigEntity::getKeyName, baseConfigVO.getKeyName())
        );

        if(CollectionUtil.isEmpty(list)){
            return ActionResult.fail(MsgCode.FA001.get());
        }

        SysConfigEntity saveEntity = list.get(0);
        String jsonObj = JsonUtil.getObjectToString(baseConfigVO);
        saveEntity.setName(baseConfigVO.getName());
        saveEntity.setKeyName(baseConfigVO.getKeyName());
        saveEntity.setKeyValue(jsonObj);
        sysConfigService.saveOrUpdate(saveEntity);

        return ActionResult.success(MsgCode.SU005.get());
    }

    /**
     * 根据键删除
     * @param keyName
     * @return
     */
    @DeleteMapping("{keyName}")
    public ActionResult delete(@PathVariable("keyName") String keyName){
        List<SysConfigEntity> list = sysConfigService.list(
                Wrappers.lambdaQuery(SysConfigEntity.class)
                        .eq(SysConfigEntity::getCategory, DifyCategoryEnum.BASE_CONFIG.getCode())
                        .eq(SysConfigEntity::getKeyName, keyName)
        );

        if(CollectionUtil.isEmpty(list)){
            return ActionResult.success(MsgCode.SU005.get());
        }

        sysConfigService.removeById(list.get(0).getId());

        return ActionResult.success(MsgCode.SU005.get());
    }

    /**
     * 查询默认配置
     * @return
     */
    @NotCheckLogin
    @GetMapping("/getDefaultConfig")
    public ActionResult getDefaultConfig(){
        DefaultConfig defaultConfig = new DefaultConfig(difyConfig.getDifyBaseUrl(),difyConfig.getApiKey());
        return ActionResult.success(defaultConfig);
    }

    record DefaultConfig(String difyBaseUrl,String apiKey){}
}
