package com.bringspring.ai.controller;

import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.core.toolkit.Wrappers;
import com.bringspring.ai.enmu.DifyCategoryEnum;
import com.bringspring.ai.service.DifyDatasetService;
import com.bringspring.ai.vo.DifyDataSetConfigVO;
import com.bringspring.ai.vo.DifyDataSetSynchronousVO;
import com.bringspring.common.annotation.NotCheckLogin;
import com.bringspring.common.base.ActionResult;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.system.base.entity.SysConfigEntity;
import com.bringspring.system.base.service.SysConfigService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.*;

/**
 * Dify知识库
 */
@Slf4j
@RestController
@RequestMapping("/ai/difyconfig/dataset")
public class DifyDataSetConfigController {

    @Autowired
    private SysConfigService sysConfigService;

    @Autowired
    private DifyDatasetService difyDatasetService;

    /**
     * 查询列表
     * @return
     */
    @NotCheckLogin
    @GetMapping("/list")
    public ActionResult list(){
        List<SysConfigEntity> list = sysConfigService.list(Wrappers.lambdaQuery(SysConfigEntity.class).eq(SysConfigEntity::getCategory, DifyCategoryEnum.KNOWLEDGE_BASE.getCode()));
        List<DifyDataSetConfigVO> result = new ArrayList<>();
        for (SysConfigEntity sys : list) {
            DifyDataSetConfigVO vo = JsonUtil.getJsonToBean(sys.getKeyValue(), DifyDataSetConfigVO.class);
            result.add(vo);
        }

        return ActionResult.success(result);
    }

    /**
     * 根据keyName查询配置
     * @param keyName 键
     * @return
     */
    @NotCheckLogin
    @GetMapping("/{keyName}")
    public ActionResult<DifyDataSetConfigVO> get(@PathVariable("keyName") String keyName){
        DifyDataSetConfigVO rs = null;
        List<SysConfigEntity> list = sysConfigService.list(
                Wrappers.lambdaQuery(SysConfigEntity.class)
                        .eq(SysConfigEntity::getCategory, DifyCategoryEnum.KNOWLEDGE_BASE.getCode())
                        .eq(SysConfigEntity::getKeyName, keyName)
        );
        if(CollectionUtil.isNotEmpty(list)){
            SysConfigEntity entity = list.get(0);
            rs = JsonUtil.getJsonToBean(entity.getKeyValue(), DifyDataSetConfigVO.class);
        }
        return ActionResult.success(rs);
    }

    /**
     * 创建知识库及空文档
     * @param difyDataSetConfigVO 知识库参数集合
     * @return
     */
    @NotCheckLogin
    @PostMapping
    public ActionResult create(@RequestBody DifyDataSetConfigVO difyDataSetConfigVO){
        List<SysConfigEntity> list = sysConfigService.list(
                Wrappers.lambdaQuery(SysConfigEntity.class)
                        .eq(SysConfigEntity::getCategory, DifyCategoryEnum.KNOWLEDGE_BASE.getCode())
                        .eq(SysConfigEntity::getKeyName, difyDataSetConfigVO.getKeyName())
        );

        if(CollectionUtil.isNotEmpty(list)){
            return ActionResult.fail(MsgCode.EXIST002.get());
        }

        // 调用Dify
        String datasetId = difyDatasetService.createDataset(difyDataSetConfigVO.getDataSetName(), difyDataSetConfigVO.getDescription());
        String documentId = difyDatasetService.createDocument(datasetId, difyDataSetConfigVO.getDocumentName());

        difyDataSetConfigVO.setDataSetId(datasetId);
        difyDataSetConfigVO.setDocumentId(documentId);

        String jsonObj = JsonUtil.getObjectToString(difyDataSetConfigVO);
        SysConfigEntity saveEntity = new SysConfigEntity();
        saveEntity.setCategory(DifyCategoryEnum.KNOWLEDGE_BASE.getCode());
        saveEntity.setName(difyDataSetConfigVO.getName());
        saveEntity.setKeyName(difyDataSetConfigVO.getKeyName());
        saveEntity.setKeyValue(jsonObj);
        sysConfigService.create(saveEntity);

        return ActionResult.success(MsgCode.SU005.get());
    }

    /**
     * 修改知识库配置
     * @param difyDataSetConfigVO
     * @return
     */
    @NotCheckLogin
    @PutMapping
    public ActionResult update(@RequestBody DifyDataSetConfigVO difyDataSetConfigVO){
        List<SysConfigEntity> list = sysConfigService.list(
                Wrappers.lambdaQuery(SysConfigEntity.class)
                        .eq(SysConfigEntity::getCategory, DifyCategoryEnum.KNOWLEDGE_BASE.getCode())
                        .eq(SysConfigEntity::getKeyName, difyDataSetConfigVO.getKeyName())
        );

        if(CollectionUtil.isEmpty(list)){
            return ActionResult.fail(MsgCode.FA001.get());
        }

        SysConfigEntity saveEntity = list.get(0);

        String jsonObj = JsonUtil.getObjectToString(difyDataSetConfigVO);
        saveEntity.setName(difyDataSetConfigVO.getName());
        saveEntity.setKeyName(difyDataSetConfigVO.getKeyName());
        saveEntity.setKeyValue(jsonObj);
        sysConfigService.create(saveEntity);

        return ActionResult.success(MsgCode.SU005.get());
    }

    /**
     * 根据keyName删除
     * @param keyName
     * @return
     */
    @DeleteMapping("{keyName}")
    public ActionResult delete(@PathVariable("keyName") String keyName){
        List<SysConfigEntity> list = sysConfigService.list(
                Wrappers.lambdaQuery(SysConfigEntity.class)
                        .eq(SysConfigEntity::getCategory, DifyCategoryEnum.KNOWLEDGE_BASE.getCode())
                        .eq(SysConfigEntity::getKeyName, keyName)
        );

        if(CollectionUtil.isEmpty(list)){
            return ActionResult.success(MsgCode.SU005.get());
        }

        sysConfigService.removeById(list.get(0).getId());

        return ActionResult.success(MsgCode.SU005.get());
    }

    /**
     * 同步数据
     * @param data
     * @return
     */
    @NotCheckLogin
    @PostMapping("/synchronous")
    public ActionResult synchronous(@RequestBody DifyDataSetSynchronousVO data){
        String jsonData = JsonUtil.getObjectToString(data.getData());
        difyDatasetService.invokeDatasetSynchronous(data.getDataSetId(),data.getDocumentId(),jsonData);
        return ActionResult.success(MsgCode.SU005.get());
    }
}
