package com.bringspring.ai.config;

import com.bringspring.ai.service.McpBaseServer;
import org.noear.solon.Solon;
import org.noear.solon.ai.chat.tool.MethodToolProvider;
import org.noear.solon.ai.mcp.server.McpServerEndpointProvider;
import org.noear.solon.ai.mcp.server.annotation.McpServerEndpoint;
import org.noear.solon.ai.mcp.server.prompt.MethodPromptProvider;
import org.noear.solon.ai.mcp.server.resource.MethodResourceProvider;
import org.noear.solon.web.servlet.SolonServletFilter;
import org.springframework.aop.support.AopUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.annotation.AnnotationUtils;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import java.util.List;

/**
 * 这个类独立一个目录，可以让 Solon 扫描范围最小化
 * 如果需要启用，请配置config。mcp.enabled=true
 * @author zhujunjie
 *
 **/
@ConditionalOnProperty(name = "config.mcp.enabled", havingValue = "true", matchIfMissing = false)
@Configuration
public class McpServerConfig {

    @PostConstruct
    public void start() {
        Solon.start(McpServerConfig.class, new String[]{"--cfg=application-mcp.yml"});
    }

    @PreDestroy
    public void stop() {
        if (Solon.app() != null) {
            Solon.stopBlock(false, Solon.cfg().stopDelay());
        }
    }

    /**
     * 初始化服务端点.自动注入实现了 McpBaseServer 接口的 bean 并注册为服务端点
     * @param serverEndpoints
     * @return
     */
    @Bean
    public McpServerConfig init(List<McpBaseServer> serverEndpoints) {
        //提取实现容器里 McpBaseServer 接口的 bean ，并注册为服务端点
        for (McpBaseServer serverEndpoint : serverEndpoints) {
            Class<?> serverEndpointClz = AopUtils.getTargetClass(serverEndpoint);
            McpServerEndpoint anno = AnnotationUtils.findAnnotation(serverEndpointClz, McpServerEndpoint.class);

            if (anno == null) {
                continue;
            }

            McpServerEndpointProvider serverEndpointProvider = McpServerEndpointProvider.builder()
                    .from(serverEndpointClz, anno)
                    .build();

            serverEndpointProvider.addTool(new MethodToolProvider(serverEndpointClz, serverEndpoint));
            serverEndpointProvider.addResource(new MethodResourceProvider(serverEndpointClz, serverEndpoint));
            serverEndpointProvider.addPrompt(new MethodPromptProvider(serverEndpointClz, serverEndpoint));

            serverEndpointProvider.postStart();

            //可以再把 serverEndpointProvider 手动转入 SpringBoot 容器
        }

        //为了能让这个 init 能正常运行
        return this;
    }

    @Bean
    public FilterRegistrationBean mcpServerFilter() {
        FilterRegistrationBean<SolonServletFilter> filter = new FilterRegistrationBean<>();
        filter.setName("SolonFilter");
        filter.addUrlPatterns("/mcp/*");
        SolonServletFilter solonServletFilter = new SolonServletFilter();
        filter.setFilter(solonServletFilter);
        return filter;
    }
}