package com.bringspring.app.controller;

import com.bringspring.app.entity.AppDataEntity;
import com.bringspring.app.model.AppDataCrForm;
import com.bringspring.app.model.AppDataListAllVO;
import com.bringspring.app.model.AppDataListVO;
import com.bringspring.app.model.AppFlowListAllVO;
import com.bringspring.app.service.AppDataService;
import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.vo.ListVO;
import com.bringspring.common.base.vo.PaginationVO;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.common.util.StringUtils;
import com.bringspring.workflow.engine.model.flowengine.FlowPagination;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.validation.Valid;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * app常用数据
 *
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2021-07-08
 */
@Tag(name = "APP常用数据", description = "移动端应用数据管理，包括常用数据、流程、应用等")
@RestController
@RequestMapping("/api/app/Data")
public class AppDataController {

    @Autowired
    private AppDataService appDataService;

    /**
     * 常用数据
     *
     * @return
     */
    @Operation(summary = "获取常用数据列表", description = "根据类型获取APP常用数据列表")
    @ApiResponses({
        @ApiResponse(responseCode = "200", description = "成功获取常用数据列表"),
        @ApiResponse(responseCode = "400", description = "参数错误"),
        @ApiResponse(responseCode = "500", description = "服务器内部错误")
    })
    @GetMapping
    public ActionResult list(
        @Parameter(description = "数据类型") String type) {
        List<AppDataEntity> list = appDataService.getList(type);
        List<AppDataListVO> data = JsonUtil.getJsonToList(list, AppDataListVO.class);
        ListVO listVO = new ListVO();
        listVO.setList(data);
        return ActionResult.success(listVO);
    }

    /**
     * 新建
     *
     * @param appDataCrForm dto实体
     * @return
     */
    @Operation(summary = "添加常用数据", description = "添加新的APP常用数据")
    @ApiResponses({
        @ApiResponse(responseCode = "200", description = "常用数据添加成功"),
        @ApiResponse(responseCode = "400", description = "参数验证失败"),
        @ApiResponse(responseCode = "409", description = "常用数据已存在"),
        @ApiResponse(responseCode = "500", description = "服务器内部错误")
    })
    @PostMapping
    public ActionResult create(
        @Parameter(description = "常用数据表单", required = true) @RequestBody @Valid AppDataCrForm appDataCrForm) {
        AppDataEntity entity = JsonUtil.getJsonToBean(appDataCrForm, AppDataEntity.class);
        if (appDataService.isExistByObjectId(entity.getObjectId())) {
            return ActionResult.fail("常用数据已存在");
        }
        appDataService.create(entity);
        return ActionResult.success("添加成功");
    }

    /**
     * 删除
     *
     * @param objectId 对象主键
     * @return
     */
    @Operation(summary = "删除常用数据", description = "根据对象ID删除常用数据")
    @ApiResponses({
        @ApiResponse(responseCode = "200", description = "常用数据删除成功"),
        @ApiResponse(responseCode = "404", description = "常用数据不存在"),
        @ApiResponse(responseCode = "500", description = "服务器内部错误")
    })
    @DeleteMapping("/{objectId}")
    public ActionResult delete(
        @Parameter(description = "对象ID", required = true) @PathVariable("objectId") String objectId) {
        AppDataEntity entity = appDataService.getInfo(objectId);
        if (entity != null) {
            appDataService.delete(entity);
            return ActionResult.success("移除成功");
        }
        return ActionResult.fail(MsgCode.FA003.get());
    }

    /**
     * 所有流程
     *
     * @return
     */
    @Operation(summary = "所有流程")
    @GetMapping("/getFlowList")
    public ActionResult getFlowList(FlowPagination pagination) {
        List<AppFlowListAllVO> list = appDataService.getFlowList(pagination);
        PaginationVO paginationVO = JsonUtil.getJsonToBean(pagination, PaginationVO.class);
        return ActionResult.page(list, paginationVO);
    }

    /**
     * 所有应用
     *
     * @return
     */
    @Operation(summary = "所有应用")
    @GetMapping( path = {"/getDataList/{systemId}","/getDataList"})
    public ActionResult getAllList(@PathVariable(value = "systemId",required = false) String systemId) {
        if(StringUtils.isEmpty(systemId)) systemId ="-1";
        List<AppDataListAllVO> result = appDataService.getDataList("2", systemId);
        ListVO listVO = new ListVO();
        listVO.setList(result);
        return ActionResult.success(listVO);
    }

}
