package com.bringspring.common.util;


import cn.hutool.core.util.ObjectUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.cache.CacheProperties;
import org.springframework.boot.autoconfigure.cache.CacheType;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.concurrent.TimeUnit;

/**
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2021/3/16 8:54
 */
@Slf4j
@Component
@Lazy(false)
public class CacheUtil {
    private CacheType cacheType;
    @Autowired
    private CacheProperties cacheProperties;
    @Autowired
    private RedisUtil redisUtil;
    @Autowired
    private EhcacheUtil ehcacheUtil;
    @Autowired
    private CacheKeyUtil cacheKeyUtil;

    public CacheType getCacheType() {
        if (ObjectUtil.isEmpty(cacheType)) {
            cacheType = cacheProperties.getType();
        }
        return cacheType;
    }

    public void setType(CacheType cacheType) {
        this.cacheType = cacheType;
    }

    @Bean
    @ConditionalOnClass(CacheProperties.class)
    public void initCacheType() {
        cacheType = cacheProperties.getType();
        log.info("CacheUtil spring.cache.type：" + cacheType);
    }

    /**
     * 插入缓存(无时间)
     *
     * @param cacheName
     * @param key
     * @param value
     */
    public void insert(String cacheName, String key, Object value) {
        switch (cacheType) {
            case EHCACHE:
                ehcacheUtil.insert(cacheName, key, value);
                break;
            case REDIS:
                redisUtil.insert(key, value);
                break;
            default:
                break;
        }
    }

    /**
     * 插入缓存(有时间)
     *
     * @param cacheName
     * @param key
     * @param value
     * @param time
     */
    public void insert(String cacheName, String key, Object value, long time) {
        switch (cacheType) {
            case EHCACHE:
                ehcacheUtil.insert(cacheName, key, value, (int) time);
                break;
            case REDIS:
                redisUtil.insert(key, value, time);
                break;
            default:
                break;
        }
    }

    /**
     * 指定缓存失效时间
     *
     * @param key
     * @param time
     * @return
     */
    public void expire(String cacheName, String key, int time) {
        switch (cacheType) {
            case EHCACHE:
                ehcacheUtil.expire(cacheName, key, time);
                break;
            case REDIS:
                redisUtil.expire(key, time);
                break;
            default:
                break;
        }
    }

    /**
     * 删除指定key
     *
     * @param key
     */
    public void remove(String key) {
        switch (cacheType) {
            case EHCACHE:
                ehcacheUtil.remove(key);
                break;
            case REDIS:
                redisUtil.remove(key);
                break;
            default:
                break;
        }
    }

    /**
     * 删除指定key
     *
     * @param key
     */
    public void remove(String cacheName, String key) {
        if (StringUtils.isNotEmpty(key)) {
            switch (cacheType) {
                case EHCACHE:
                    ehcacheUtil.remove(cacheName, key);
                    break;
                case REDIS:
                    redisUtil.remove(key);
                    break;
                default:
                    break;
            }
        }
    }

    /**
     * 删除指定map的key
     *
     * @param key
     */
    public void removeHash(String cacheName, String hashId, String key) {
        switch (cacheType) {
            case EHCACHE:
                ehcacheUtil.removeHash(cacheName, key);
                break;
            case REDIS:
                redisUtil.removeHash(hashId, key);
                break;
            default:
                break;
        }
    }

    /**
     * 查
     */
    public Object query(String key) {
        if (StringUtils.isNotEmpty(key)) {
            switch (cacheType) {
                case EHCACHE:
                    return ehcacheUtil.query(key);
                case REDIS:
                    return redisUtil.getString(key);
                default:
                    return null;
            }
        } else {
            return null;
        }
    }


    /**
     * 查
     */
    public Object query(String cacheName, String key) {
        if (StringUtils.isNotEmpty(cacheName) && StringUtils.isNotEmpty(key)) {
            switch (cacheType) {
                case EHCACHE:
                    return ehcacheUtil.query(cacheName, key);
                case REDIS:
                    return redisUtil.getString(key);
                default:
                    return null;
            }
        } else {
            return null;
        }
    }


    /**
     * 获取所有的map缓存
     *
     * @param key
     * @return
     */
    public <K, V> Map<K, V> getMap(String key) {
        if (StringUtils.isNotEmpty(key)) {
            switch (cacheType) {
                case EHCACHE:
                    return ehcacheUtil.getMap(key);
                case REDIS:
                    return redisUtil.getMap(key);
                default:
                    return null;
            }
        } else {
            return null;
        }
    }

    /**
     * 获取list缓存的内容
     *
     * @param key   键
     * @param start 开始 0 是第一个元素
     * @param end   结束 -1代表所有值
     * @return
     * @取出来的元素 总数 end-start+1
     */
    public List<Object> getList(String key, long start, long end) {
        try {
            if (StringUtils.isNotEmpty(key)) {
                switch (cacheType) {
                    case EHCACHE:
                        return ehcacheUtil.getList(key);
                    case REDIS:
                        return redisUtil.get(key, start, end);
                    default:
                        return null;
                }
            } else {
                return null;
            }
        } catch (Exception e) {
            log.error(key, e);
            return null;
        }
    }

    /**
     * 根据key获取Set中的所有值
     *
     * @param key 键
     * @return
     */
    public Set<Object> getSet(String key) {
        try {
            if (StringUtils.isNotEmpty(key)) {
                switch (cacheType) {
                    case EHCACHE:
                        return ehcacheUtil.getSet(key);
                    case REDIS:
                        return redisUtil.getSet(key);
                    default:
                        return null;
                }
            } else {
                return null;
            }
        } catch (Exception e) {
            log.error(key, e);
            return null;
        }
    }

    /**
     * 判断缓存中是否有对应的value
     *
     * @param key
     * @return
     */
    public boolean exists(final String key) {
        return this.exists(null, key);
    }

    /**
     * 判断缓存中是否有对应的value
     *
     * @param key
     * @return
     */
    public boolean exists(String cacheName, final String key) {
        if (StringUtils.isNotEmpty(key)) {
            switch (cacheType) {
                case EHCACHE:
                    return ehcacheUtil.exists(cacheName, key);
                case REDIS:
                    return redisUtil.exists(key);
                default:
                    return false;
            }
        } else {
            return false;
        }
    }

    /**
     * 获取所有的key
     *
     * @return
     */
    public Set<String> getAllKeys() {
        return this.getAllKeys(null);
    }

    /**
     * 获取指定cacheName所有的key
     *
     * @param cacheName
     * @return
     */
    public Set<String> getAllKeys(String cacheName) {
        Set<String> keys = null;
        switch (cacheType) {
            case EHCACHE:
                keys = ehcacheUtil.getAllKeys(cacheName);
                break;
            case REDIS:
                keys = redisUtil.getAllKeys();
                break;
            default:
                break;
        }
        return keys;
    }


    /**
     * 获取所有的key
     *
     * @return
     */
    public Set<String> getAllVisiualKeys() {
        Set<String> allKey = new HashSet<>(16);
        switch (cacheType) {
            case EHCACHE:
                allKey = ehcacheUtil.getAllVisiualKeys();
                break;
            case REDIS:
                allKey = redisUtil.getAllVisiualKeys();
                break;
            default:
                break;
        }
        return allKey;
    }


    /**
     * 返回 key 的剩余的过期时间
     *
     * @param key
     * @return
     */
    public Long getLiveTime(String key) {
        switch (cacheType) {
            case EHCACHE:
                return ehcacheUtil.getLiveTime(key);
            case REDIS:
                return new Date((DateUtil.getTime(new Date()) + redisUtil.getLiveTime(key)) * 1000).getTime();
            default:
                break;
        }
        return 0L;
    }

    /**
     * string类型-设置key-value 原子性操作
     *
     * @param key   key
     * @param value value
     * @throws Exception
     */
    public Boolean setStringByAtomicity(String key, String value, Long timeout, TimeUnit timeUnit) throws Exception {
        switch (cacheType) {
            case EHCACHE:
                return ehcacheUtil.setStringByAtomicity(key, value);
            case REDIS:
                return redisUtil.setStringByAtomicity(key, value, timeout, TimeUnit.MILLISECONDS);
            default:
                break;
        }
        return false;
    }

}
