package com.bringspring.common.properties;

import com.bringspring.common.annotation.NotCheckLogin;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.stereotype.Component;
import org.springframework.stereotype.Controller;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.*;
import jakarta.annotation.PostConstruct;
import java.lang.reflect.Method;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;

/**
 * NotCheckLogin注解路径扫描器
 */
@Slf4j
@Component
public class NotCheckLoginPathScanner implements ApplicationContextAware {

    private ApplicationContext applicationContext;

    /**
     * 存储所有不需要登录检查的路径
     */
    private final Map<String, NotCheckLoginPathInfo> notCheckLoginPaths = new ConcurrentHashMap<>();

    /**
     * 按控制器分类的路径
     */
    private final Map<String, List<NotCheckLoginPathInfo>> controllerPaths = new ConcurrentHashMap<>();

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    /**
     * 应用启动后自动扫描
     */
    @PostConstruct
    public void init() {
        log.info("开始扫描带有@NotCheckLogin注解的Controller方法...");
        scanNotCheckLoginPaths();
        log.info("扫描完成，共找到 {} 个不需要登录检查的接口", notCheckLoginPaths.size());

        // 打印扫描结果
//        printScanResult();
    }

    /**
     * 扫描所有带有@NotCheckLogin注解的方法
     */
    private void scanNotCheckLoginPaths() {
        // 获取所有Controller Bean
        Map<String, Object> controllerBeans = applicationContext.getBeansWithAnnotation(Controller.class);
        controllerBeans.putAll(applicationContext.getBeansWithAnnotation(RestController.class));

        for (Object controller : controllerBeans.values()) {
            Class<?> controllerClass = controller.getClass();

            // 处理代理类的情况（Spring AOP/CGLIB）
            Class<?> targetClass = getTargetClass(controllerClass);

            // 获取类级别的RequestMapping注解
            RequestMapping classRequestMapping = AnnotationUtils.findAnnotation(targetClass, RequestMapping.class);
            String classPath = "";
            if (classRequestMapping != null && classRequestMapping.value().length > 0) {
                classPath = normalizePath(classRequestMapping.value()[0]);
            }

            // 扫描类中的方法
            scanMethodsInClass(targetClass, classPath);
        }
    }

    /**
     * 扫描单个类中的所有方法
     */
    private void scanMethodsInClass(Class<?> controllerClass, String classPath) {
        Method[] methods = controllerClass.getDeclaredMethods();

        for (Method method : methods) {
            // 检查方法是否有@NotCheckLogin注解
            NotCheckLogin notCheckLogin = AnnotationUtils.findAnnotation(method, NotCheckLogin.class);
            if (notCheckLogin == null) {
                continue;
            }

            // 获取方法上的RequestMapping注解
            RequestMapping requestMapping = AnnotationUtils.findAnnotation(method, RequestMapping.class);
            GetMapping getMapping = AnnotationUtils.findAnnotation(method, GetMapping.class);
            PostMapping postMapping = AnnotationUtils.findAnnotation(method, PostMapping.class);
            PutMapping putMapping = AnnotationUtils.findAnnotation(method, PutMapping.class);
            DeleteMapping deleteMapping = AnnotationUtils.findAnnotation(method, DeleteMapping.class);
            PatchMapping patchMapping = AnnotationUtils.findAnnotation(method, PatchMapping.class);

            // 处理不同类型的映射注解
            processMethodMapping(method, controllerClass, classPath,
                    "GET", getMapping != null ? getMapping.value() :
                            requestMapping != null ? requestMapping.value() : new String[0]);

            processMethodMapping(method, controllerClass, classPath,
                    "POST", postMapping != null ? postMapping.value() :
                            requestMapping != null && requestMapping.method().length > 0 ?
                                    (Arrays.asList(requestMapping.method()).contains(RequestMethod.POST) ? requestMapping.value() : new String[0]) : new String[0]);

            // 继续处理其他HTTP方法...
            if (putMapping != null) {
                processMethodMapping(method, controllerClass, classPath, "PUT", putMapping.value());
            }
            if (deleteMapping != null) {
                processMethodMapping(method, controllerClass, classPath, "DELETE", deleteMapping.value());
            }
            if (patchMapping != null) {
                processMethodMapping(method, controllerClass, classPath, "PATCH", patchMapping.value());
            }
        }
    }

    /**
     * 处理方法映射
     */
    private void processMethodMapping(Method method, Class<?> controllerClass,
                                      String classPath, String httpMethod, String[] methodPaths) {
        if (methodPaths.length == 0) {
            // 如果没有指定具体路径，默认为空路径
            String fullPath = buildFullPath(classPath, "");
            addPathInfo(fullPath, controllerClass, method, httpMethod);
            return;
        }

        for (String methodPath : methodPaths) {
            String fullPath = buildFullPath(classPath, methodPath);
            addPathInfo(fullPath, controllerClass, method, httpMethod);
        }
    }

    /**
     * 构建完整路径
     */
    private String buildFullPath(String classPath, String methodPath) {
        // 处理路径拼接
        StringBuilder fullPath = new StringBuilder();

        if (StringUtils.hasText(classPath)) {
            fullPath.append(normalizePath(classPath));
        }

        if (StringUtils.hasText(methodPath)) {
            String normalizedMethodPath = normalizePath(methodPath);
            if (!normalizedMethodPath.startsWith("/") && fullPath.length() > 0) {
                fullPath.append("/");
            }
            fullPath.append(normalizedMethodPath);
        }

        // 确保路径以/开头
        String result = fullPath.toString();
        if (!result.startsWith("/")) {
            result = "/" + result;
        }

        return result;
    }

    /**
     * 规范化路径（去除首尾空格，确保格式正确）
     */
    private String normalizePath(String path) {
        if (!StringUtils.hasText(path)) {
            return "";
        }

        String normalized = path.trim();

        // 移除开头的斜杠（我们会在拼接时统一添加）
        while (normalized.startsWith("/")) {
            normalized = normalized.substring(1);
        }

        // 确保路径中的变量格式正确
        normalized = normalized.replaceAll("/+", "/");

        return normalized;
    }

    /**
     * 添加路径信息到集合
     */
    private void addPathInfo(String fullPath, Class<?> controllerClass,
                             Method method, String httpMethod) {
        String key = httpMethod + ":" + fullPath;

        NotCheckLoginPathInfo pathInfo = new NotCheckLoginPathInfo(
                fullPath,
                controllerClass.getName(),
                method.getName(),
                httpMethod,
                method
        );

        notCheckLoginPaths.put(key, pathInfo);

        // 按控制器分类存储
        String controllerName = controllerClass.getSimpleName();
        controllerPaths.computeIfAbsent(controllerName, k -> new ArrayList<>()).add(pathInfo);
    }

    /**
     * 处理代理类获取原始类
     */
    private Class<?> getTargetClass(Class<?> clazz) {
        // 处理Spring AOP/CGLIB代理类
        Class<?> targetClass = clazz;
        while (targetClass.getName().contains("$$")) {
            targetClass = targetClass.getSuperclass();
        }
        return targetClass;
    }

    /**
     * 打印扫描结果
     */
    private void printScanResult() {
        if (notCheckLoginPaths.isEmpty()) {
            log.warn("未找到任何带有@NotCheckLogin注解的接口");
            return;
        }

        log.info("========== 不需要登录检查的接口列表 ==========");
        controllerPaths.forEach((controllerName, paths) -> {
            log.info("\n控制器: {}", controllerName);
            paths.forEach(pathInfo -> {
                log.info("  {}", pathInfo);
            });
        });
        log.info("============================================");
    }

    /**
     * 获取所有不需要登录检查的路径
     */
    public List<NotCheckLoginPathInfo> getAllNotCheckLoginPaths() {
        return new ArrayList<>(notCheckLoginPaths.values());
    }

    /**
     * 按控制器获取路径
     */
    public Map<String, List<NotCheckLoginPathInfo>> getPathsByController() {
        return new HashMap<>(controllerPaths);
    }

    /**
     * 检查指定路径是否需要登录
     */
    public boolean isNotCheckLoginPath(String httpMethod, String requestPath) {
        // 精确匹配
        String key = httpMethod + ":" + requestPath;
        if (notCheckLoginPaths.containsKey(key)) {
            return true;
        }

        // 模式匹配（处理路径变量）
        for (NotCheckLoginPathInfo pathInfo : notCheckLoginPaths.values()) {
            if (pathInfo.getHttpMethod().equalsIgnoreCase(httpMethod) &&
                    pathMatches(patternToRegex(pathInfo.getFullPath()), requestPath)) {
                return true;
            }
        }

        return false;
    }

    /**
     * 路径模式匹配
     */
    private boolean pathMatches(String pattern, String path) {
        if (pattern.equals(path)) {
            return true;
        }

        // 简单的路径变量匹配：将{xxx}转换为正则表达式.*
        String regex = pattern.replaceAll("\\{[^/]+\\}", "[^/]+");
        return path.matches(regex);
    }

    /**
     * 将路径模式转换为正则表达式
     */
    private String patternToRegex(String pattern) {
        return pattern.replaceAll("\\{[^/]+\\}", "([^/]+)");
    }

    /**
     * 获取路径模式列表（可用于Sa-Token配置）
     */
    public List<String> getPathPatterns() {
        return notCheckLoginPaths.values().stream()
                .map(NotCheckLoginPathInfo::getFullPath)
                .distinct()
                .sorted()
                .toList();
    }

    /**
     * 获取HTTP方法和路径的映射（用于直接配置）
     */
    public Map<String, List<String>> getHttpMethodPathMap() {
        Map<String, List<String>> result = new HashMap<>();

        notCheckLoginPaths.values().forEach(pathInfo -> {
            result.computeIfAbsent(pathInfo.getHttpMethod(), k -> new ArrayList<>())
                    .add(pathInfo.getFullPath());
        });

        return result;
    }
}