package com.bringspring.common.auth.granter;

import cn.dev33.satoken.context.SaHolder;
import cn.dev33.satoken.stp.StpUtil;
import cn.dev33.satoken.stp.parameter.SaLoginParameter;
import com.bringspring.common.auth.consts.AuthConsts;
import com.bringspring.common.auth.consts.DeviceType;
import com.bringspring.common.auth.consts.LoginTicketStatus;
import com.bringspring.common.auth.model.LoginTicketModel;
import com.bringspring.common.auth.service.LoginSatokenService;
import com.bringspring.common.auth.util.TenantProvider;
import com.bringspring.common.auth.util.TicketUtil;
import com.bringspring.common.auth.util.UserProvider;
import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.UserInfo;
import com.bringspring.common.config.ConfigValueUtil;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.database.model.TenantVO;
import com.bringspring.common.exception.LoginException;
import com.bringspring.common.exception.TenantDatabaseException;
import com.bringspring.common.model.login.BaseSystemInfo;
import com.bringspring.common.model.logout.LogoutResultModel;
import com.bringspring.common.util.CacheUtil;
import com.bringspring.common.util.RedisUtil;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.Ordered;
import org.springframework.util.AntPathMatcher;
import org.springframework.util.PathMatcher;

import java.util.HashMap;
import java.util.Map;

import static com.bringspring.common.auth.consts.AuthConsts.DEF_TENANT_DB;
import static com.bringspring.common.auth.consts.AuthConsts.DEF_TENANT_ID;


/**
 * @author RKKJ开发平台组
 * @copyright 荣科科技股份有限公司
 */
public abstract class AbstractTokenGranter implements TokenGranter, Ordered {


    @Autowired(required = false)
    protected LoginSatokenService loginSatokenService;
    @Autowired
    protected ConfigValueUtil configValueUtil;
    @Autowired
    protected RedisUtil redisUtil;
    @Autowired
    protected CacheUtil cacheUtil;

    protected static PathMatcher pathMatcher = new AntPathMatcher();

    private String authenticationUrl;


    public AbstractTokenGranter(String authenticationUrl) {
        this.authenticationUrl = authenticationUrl;
    }


    /**
     * 最终登录用户
     *
     * @param userInfo 包含账户名, 登录方式
     * @return
     */
    protected String loginAccount(UserInfo userInfo, BaseSystemInfo baseSystemInfo) throws LoginException {
        try {
            //获取用户实现类接口名称
            userInfo.setUserDetailKey(getUserDetailKey());
            //获取登录信息
            userInfo = getUserInfo(userInfo, baseSystemInfo);
            //预登陆
            preLogin(userInfo, baseSystemInfo);
            //登录
            login(userInfo, baseSystemInfo);
        } catch (Exception e) {
            try {
                loginFailure(userInfo, baseSystemInfo, e);
            } catch (Exception e1) {
                throw e1;
            }
            throw e;
        }
        loginSuccess(userInfo, baseSystemInfo);
        //返回token信息
        return userInfo.getToken();
    }

    /**
     * 切换多租户
     *
     * @param userInfo
     * @return userAccount, tenantId, tenandDb
     * @throws LoginException
     */
    protected UserInfo switchTenant(UserInfo userInfo) throws LoginException {
        if (configValueUtil.isMultiTenancy()) {
            userInfo = loginSatokenService.getTenantAccount(userInfo);
            return userInfo;

        }
        userInfo.setTenantId(DEF_TENANT_ID);
        userInfo.setTenantDbConnectionString(DEF_TENANT_DB);
        userInfo.setTenantDbType(TenantVO.NONE);
        return userInfo;
    }

    /**
     * 获取系统配置
     *
     * @param userInfo
     * @return
     */
    protected BaseSystemInfo getSysconfig(UserInfo userInfo) throws LoginException {
        BaseSystemInfo baseSystemInfo = loginSatokenService.getBaseSystemConfig(userInfo.getTenantId());
        if (baseSystemInfo != null && baseSystemInfo.getSingleLogin() != null) {
            TenantProvider.setBaseSystemInfo(baseSystemInfo);
        } else {
            throw new TenantDatabaseException().setLogMsg(MsgCode.LOG110.get());
        }
        return baseSystemInfo;
    }

    /**
     * 根据关键词获取系统配置
     *
     * @param keyName
     * @return
     */
    protected String getConfigByKeyName(String keyName) throws LoginException {
        return loginSatokenService.getConfigByKeyName(keyName);
    }

    /**
     * 获取登录设备
     *
     * @return
     */
    protected DeviceType getDeviceType() {
        return UserProvider.getDeviceForAgent();
    }


    /**
     * 生成登录用户信息
     *
     * @param userInfo
     * @return
     */
    protected UserInfo getUserInfo(UserInfo userInfo, BaseSystemInfo sysConfigInfo) throws LoginException {
        userInfo.setGrantType(getGrantType());
        userInfo = loginSatokenService.userInfo(userInfo, sysConfigInfo);
        return userInfo;
    }


    /**
     * 登录前执行
     *
     * @param userInfo
     * @param baseSystemInfo
     */
    protected void preLogin(UserInfo userInfo, BaseSystemInfo baseSystemInfo) throws LoginException {

    }

    /**
     * 登录操作
     *
     * @param userInfo
     * @param baseSystemInfo
     */
    protected void login(UserInfo userInfo, BaseSystemInfo baseSystemInfo) throws LoginException {
        UserProvider.login(userInfo, getLoginModel(userInfo, baseSystemInfo));
    }

    /**
     * 登录成功触发
     *
     * @param userInfo
     * @param baseSystemInfo
     */
    protected void loginSuccess(UserInfo userInfo, BaseSystemInfo baseSystemInfo) {

    }

    /**
     * 登录失败触发
     *
     * @param baseSystemInfo
     */
    protected void loginFailure(UserInfo userInfo, BaseSystemInfo baseSystemInfo, Exception e) {

    }

    protected abstract String getUserDetailKey();

    protected String createToken(UserInfo userInfo, BaseSystemInfo baseSystemInfo) {
        //登录
        UserProvider.login(userInfo, getLoginModel(userInfo, baseSystemInfo));
        return StpUtil.getTokenValueNotCut();
    }

    /**
     * 更新轮询结果为成功
     */
    protected void updateTicketSuccess(UserInfo userInfo) {
        String ticket = getJsbosTicket();
        if (!ticket.isEmpty()) {
            LoginTicketModel loginTicketModel = new LoginTicketModel()
                    .setStatus(LoginTicketStatus.Success.getStatus())
                    .setValue(StpUtil.getTokenValueNotCut())
                    .setTheme(userInfo.getTheme());
            TicketUtil.updateTicket(ticket, loginTicketModel, null);
        }
    }

    /**
     * 更新轮询结果为失败
     */
    protected void updateTicketError(String msg) {
        String ticket = getJsbosTicket();
        if (!ticket.isEmpty()) {
            LoginTicketModel loginTicketModel = new LoginTicketModel()
                    .setStatus(LoginTicketStatus.ErrLogin.getStatus())
                    .setValue(msg);
            TicketUtil.updateTicket(ticket, loginTicketModel, null);
        }
    }


    /**
     * 获取轮询ticket
     *
     * @return
     */
    protected String getJsbosTicket() {
        return SaHolder.getRequest().getParam(AuthConsts.PARAMS_JSBOS_TICKET, "");
    }

    protected boolean isValidJsbosTicket() {
        String jsbosTicket = getJsbosTicket();
        if (!jsbosTicket.isEmpty()) {
            LoginTicketModel loginTicketModel = TicketUtil.parseTicket(jsbosTicket);
            if (loginTicketModel == null) {
                return false;
            }
        }
        return true;
    }

    /**
     * 获取登录参数
     *
     * @param userInfo
     * @param baseSystemInfo
     * @return
     */
    protected SaLoginParameter getLoginModel(UserInfo userInfo, BaseSystemInfo baseSystemInfo) {
        SaLoginParameter loginModel = new SaLoginParameter();
        loginModel.setTimeout(userInfo.getTokenTimeout() * 60L);
        loginModel.setExtraData(getTokenExtraData(userInfo, baseSystemInfo));
        if (userInfo.getLoginDevice() == null) {
            loginModel.setDevice(getDeviceType().getDevice());
            userInfo.setLoginDevice(loginModel.getDeviceType());
        } else {
            loginModel.setDevice(userInfo.getLoginDevice());
        }
        return loginModel;
    }

    /**
     * 获取额外的JWT内容
     *
     * @param userInfo
     * @param baseSystemInfo
     * @return
     */
    protected Map<String, Object> getTokenExtraData(UserInfo userInfo, BaseSystemInfo baseSystemInfo) {
        Map<String, Object> tokenInfo = new HashMap<>();
//        tokenInfo.put("token", StpUtil.getTokenValue());
        tokenInfo.put("singleLogin", baseSystemInfo == null ? null : baseSystemInfo.getSingleLogin());
        tokenInfo.put("user_name", userInfo.getUserAccount());
        tokenInfo.put("user_id", userInfo.getUserId());
        tokenInfo.put("exp", userInfo.getOverdueTime().getTime());
        tokenInfo.put("token", userInfo.getId());
        return tokenInfo;
    }

    @Override
    public ActionResult<LogoutResultModel> logout() {
        UserProvider.logout();
        return ActionResult.success();
    }

    protected abstract String getGrantType();


    @Override
    public boolean requiresAuthentication() {
        String path = SaHolder.getRequest().getRequestPath();
        if (path != null && path.startsWith("/api/oauth")) {
            path = path.replace("/api/oauth", "");
        }
        return pathMatcher.match(authenticationUrl, path);
    }
}
