package com.bringspring.common.util;

import cn.afterturn.easypoi.excel.annotation.Excel;
import cn.hutool.core.util.ObjectUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.Assert;

import java.lang.reflect.*;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * 反射工具类
 *
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2021/3/16 8:56
 */
@Slf4j
public class ReflectionUtil {

    /**
     * 是否Debug模式
     */
    private static boolean isDebug = false;

    /**
     * 调用Getter方法.
     */
    public static Object invokeGetterMethod(Object obj, String propertyName) {
        String getterMethodName = "get" + StringUtils.capitalize(propertyName);
        return invokeMethod(obj, getterMethodName, new Class[]{}, new Object[]{});
    }

    /**
     * 调用Setter方法.使用value的Class来查找Setter方法.
     */
    public static void invokeSetterMethod(Object obj, String propertyName, Object value) {
        invokeSetterMethod(obj, propertyName, value, null);
    }

    /**
     * 调用Setter方法.
     *
     * @param propertyType 用于查找Setter方法,为空时使用value的Class替代.
     */
    public static void invokeSetterMethod(Object obj, String propertyName, Object value, Class<?> propertyType) {
        Class<?> type = propertyType != null ? propertyType : value.getClass();
        String setterMethodName = "set" + StringUtils.capitalize(propertyName);
        invokeMethod(obj, setterMethodName, new Class[]{type}, new Object[]{value});
    }

    /**
     * 直接读取对象属性值, 无视private/protected修饰符, 不经过getter函数.
     */
    public static Object getFieldValue(final Object obj, final String fieldName) {
        Field field = getAccessibleField(obj, fieldName);
        if (field == null) {
            throw new IllegalArgumentException("Could not find field [" + fieldName + "] on target [" + obj + "]");
        }
        Object result = null;
        try {
            result = field.get(obj);
        } catch (IllegalAccessException e) {
            log.error("不可能抛出的异常{}", e);
        }
        return result;
    }

    /**
     * 直接设置对象属性值, 无视private/protected修饰符, 不经过setter函数.
     */
    public static void setFieldValue(final Object obj, final String fieldName, final Object value) {
        Field field = getAccessibleField(obj, fieldName);
        if (field == null) {
            throw new IllegalArgumentException("Could not find field [" + fieldName + "] on target [" + obj + "]");
        }
        try {
            field.set(obj, value);
        } catch (IllegalAccessException e) {
            log.error("不可能抛出的异常:{}", e);
        }
    }

    /**
     * 循环向上转型, 获取对象的DeclaredField, 并强制设置为可访问.
     * <p>
     * 如向上转型到Object仍无法找到, 返回null.
     */
    public static Field getAccessibleField(final Object obj, final String fieldName) {
        Assert.notNull(obj, "object不能为空");
        Assert.hasText(fieldName, "fieldName");
        for (Class<?> superClass = obj.getClass(); superClass != Object.class; superClass = superClass.getSuperclass()) {
            try {
                Field field = superClass.getDeclaredField(fieldName);
                field.setAccessible(true);
                return field;
            } catch (NoSuchFieldException e) {// NOSONAR
                // Field不在当前类定义,继续向上转型
            }
        }
        return null;
    }

    /**
     * 直接调用对象方法, 无视private/protected修饰符. 用于一次性调用的情况.
     */
    public static Object invokeMethod(final Object obj, final String methodName, final Class<?>[] parameterTypes, final Object[] args) {
        Method method = getAccessibleMethod(obj, methodName, parameterTypes);
        if (method == null) {
            throw new IllegalArgumentException("Could not find method [" + methodName + "] on target [" + obj + "]");
        }
        try {
            return method.invoke(obj, args);
        } catch (Exception e) {
            throw convertReflectionExceptionToUnchecked(e);
        }
    }


    /**
     * 系统调度使用！！！
     * 直接调用对象方法, 无视private/protected修饰符. 用于一次性调用的情况.
     */
    public static Boolean invokeMethodByTask(final Object obj, final String methodName, final Class<?>[] parameterTypes, final Map<String, Object> args) {
        Method method = getAccessibleMethod(obj, methodName, parameterTypes);
        if (method == null) {
            return false;
        }
        try {
            if (ObjectUtil.isEmpty(args)) {
                method.invoke(obj, new Object[0]);
            } else {
                method.invoke(obj, args);
            }
            return true;
        } catch (Exception e) {
            throw convertReflectionExceptionToUnchecked(e);
        }
    }

    /**
     * 循环向上转型, 获取对象的DeclaredMethod,并强制设置为可访问. 如向上转型到Object仍无法找到, 返回null.
     * <p>
     * 用于方法需要被多次调用的情况. 先使用本函数先取得Method,然后调用Method.invoke(Object obj, Object... args)
     */
    public static Method getAccessibleMethod(final Object obj, final String methodName, final Class<?>... parameterTypes) {
        Assert.notNull(obj, "object不能为空");
        for (Class<?> superClass = obj.getClass(); superClass != Object.class; superClass = superClass.getSuperclass()) {
            try {
                Method method = superClass.getDeclaredMethod(methodName, parameterTypes);
                method.setAccessible(true);
                return method;
            } catch (NoSuchMethodException e) {// NOSONAR
                // Method不在当前类定义,继续向上转型
            }
        }
        return null;
    }

    /**
     * 通过反射, 获得Class定义中声明的父类的泛型参数的类型. 如无法找到, 返回Object.class. eg. public UserDao extends HibernateDao<User>
     *
     * @param clazz The class to introspect
     * @return the first generic declaration, or Object.class if cannot be determined
     */
    @SuppressWarnings("unchecked")
    public static <T> Class<T> getSuperClassGenricType(final Class<?> clazz) {
        return (Class<T>) getSuperClassGenricType(clazz, 0);
    }

    /**
     * 通过反射, 获得Class定义中声明的父类的泛型参数的类型. 如无法找到, 返回Object.class.
     * <p>
     * 如public UserDao extends HibernateDao<User,Long>
     *
     * @param clazz clazz The class to introspect
     * @param index the Index of the generic ddeclaration,start from 0.
     * @return the index generic declaration, or Object.class if cannot be determined
     */
    public static Class<?> getSuperClassGenricType(final Class<?> clazz, final int index) {
        Type genType = clazz.getGenericSuperclass();
        if (!(genType instanceof ParameterizedType)) {
            log.warn(clazz.getSimpleName() + "'s superclass not ParameterizedType");
            return Object.class;
        }
        Type[] params = ((ParameterizedType) genType).getActualTypeArguments();
        if (index >= params.length || index < 0) {
            log.warn("Index: " + index + ", Size of " + clazz.getSimpleName() + "'s Parameterized Type: " + params.length);
            return Object.class;
        }
        if (!(params[index] instanceof Class)) {
            log.warn(clazz.getSimpleName() + " not set the actual class on superclass generic parameter");
            return Object.class;
        }
        return (Class<?>) params[index];
    }

    /**
     * 将反射时的checked exception转换为unchecked exceptions.
     */
    public static RuntimeException convertReflectionExceptionToUnchecked(Exception e) {
        if (e instanceof IllegalAccessException || e instanceof IllegalArgumentException || e instanceof NoSuchMethodException) {
            return new IllegalArgumentException("Reflection com.bringspring.exception.", e);
        } else if (e instanceof InvocationTargetException) {
            return new RuntimeException("Reflection com.bringspring.exception.", ((InvocationTargetException) e).getTargetException());
        } else if (e instanceof RuntimeException) {
            return (RuntimeException) e;
        }
        return new RuntimeException("Unexpected Checked com.bringspring.exception.", e);
    }

    /**
     * 调用传入对象的toString方法或反射返回对象成员变量值字符串。
     *
     * @param obj 传入对象
     * @return
     * @author Lin Chenglin 2013-4-9
     */
    public static String toString(final Object obj) {
        if (obj == null) {
            return null;
        }
        if (obj.getClass() == Object.class || obj.getClass().isPrimitive()) {
            return obj.toString();
        }
        try {
            Method method = obj.getClass().getDeclaredMethod("toString", new Class[]{});
            if (isDebug) {
                log.debug("传入的对象实现了自己的toString方法，直接调用！");
            }
            return (String) method.invoke(obj, new Object[]{});
        } catch (NoSuchMethodException e) {
            if (isDebug) {
                log.debug("传入的对象没有实现自己的toString方法，反射获取！");
            }
            StringBuffer buf = new StringBuffer(obj.getClass().getName());
            buf.append(" [");
            // 获取所有成员变量
            Field[] fileds = obj.getClass().getDeclaredFields();
            int size = fileds.length;
            for (int i = 0; i < size; i++) {
                Field field = fileds[i];
                Object value = ReflectionUtil.getFieldValue(obj, field.getName());
                buf.append(field.getName() + "=" + ReflectionUtil.toString(value));
                if (i != size - 1) {
                    buf.append(", ");
                }
            }
            buf.append("]");
            return buf.toString();
        } catch (Exception e) {
            throw convertReflectionExceptionToUnchecked(e);
        }
    }

    /**
     * 获取累中的所有属性名称及注解内容
     * 格式:注解->字段
     *
     * @param instance
     * @param fieldMap
     * @return
     */
    public static Map<String, String> getDeclaredFieldsInfo2(Object instance, Map<String, String> fieldMap) throws NoSuchFieldException {


        Map<String, String> map = new LinkedHashMap<>();
        Class<?> clazz = instance.getClass();
        Field[] fields = clazz.getDeclaredFields();
        boolean b = false;
        for (int i = 0; i < fields.length; i++) {


            // 除过fieldMap中的属性，其他属性都获取
            if (!fieldMap.containsValue(fields[i].getName())) {


//                Field field=clazz.getDeclaredField(fields[i].getName());
                boolean annotationPresent = fields[i].isAnnotationPresent(Excel.class);
                if (annotationPresent) {


                    // 获取注解值
                    String name = fields[i].getAnnotation(Excel.class).name();
                    map.put(name, fields[i].getName());
                }
            }
        }
        return map;

    }
}
