package com.bringspring.common.util;

import org.springframework.web.multipart.MultipartFile;
import java.io.*;
import java.nio.file.Files;

/**
 *  自定义MultipartFile实现
 * 用于MultipartFile文件上传
 */
public class CustomMultipartFile implements MultipartFile {

    private final String name;
    private final String originalFilename;
    private final String contentType;
    private final byte[] content;

    public CustomMultipartFile(File file) throws IOException {
        this.originalFilename = file.getName();
        this.name = "file";
        this.contentType = Files.probeContentType(file.toPath()); // 可选，也可硬编码
        this.content = Files.readAllBytes(file.toPath());
    }

    // 构造函数重载：允许指定字段名、文件名、内容类型和字节
    public CustomMultipartFile(String name, String originalFilename, String contentType, byte[] content) {
        this.name = name;
        this.originalFilename = originalFilename;
        this.contentType = contentType;
        this.content = content;
    }

    @Override
    public String getName() {
        return name;
    }

    @Override
    public String getOriginalFilename() {
        return originalFilename;
    }

    @Override
    public String getContentType() {
        return contentType;
    }

    @Override
    public boolean isEmpty() {
        return content == null || content.length == 0;
    }

    @Override
    public long getSize() {
        return content != null ? content.length : 0;
    }

    @Override
    public byte[] getBytes() throws IOException {
        return content.clone(); // 避免外部修改
    }

    @Override
    public InputStream getInputStream() throws IOException {
        return new ByteArrayInputStream(content);
    }

    @Override
    public void transferTo(File dest) throws IOException, IllegalStateException {
        Files.write(dest.toPath(), content);
    }
}
