package com.bringspring.common.util;


import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import lombok.extern.slf4j.Slf4j;
import net.sf.ehcache.Ehcache;
import net.sf.ehcache.Element;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.Cache;
import org.springframework.cache.CacheManager;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.stream.Collectors;

import static com.bringspring.common.constant.CacheConsts.REPEATSUBMIT_CACHE;

/**
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2021/3/16 8:54
 */
@Slf4j
@Component
public class EhcacheUtil {

    //注入缓存管理器
    @Autowired
    CacheManager cacheManager;
    @Autowired
    private CacheKeyUtil cacheKeyUtil;

    /**
     * 插入缓存(无时间)
     *
     * @param cacheName
     * @param key
     * @param value
     */
    public void insert(String cacheName, String key, Object value) {
        if (StringUtils.isNotEmpty(cacheName) && StringUtils.isNotEmpty(key)) {
            Cache cache = cacheManager.getCache(cacheName);
            cache.put(key, value);
        }
    }

    /**
     * 插入缓存(有时间)
     *
     * @param cacheName
     * @param key
     * @param value
     * @param time
     */
    public void insert(String cacheName, String key, Object value, int time) {
        if (StringUtils.isNotEmpty(cacheName) && StringUtils.isNotEmpty(key)) {
            Cache cache = cacheManager.getCache(cacheName);
            Ehcache ehcache = (Ehcache) cache.getNativeCache();
            Element element = new Element(key, value, time, time);
            ehcache.put(element);
        }
    }

    /**
     * 指定缓存失效时间
     *
     * @param key
     * @param time
     * @return
     */
    public void expire(String cacheName, String key, int time) {
        if (time > 0) {
            insert(cacheName, key, time);
        } else {
            insert(cacheName, key, RedisUtil.CAHCEHOUR);
        }
    }

    /**
     * 删除指定key
     *
     * @param key
     */
    public void remove(String key) {
        Collection<String> cacheNames = cacheManager.getCacheNames();
        if (CollectionUtil.isNotEmpty(cacheNames)) {
            for (String name : cacheNames) {
                Cache cache = cacheManager.getCache(name);
                Ehcache ehcache = (Ehcache) cache.getNativeCache();
                ehcache.remove(key);
            }
        }
    }

    /**
     * 删除指定key
     *
     * @param key
     */
    public void remove(String cacheName, String key) {
        if (StringUtils.isNotEmpty(cacheName) && StringUtils.isNotEmpty(key)) {
            Cache cache = cacheManager.getCache(cacheName);
            Ehcache ehcache = (Ehcache) cache.getNativeCache();
            ehcache.remove(key);
        }
    }

    /**
     * 删除指定map的key
     *
     * @param key
     */
    public void removeHash(String cacheName, String key) {
        if (StringUtils.isNotEmpty(cacheName) && StringUtils.isNotEmpty(key)) {
            Cache cache = cacheManager.getCache(cacheName);
            Ehcache ehcache = (Ehcache) cache.getNativeCache();
            ehcache.remove(key);
        }
    }

    /**
     * 查
     */
    public Object query(String key) {
        Collection<String> cacheNames = cacheManager.getCacheNames();
        if (CollectionUtil.isNotEmpty(cacheNames)) {
            for (String name : cacheNames) {
                Cache cache = cacheManager.getCache(name);
                Cache.ValueWrapper valueWrapper = cache.get(key);
                if (ObjectUtil.isNotEmpty(valueWrapper)) {
                    return valueWrapper.get();
                }
            }
        }
        return null;
    }


    /**
     * 查
     */
    public Object query(String cacheName, String key) {
        if (StringUtils.isNotEmpty(cacheName) && StringUtils.isNotEmpty(key)) {
            Cache cache = cacheManager.getCache(cacheName);
            Cache.ValueWrapper valueWrapper = cache.get(key);
            return ObjectUtil.isEmpty(valueWrapper) ? null : valueWrapper.get();
        } else {
            return null;
        }
    }


    /**
     * 获取所有的map缓存
     *
     * @param key
     * @return
     */
    public <K, V> Map<K, V> getMap(String key) {
        if (StringUtils.isNotEmpty(key)) {
            Object o = this.query(key);
            if (ObjectUtil.isNotEmpty(o) && o instanceof Map) {
                return (Map) o;
            }
        }
        return null;
    }

    /**
     * 获取所有的List缓存
     *
     * @param key
     * @return
     */
    public <V> List<V> getList(String key) {
        if (StringUtils.isNotEmpty(key)) {
            Object o = this.query(key);
            if (ObjectUtil.isNotEmpty(o) && o instanceof List) {
                return (List) o;
            }
        }
        return null;
    }

    /**
     * 根据key获取Set中的所有值
     *
     * @param key 键
     * @return
     */
    public Set<Object> getSet(String key) {
        if (StringUtils.isNotEmpty(key)) {
            Object o = this.query(key);
            if (ObjectUtil.isNotEmpty(o) && o instanceof Set) {
                return (Set) o;
            }
        }
        return null;
    }

    /**
     * 判断缓存中是否有对应的value
     *
     * @param key
     * @return
     */
    public boolean exists(final String key) {
        return this.exists(null, key);
    }

    /**
     * 判断缓存中是否有对应的value
     *
     * @param key
     * @return
     */
    public boolean exists(String cacheName, final String key) {
        if (StringUtils.isNotEmpty(key)) {
            Collection<String> cacheNames = new ArrayList<>();
            if (StringUtils.isEmpty(cacheName)) {
                cacheNames = cacheManager.getCacheNames();
            } else {
                cacheNames.add(cacheName);
            }

            if (CollectionUtil.isNotEmpty(cacheNames)) {
                for (String name : cacheNames) {
                    Cache cache = cacheManager.getCache(name);
                    Cache.ValueWrapper valueWrapper = cache.get(key);
                    return ObjectUtil.isNotEmpty(valueWrapper);
                }
            }
        }
        return false;
    }

    /**
     * 获取所有的key
     *
     * @return
     */
    public Set<String> getAllKeys() {
        return this.getAllKeys(null);
    }

    /**
     * 获取指定cacheName所有的key
     *
     * @param cacheName
     * @return
     */
    public Set<String> getAllKeys(String cacheName) {
        Set<String> keys = new HashSet<>();
        Collection<String> cacheNames = new ArrayList<>();
        if (StringUtils.isEmpty(cacheName)) {
            cacheNames = cacheManager.getCacheNames();
        } else {
            cacheNames.add(cacheName);
        }

        if (CollectionUtil.isNotEmpty(cacheNames)) {
            for (String name : cacheNames) {
                Cache cache = cacheManager.getCache(name);
                Ehcache ehcache = (Ehcache) cache.getNativeCache();
                List keys1 = ehcache.getKeys();
                keys.addAll(keys1);
            }
        }
        return keys;
    }


    /**
     * 获取所有的key
     *
     * @return
     */
    public Set<String> getAllVisiualKeys() {
        Set<String> ehcacheAllKeys = this.getAllKeys(null);
        return ehcacheAllKeys.stream().filter(o -> o.contains(cacheKeyUtil.getAllUser()) || o.contains(cacheKeyUtil.getCompanySelect()) || o.contains(cacheKeyUtil.getDictionary()) || o.contains(cacheKeyUtil.getDynamic()) || o.contains(cacheKeyUtil.getOrganizeList()) || o.contains(cacheKeyUtil.getPositionList()) || o.contains(cacheKeyUtil.getVisiualData())).collect(Collectors.toSet());
    }


    /**
     * 返回 key 的剩余的过期时间
     *
     * @param key
     * @return
     */
    public Long getLiveTime(String key) {
        Collection<String> cacheNames = cacheManager.getCacheNames();
        if (CollectionUtil.isNotEmpty(cacheNames)) {
            for (String name : cacheNames) {
                Cache cache = cacheManager.getCache(name);
                Ehcache ehcache = (Ehcache) cache.getNativeCache();
                Element element = ehcache.get(key);
                if (ObjectUtil.isNotEmpty(element)) {
                    return element.getExpirationTime();
                }
            }
        }
        return 0L;
    }

    /**
     * string类型-设置key-value 原子性操作
     * @param key           key
     * @param value         value
     * @throws Exception
     */
    public Boolean setStringByAtomicity(String key, String value) throws Exception{
        Cache cache = cacheManager.getCache(REPEATSUBMIT_CACHE);
        Ehcache ehcache = (Ehcache) cache.getNativeCache();
        Element element = ehcache.get(key);
        if (ObjectUtil.isNotEmpty(element) && value.equals(element.getObjectValue())) {
            return true;
        } else {
            element = new Element(key, value);
            ehcache.put(element);
            return false;
        }
    }
}
