package com.bringspring.common.database.source;

import com.baomidou.mybatisplus.annotation.DbType;
import com.baomidou.mybatisplus.extension.plugins.handler.TableNameHandler;
import com.bringspring.common.database.constant.DbConst;
import com.bringspring.common.database.data.DataSourceContextHolder;
import com.bringspring.common.database.datatype.model.DtModelDTO;
import com.bringspring.common.database.enums.DbAliasEnum;
import com.bringspring.common.database.enums.ParamEnum;
import com.bringspring.common.database.enums.datatype.interfaces.DtInterface;
import com.bringspring.common.database.enums.datatype.viewshow.DataTypeEnum;
import com.bringspring.common.database.model.DataTypeModel;
import com.bringspring.common.database.model.DbTableFieldModel;
import com.bringspring.common.database.model.dbfield.DbFieldModel;
import com.bringspring.common.database.model.dto.DataSourceDTO;
import com.bringspring.common.database.model.entity.DbLinkEntity;
import com.bringspring.common.database.model.interfaces.DbSourceOrDbLink;
import com.bringspring.common.database.source.impl.*;
import com.bringspring.common.database.sql.SqlBase;
import com.bringspring.common.database.sql.model.DbStruct;
import com.bringspring.common.database.util.DataSourceUtil;
import com.bringspring.common.database.util.DbTypeUtil;
import com.bringspring.common.exception.DataException;
import com.bringspring.common.util.StringUtils;
import lombok.Data;

import java.lang.reflect.Method;
import java.sql.ResultSet;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;

/**
 * 数据库基础模型表
 *
 * @author RKKJ开发平台组 YanYu
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2021/6/21
 */
@Data
public abstract class DbBase {

    /**
     * 数据库编码标准
     */
    public static final String MYSQL = "MySQL";
    public static final String DM = "DM8";
    public static final String KINGBASE_ES = "KingbaseES";
    public static final String ORACLE = "Oracle";
    public static final String POSTGRE_SQL = "PostgreSQL";
    public static final String SQL_SERVER = "SQLServer";
    public static final String DORIS = "Doris";

    public static final DbBase[] DB_BASES = {new DbMySQL(),
            new DbSQLServer(), new DbDM(), new DbOracle(), new DbKingbase(), new DbPostgre()};

    public static final String[] DB_ENCODES = {MYSQL, DM, KINGBASE_ES, ORACLE, POSTGRE_SQL, SQL_SERVER};

    /**
     * 数据库编码
     */
    protected String dbEncode;
    /**
     * MybatisPlus数据库编码
     */
    protected DbType mpDbType;
    /**
     * url里数据库标识
     */
    protected String connUrlEncode;
    /**
     * 数据库驱动
     */
    protected String driver;
    /**
     * 默认端口
     */
    protected String defaultPort;
    /**
     * 默认预备url
     */
    protected String defaultPrepareUrl;
    /**
     * 获取SQL基础模型
     */
    protected SqlBase sqlBase;

    /**
     * oracle连接扩展参数
     */
    public String oracleParam;

    /**
     * DruidDbType
     */
    protected DbType druidDbType;
    /**
     * 管理员用户名
     */
    protected String dbaUsername;


    /**
     * 无参构造
     */
    protected DbBase() {
        init();
    }

    /**
     * 初始赋值
     */
    protected void init() {}

    /**
     * 数据库对象初始化
     * 指定子类被创建时，需要提供的参数
     */
    protected void setInstance(String dbEncode, DbType mpDbType, String defaultPort, String connUrlEncode,
                               String driver, String defaultPrepareUrl, SqlBase sqlBase) {
        // 绑定：数据库编码
        this.dbEncode = dbEncode;
        // 绑定：MybatisPlus数据库编码
        this.mpDbType = mpDbType;
        // 绑定：Url数据库标识
        this.connUrlEncode = connUrlEncode;
        // 绑定：数据库SQL对象类
        this.sqlBase = sqlBase;
        this.driver = driver;
        this.defaultPrepareUrl = defaultPrepareUrl;
        // 默认端口
        this.defaultPort = defaultPort;
    }

    @Override
    public String toString() {
        return "DbBase{" +
                "dbEncode='" + dbEncode + '\'' +
                ", mpDbType=" + mpDbType +
                ", connUrlEncode='" + connUrlEncode + '\'' +
                ", driver='" + driver + '\'' +
                ", defaultPort='" + defaultPort + '\'' +
                ", defaultPrepareUrl='" + defaultPrepareUrl + '\'' +
                ", sqlBase=" + sqlBase +
                ", oracleParam='" + oracleParam + '\'' +
                '}';
    }

    /**
     * 获取数据类型
     * 使用反射方法，抽取所有子类里面的复用方法。
     *
     * @param dte 数据类型枚举
     * @return 数据类型code
     */
    public DataTypeModel getDataTypeModel(DataTypeEnum dte) throws DataException {
        try{
            // DM8 后期要统一改成 DM
            String jsbosDbEncode = this.getDbEncode().equals(DM) ? "DM" : this.getDbEncode();
            Class<DataTypeEnum> clz = (Class<DataTypeEnum>) Class.forName("com.bringspring.common.database.enums.datatype.viewshow.DataTypeEnum");
            // 方法命名规则：getDt + jsbosDbEncode
            Method method = clz.getMethod("getDt" + jsbosDbEncode);
            DtInterface dt = (DtInterface) method.invoke(dte);
            return dt.getDataTypeModel();
        }catch (Exception e){
            throw new DataException(e.getMessage());
        }
    }

    /**
     * 获取数据库字段类型模型 通过view字段类型
     *
     * @param dbFieldType 数据库字段类型模型
     * @return 数据库字段类型模型
     */
    public DataTypeModel getDataTypeModel(String dbFieldType) throws Exception {
        // DM8 后期要统一改成 DM
        String jsbosDbEncode = this.getDbEncode().equals(DM) ? "DM" : this.getDbEncode();
        // 数据类型枚举类命名规则：Dt + jsbosDbEncode
        Class<DtInterface> clz = (Class<DtInterface>) Class.forName("com.bringspring.common.database.enums.datatype.Dt" + jsbosDbEncode);
        Method method = clz.getMethod("values");
        Object result = method.invoke(null);
        DtInterface[] dataTypes;
        if (result instanceof DtInterface[]) { // 运行时校验类型
            dataTypes = (DtInterface[]) result;
        } else {
            // 处理类型不匹配的情况（比如抛异常、返回空数组）
//            throw new IllegalStateException("反射方法返回值不是 DtInterface[] 类型");
            dataTypes = new DtInterface[0];
        }
        for (DtInterface dataType : dataTypes) {
            if (dbFieldType.equals(dataType.getDbFieldType())) {
                return dataType.getDataTypeModel();
            }
        }
        return null;
    }



    public static List<String> dynamicAllTableName = Collections.emptyList();

    /**
     * 获取最终动态表名， 处理是否动态表名
     * @return
     */
    public TableNameHandler getDynamicTableNameHandler(){
        return (sql, tableName) -> {
            //是否租户系统指定数据源
            boolean isAssignDataSource = StringUtils.isNotEmpty(DataSourceContextHolder.getDatasourceName()) && "true".equals(DataSourceContextHolder.getDatasourceName());
            if(isAssignDataSource){
                return tableName;
            } else{
                //是否指定数据源, 且在初始库中包含的表
                boolean hasDataSource = StringUtils.isNotEmpty(DataSourceContextHolder.getDatasourceName())
                        && dynamicAllTableName.contains(tableName.toLowerCase());
                return hasDataSource ? getDynamicTableName(tableName) : tableName;
            }
        };
    }

    /**
     * 获取动态组合表名
     * @param tableName
     * @return
     */
    protected String getDynamicTableName(String tableName){
        return DataSourceContextHolder.getDatasourceName() + "." + tableName;
    }

    /**
     * 获取部分字段信息
     *
     * @param result 结果集
     * @return 表字段模型
     * @throws DataException ignore
     */
    public DbTableFieldModel getPartFieldModel(ResultSet result) throws Exception {
        return new DbTableFieldModel();
    }

    /**
     * 获取数据库连接Url   关键参数：
     * 1、地址
     * 2、端口
     * 3、数据库名
     * 4、模式 （参数：?currentSchema = schema）
     * 5、jdbc-url自定义参数
     *
     * 此方法对DbTypeUtil与内部开放，对外关闭。外部调用url，用DbTypeUtil.getUrl()方法
     * @return String 连接
     */
    protected String getConnUrl(String prepareUrl, String host, Integer port, String dbName, String schema){
        // 配置文件是否存在自定义数据连接url
        if (StringUtils.isEmpty(prepareUrl)) {
            prepareUrl = this.defaultPrepareUrl;
        }
        if(StringUtils.isNotEmpty(dbName)){
            prepareUrl = prepareUrl.replace(DbConst.DB_NAME, dbName);
        }
        // 模式替换
        if(StringUtils.isNotEmpty(schema)){
            prepareUrl = prepareUrl.replace(DbConst.DB_SCHEMA, schema);
        }
        if(StringUtils.isNotEmpty(host)){
            prepareUrl =  prepareUrl.replace(DbConst.HOST, host);
        }
        if(port != null){
            prepareUrl =  prepareUrl.replace(DbConst.PORT, port.toString());
        }
        return prepareUrl;
    }

    /**
     * 不同数据库结构性
     *
     * @param structParams 结构参数
     * @param table 表
     * @return 转换后SQL语句
     */
    public LinkedList<Object> getStructParams(String structParams, String table, DataSourceUtil dbSourceOrDbLink) {
        DataSourceDTO dto = dbSourceOrDbLink.convertDTO();
        LinkedList<Object> data = new LinkedList<>();
        for(String paramStr : structParams.split(":")){
            if(paramStr.equals(ParamEnum.TABLE.getTarget())){
                data.add(table);
            }else if(paramStr.equals(ParamEnum.DB_NAME.getTarget())){
                // 自动库名
                String dbName = dto.getDbName();
                if(dbName == null){
                    dbName = dto.getUserName();
                }
                data.add(dbName);
            }else if(paramStr.equals(ParamEnum.DB_SCHEMA.getTarget())){
                data.add(dto.getDbSchema());
            }else if(paramStr.equals(ParamEnum.TABLE_SPACE.getTarget())) {
                data.add(dto.getDbTableSpace());
            }
        }
        return data;
    }

    /*
     * 提供内部封装方法所独有的方法调用
     * 保持全局只有一处显性getUrl,getConn的方法（即在ConnUtil里）======================================
     */

//    public static class BaseCommon{
//        public static String getDbBaseConnUrl(DbBase dbBase, String prepareUrl, String host, Integer port, String dbName, String schema){
//            return dbBase.getConnUrl(prepareUrl, host, port, dbName, schema);
//        }
//    }


    /**
     * 数据库对象初始化
     * 指定子类被创建时，需要提供的参数
     */
    protected void setInstance(String dbEncode, DbType mpDbType, DbType druidDbType, String defaultPort, String dbaUsername, String connUrlEncode,
                               String driver, String defaultPrepareUrl) {
        // 绑定：JNPF数据库编码
        this.dbEncode = dbEncode;
        // 绑定：MybatisPlus数据库编码
        this.mpDbType = mpDbType;
        this.druidDbType = druidDbType;
        // 绑定：Url数据库标识
        this.connUrlEncode = connUrlEncode;
        this.driver = driver;
        this.defaultPrepareUrl = defaultPrepareUrl;
        // 默认端口
        this.defaultPort = defaultPort;
        this.dbaUsername = dbaUsername;
    }

    public DbBase[] DB_BASES(){
        return null;
    }



    /**
     * 不同库间设置字段信息
     *
     * @param model 字段模型
     * @param result 结果集
     * @return 表字段模型
     * @throws DataException ignore
     */
    public void setPartFieldModel(DbFieldModel model, ResultSet result) throws Exception{
        model.setDtModelDTO(new DtModelDTO(
                com.bringspring.common.database.datatype.db.interfaces.DtInterface.newInstanceByDt(model.getDataType(), this.getDbEncode()),
                result.getLong(DbAliasEnum.CHAR_LENGTH.getAlias(dbEncode)),
                result.getInt(DbAliasEnum.NUM_PRECISION.getAlias(dbEncode)),
                result.getInt(DbAliasEnum.NUM_SCALE.getAlias(dbEncode))
        ));
    }

    /**
     * 获取数据库连接Url   关键参数：
     * 1、地址
     * 2、端口
     * 3、数据库名
     * 4、模式 （参数：?currentSchema = schema）
     * 5、jdbc-url自定义参数
     *
     * 此方法对DbTypeUtil与内部开放，对外关闭。外部调用url，用DbTypeUtil.getUrl()方法
     * @return String 连接
     */
    protected String getConnUrl(String prepareUrl, String host, Integer port, DbStruct struct){
        // 配置文件是否存在自定义数据连接url
        prepareUrl = StringUtils.isNotEmpty(prepareUrl) ? prepareUrl : defaultPrepareUrl;
        // 当地址为空，用本地回环地址
        prepareUrl =  prepareUrl.replace(DbConst.HOST, StringUtils.isNotEmpty(host) ? host : "127.0.0.1");
        // 当端口为空，用数据库一般默认端口
        prepareUrl =  prepareUrl.replace(DbConst.PORT, port != null ? port.toString() : defaultPort);
        return prepareUrl;
    }

    /*
     * 提供内部封装方法所独有的方法调用
     * 保持全局只有一处显性getUrl,getConn的方法（即在ConnUtil里）======================================
     */

    public static class BaseCommon{
        public static String getDbBaseConnUrl(DbSourceOrDbLink dbSourceOrDbLink, String dbName){
            DbLinkEntity dsd = dbSourceOrDbLink.init(dbName);
            try {
                return DbTypeUtil.getDb(dbSourceOrDbLink).getConnUrl(dsd.getPrepareUrl(), dsd.getHost(), dsd.getPort(), dsd.getDbStruct());
            } catch (DataException e) {
                e.printStackTrace();
            }
            return "";
        }
    }
}
