package com.bringspring.common.database.util;

import com.baomidou.mybatisplus.annotation.TableField;
import com.bringspring.common.base.entity.SuperExtendEntity;
import com.bringspring.common.database.model.dto.DataSourceDTO;
import com.bringspring.common.database.model.entity.DbLinkEntity;
import com.bringspring.common.database.model.interfaces.DbSourceOrDbLink;
import com.bringspring.common.database.sql.model.DbStruct;
import com.bringspring.common.exception.DataException;
import com.bringspring.common.util.StringUtils;
import lombok.Data;
import org.springframework.beans.BeanUtils;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.stereotype.Component;

/**
 * 数据源基础工具类
 *
 * @author JNPF开发平台组
 * @version V3.1.0
 * @copyright 引迈信息技术有限公司
 * @date 2021/3/16 8:49
 */
@Data
@Component
@ConfigurationProperties(prefix = "spring.datasource")
public class DataSourceUtil extends SuperExtendEntity.SuperExtendDescriptionEntity implements DbSourceOrDbLink {

    /**
     * 数据库类型
     */
    @TableField("db_type")
    private String dbType;

    /**
     * 主机ip
     */
    @TableField("host")
    private String host;

    /**
     * 端口
     */
    @TableField("port")
    private Integer port;

    /**
     * 库名
     */
    @TableField("service_name")
    private String serviceName;

    @TableField("db_name")
    private String dbName;
    /**
     * 用户
     */
    @TableField("user_name")
    private String userName;

    /**
     * 密码
     */
    @TableField("password")
    private String password;

    /**
     * 表空间
     */
    @TableField(value = "table_space")
    private String dbTableSpace;

    /**
     * 模式
     */
    @TableField(value = "db_schema")
    private String dbSchema;

    /**
     * 数据连接jdbc-url参数
     */
    @TableField(exist = false)
    private String urlParams;

    /**
     * url地址
     */
    @TableField(exist = false)
    private String url;

    /**
     * 数据连接jdbc-url参数
     */
    @TableField(exist = false)
    private String prepareUrl;

    /**
     * 驱动包
     */
    @TableField(exist = false)
    private String driver;

    /**
     * oracle多方式登录参数
     */
    @TableField(value = "oracle_param")
    private String oracleParam;

    public String getAutoUsername() throws DataException {
        if(StringUtils.isEmpty(this.userName)){
            return DbTypeUtil.getEncodeDb(this.dbType).getDbaUsername();
        }else {
            return this.userName;
        }
    }

    public String getAutoPassword() throws DataException {
        if(StringUtils.isEmpty(this.password)){
            return getAutoUsername();
        }else {
            return this.password;
        }
    }

    /**
     * -- 这里的参数dataSourceUtil是spring托管的全局唯一变量，
     * New对象防止数据源互串，防止Bean覆盖
     * 获取数据源参数传参对象
     * 注意：此处方法不能命名为 get开头的名字，
     * 会出现copy bean转换时候的错误
     */
    public DbLinkEntity init(){
        return init(null);
    }

    /**
     * 多租户：获取数据源参数传参对象
     * @param dbName 库名
     * @return ignore
     */
    public DbLinkEntity init(String dbName){
        DbLinkEntity dbLinkEntity = new DbLinkEntity();
        BeanUtils.copyProperties(this, dbLinkEntity);
        if(StringUtils.isNotEmpty(dbName)){
            dbLinkEntity.setDbName(dbName);
        }
        return dbLinkEntity;
    }

    /**
     * 数据基础结构信息
     */
    public DbStruct getDbStruct(){
        DbStruct dbStruct = new DbStruct();
        // 用户名对应 (当数据库为Oracle与DM的时，schema默认与用户同名)
        dbStruct.setUserName(getUserName());
        dbStruct.setOracleDbSchema(getUserName());
        dbStruct.setDmDbSchema(getUserName());
        // 表空间
        dbStruct.setDbTableSpace(getDbTableSpace());
        //库名
        dbStruct.setMysqlDbName(getDbName());
        dbStruct.setSqlServerDbName(getDbName());
        dbStruct.setKingBaseDbName(getDbName());
        dbStruct.setPostGreDbName(getDbName());
        // 模式
        if(StringUtils.isNotEmpty(getDbSchema())){
            dbStruct.setOracleDbSchema(getDbSchema());
            dbStruct.setDmDbSchema(getDbSchema());
            dbStruct.setSqlServerDbSchema(getDbSchema());
            dbStruct.setKingBaseDbSchema(getDbSchema());
            dbStruct.setPostGreDbSchema(getDbSchema());
        }
        dbStruct.setOracleParam(getOracleParam());
        return dbStruct;
    }

    /**
     * -- 这里的参数dataSourceUtil是spring托管的全局唯一变量，
     * New对象防止数据源互串，防止Bean覆盖
     */
    @Override
    public DataSourceDTO convertDTO(){
        return convertDTO(null);
    }

    @Override
    public DataSourceDTO convertDTO(String dbName){
        return convertDTO(dbName, 1);
    }

    protected DataSourceDTO convertDTO(String dbName, Integer dataSourceFrom){
        DataSourceDTO dataSourceDTO = new DataSourceDTO();
        BeanUtils.copyProperties(this, dataSourceDTO);
        if(StringUtils.isNotEmpty(dbName)){
            dataSourceDTO.setDbName(dbName);
        }
        dataSourceDTO.setDataSourceFrom(dataSourceFrom);
        return dataSourceDTO;
    }

}
