package com.bringspring.common.util;

import cn.hutool.http.useragent.UserAgent;
import cn.hutool.http.useragent.UserAgentUtil;
import com.bringspring.common.constant.GlobalConst;
import jakarta.servlet.http.HttpServletRequest;
import jakarta.servlet.http.HttpServletResponse;
import jakarta.servlet.http.HttpSession;
import org.springframework.util.AntPathMatcher;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import java.io.IOException;
import java.util.Collections;
import java.util.Enumeration;
import java.util.LinkedHashMap;
import java.util.Map;

/**
 * 客户端工具类
 *
 * @author jnpf
 */
public class ServletUtil {


    private static AntPathMatcher matcher = new AntPathMatcher();

    /**
     * 获取request
     */
    public static HttpServletRequest getRequest() {
        try {
            return getRequestAttributes().getRequest();
        } catch (Exception e) {
            return null;
        }
    }

    public static String getHeader(String name) {
        if (getRequest() != null) {
            return getRequest().getHeader(name);
        }
        return null;
    }


    /**
     * 判断是否是手机端登陆
     */
    public static boolean getIsMobileDevice() {
        return isMobileDevice(ServletUtil.getUserAgent());
    }

    /**
     * 判断是否是手机端登陆
     */
    public static boolean getIsMobileDevice(String userAgent) {
        return isMobileDevice(userAgent);
    }


    /**
     * 获取User-Agent
     */
    public static String getUserAgent() {
        return ServletUtil.getHeader("User-Agent");
    }

    /**
     * 判断是否是移动设备
     *
     * @param requestHeader
     * @return
     */
    public static boolean isMobileDevice(String requestHeader) {
        UserAgent userAgent = UserAgentUtil.parse(requestHeader);
        if (userAgent == null) {
            return false;
        }
        return userAgent.isMobile();
    }

    /**
     * 获取ServletPath
     */
    public static String getServletPath() {
        return ServletUtil.getRequest().getServletPath();
    }

    /**
     * 获取response
     */
    public static HttpServletResponse getResponse() {
        try {
            return getRequestAttributes().getResponse();
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 获取session
     */
    public static HttpSession getSession() {
        return getRequest().getSession();
    }

    public static ServletRequestAttributes getRequestAttributes() {
        try {
            RequestAttributes attributes = RequestContextHolder.getRequestAttributes();
            return (ServletRequestAttributes) attributes;
        } catch (Exception e) {
            return null;
        }
    }

    public static Map<String, String> getHeaders(HttpServletRequest request) {
        Map<String, String> map = new LinkedHashMap<>();
        Enumeration<String> enumeration = request.getHeaderNames();
        if (enumeration != null) {
            while (enumeration.hasMoreElements()) {
                String key = enumeration.nextElement();
                String value = request.getHeader(key);
                map.put(key, value);
            }
        }
        return map;
    }

    public static Map<String, Object> getParams(HttpServletRequest request) {
        Map<String, Object> parameters = new LinkedHashMap<>();
        Enumeration<String> parameterNames = request.getParameterNames();
        if (parameterNames != null) {
            while (parameterNames.hasMoreElements()) {
                String key = parameterNames.nextElement();
                if (key == null) {
                    continue;
                }
                parameters.put(key, request.getParameter(key));
            }
        }
        return parameters;
    }

    /**
     * 将字符串渲染到客户端
     *
     * @param response 渲染对象
     * @param string   待渲染的字符串
     * @return null
     */
    public static String renderString(HttpServletResponse response, String string) {
        try {
            response.setStatus(200);
            response.setContentType("application/json");
            response.setCharacterEncoding("utf-8");
            response.getWriter().print(string);
        } catch (IOException e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * 是否是Ajax异步请求
     *
     * @param request
     */
    public static boolean isAjaxRequest(HttpServletRequest request) {
        String accept = request.getHeader("accept");
        if (accept != null && accept.indexOf("application/json") != -1) {
            return true;
        }

        String xRequestedWith = request.getHeader("X-Requested-With");
        if (xRequestedWith != null && xRequestedWith.indexOf("XMLHttpRequest") != -1) {
            return true;
        }

        String uri = request.getRequestURI();
        if (StringUtils.inStringIgnoreCase(uri, ".json", ".xml")) {
            return true;
        }

        String ajax = request.getParameter("__ajax");
        if (StringUtils.inStringIgnoreCase(ajax, "json", "xml")) {
            return true;
        }
        return false;
    }

    /**
     * 获取URL中的path变量
     *
     * @param pattern
     * @param path
     * @return
     */
    public static Map<String, String> getPathVariables(String pattern, String path) {
        Map<String, String> vars = null;
        try {
            if (!StringUtils.isEmpty(path)) {
                vars = matcher.extractUriTemplateVariables(pattern, path);
            }
        } catch (Exception e) {
        }
        if (vars == null) {
            vars = Collections.EMPTY_MAP;
        }
        return vars;
    }


    /**
     * 获取当前访问地址中的path变量
     *
     * @param pattern
     * @return
     */
    public static Map<String, String> getPathVariables(String pattern) {
        return getPathVariables(pattern, getServletPath());
    }

    public static String getRequestHost() {
        HttpServletRequest request = getRequest();
        if (request != null) {
            String host = request.getHeader(GlobalConst.HEADER_HOST);
            if (StringUtils.isEmpty(host)) {
                host = request.getHeader("X-Forwarded-Host");
                if (StringUtils.isNotEmpty(host)) {
                    int index = host.lastIndexOf(",");
                    if (index != -1) {
                        return host.substring(index);
                    } else {
                        return host;
                    }
                } else {
                    host = request.getHeader("Host");
                }
            }
            if (StringUtils.isNotEmpty(host)) {
                return host;
            }
        }
        return StringUtils.EMPTY;
    }
}
