package com.bringspring.common.nc.util;

import cn.hutool.json.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * NC同步工具类
 * 封装NC同步相关的通用方法
 */
@Slf4j
@Component
public class NcSyncUtil {

    @Autowired
    private HttpRequestClientUtil httpClientUtil;

    /**
     * 通用同步方法
     * @param moduleType 模块类型 (NC/ECS)
     * @param dataId 数据ID
     * @param dataCode 数据编码
     * @param dataName 数据名称
     * @param ufInterface UF接口对象
     * @return 同步结果
     */
    public boolean syncToNc(String moduleType, String dataId, String dataCode, String dataName, Object ufInterface) {
        try {
            // 转换为 XML
            String toXml = XmlConverter.convertToXml(ufInterface);
            // 日志记录
            log.info("数据推送到NC，入参: {}", XmlConverter.convertXmlToJson(toXml));
            // 发送请求
            String sendRequest = httpClientUtil.sendRequest(moduleType, dataId, dataCode, dataName, toXml);
            log.info("NC响应结果: {}", sendRequest);
            // 处理响应
            return processResponse(sendRequest);
        } catch (Exception e) {
            log.error("同步数据到NC失败，数据ID: {}, 错误信息: {}", dataId, e.getMessage(), e);
            return false;
        }
    }

    /**
     * 处理NC响应结果
     * @param response 响应字符串
     * @return 是否成功
     */
    public boolean processResponse(String response) {
        try {
            if (response == null || response.trim().isEmpty()) {
                log.warn("NC响应为空");
                return false;
            }
            // 处理响应
            JSONObject jsonObject = XmlConverter.convertXmlToJson(response);
            JSONObject sendResult = jsonObject.getJSONObject("ufinterface").getJSONObject("sendresult");
            return sendResult != null && sendResult.getInt("resultcode") == 1;
        } catch (Exception e) {
            log.error("解析NC响应失败: {}", e.getMessage(), e);
            return false;
        }
    }

    /**
     * 构建通用UfInterface基础信息
     * @param ufInterface UF接口对象
     * @param billType 单据类型
     */
    public void buildCommonUfInterface(Object ufInterface, String billType) {
        try {
            // 使用反射设置通用属性
            ufInterface.getClass().getMethod("setAccount", String.class).invoke(ufInterface, "RKJT");
            ufInterface.getClass().getMethod("setBilltype", String.class).invoke(ufInterface, billType);
            ufInterface.getClass().getMethod("setGroupcode", String.class).invoke(ufInterface, "00");
            ufInterface.getClass().getMethod("setFilename", String.class).invoke(ufInterface, "");
            ufInterface.getClass().getMethod("setIsexchange", String.class).invoke(ufInterface, "Y");
            ufInterface.getClass().getMethod("setReplace", String.class).invoke(ufInterface, "Y");
            ufInterface.getClass().getMethod("setRoottag", String.class).invoke(ufInterface, "");
            ufInterface.getClass().getMethod("setSender", String.class).invoke(ufInterface, "003");
        } catch (Exception e) {
            log.error("设置通用UfInterface属性失败: {}", e.getMessage(), e);
        }
    }
}