package com.bringspring.example.controller;

import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.Pagination;
import com.bringspring.common.base.vo.PaginationVO;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.example.entity.ContractEntity;
import com.bringspring.example.model.ContractForm;
import com.bringspring.example.model.ContractInfoVO;
import com.bringspring.example.model.ContractListVO;
import com.bringspring.example.service.ContractService;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;

/**
 *
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2021/3/16 9:47
 */
@Tag(name = "合同管理", description = "合同信息的增删改查操作")
@RestController
@RequestMapping("/Contract")
public class ContractController {

    @Autowired
    private ContractService contractService;

    @Operation(summary = "获取合同列表", description = "分页获取合同信息列表")
    @ApiResponses({
        @ApiResponse(responseCode = "200", description = "成功获取合同列表"),
        @ApiResponse(responseCode = "400", description = "参数错误"),
        @ApiResponse(responseCode = "500", description = "服务器内部错误")
    })
    @GetMapping("/List")
    public ActionResult list(
        @Parameter(description = "分页参数") Pagination pagination) {
        List<ContractEntity> entity = contractService.getlist(pagination);
        List<ContractListVO> listVo = JsonUtil.getJsonToList(JsonUtil.getObjectToStringDateFormat(entity, "yyyy-MM-dd HH:mm:ss"),ContractListVO.class );
        PaginationVO vo = JsonUtil.getJsonToBean(pagination,PaginationVO.class);
        return ActionResult.page(listVo,vo);
    }

    @Operation(summary = "获取合同详情", description = "根据ID获取合同详细信息")
    @ApiResponses({
        @ApiResponse(responseCode = "200", description = "成功获取合同详情"),
        @ApiResponse(responseCode = "404", description = "合同不存在"),
        @ApiResponse(responseCode = "500", description = "服务器内部错误")
    })
    @GetMapping("/{id}")
    public ActionResult info(
        @Parameter(description = "合同ID", required = true) @PathVariable("id") String id) {
        ContractEntity entity = contractService.getInfo(id);
        ContractInfoVO vo = JsonUtil.getJsonToBean(entity, ContractInfoVO.class);
        return ActionResult.success(vo);
    }

    @Operation(summary = "创建合同", description = "创建新的合同信息")
    @ApiResponses({
        @ApiResponse(responseCode = "200", description = "合同创建成功"),
        @ApiResponse(responseCode = "400", description = "参数验证失败"),
        @ApiResponse(responseCode = "500", description = "服务器内部错误")
    })
    @PostMapping
    public ActionResult create(
        @Parameter(description = "合同表单数据", required = true) @RequestBody @Valid ContractForm contractForm) {
        ContractEntity entity = JsonUtil.getJsonToBean(contractForm, ContractEntity.class);
        contractService.create(entity);
        return ActionResult.success(MsgCode.SU002.get());
    }

    @Operation(summary = "更新合同", description = "根据ID更新合同信息")
    @ApiResponses({
        @ApiResponse(responseCode = "200", description = "合同更新成功"),
        @ApiResponse(responseCode = "400", description = "参数验证失败"),
        @ApiResponse(responseCode = "404", description = "合同不存在"),
        @ApiResponse(responseCode = "500", description = "服务器内部错误")
    })
    @PutMapping("/{id}")
    public ActionResult update(
        @Parameter(description = "合同ID", required = true) @PathVariable("id") String id,
        @Parameter(description = "合同表单数据", required = true) @RequestBody @Valid ContractForm contractForm) {
        ContractEntity entity = JsonUtil.getJsonToBean(contractForm, ContractEntity.class);
        contractService.update(id,entity);
        return ActionResult.success(MsgCode.SU004.get());
    }

    @Operation(summary = "删除合同", description = "根据ID删除合同信息")
    @ApiResponses({
        @ApiResponse(responseCode = "200", description = "合同删除成功"),
        @ApiResponse(responseCode = "404", description = "合同不存在"),
        @ApiResponse(responseCode = "500", description = "服务器内部错误")
    })
    @DeleteMapping("/{id}")
    public ActionResult delete(
        @Parameter(description = "合同ID", required = true) @PathVariable("id") String id) {
        ContractEntity entity = contractService.getInfo(id);
        contractService.delete(entity);
        return ActionResult.success(MsgCode.SU003.get());
    }

}
