package com.bringspring.extend.controller;

import cn.hutool.core.util.ObjectUtil;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.bringspring.common.annotation.NotCheckLogin;
import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.NoDataSourceBind;
import com.bringspring.common.base.Page;
import com.bringspring.common.base.UserInfo;
import com.bringspring.common.base.vo.DownloadVO;
import com.bringspring.common.base.vo.ListVO;
import com.bringspring.common.config.ConfigValueUtil;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.exception.DataException;
import com.bringspring.common.util.*;
import com.bringspring.common.util.enums.FileTypeEnum;
import com.bringspring.common.util.file.UploadUtil;
import com.bringspring.common.util.treeutil.SumTree;
import com.bringspring.common.util.treeutil.newtreeutil.TreeDotUtils;
import com.bringspring.extend.entity.DocumentEntity;
import com.bringspring.extend.entity.DocumentHistoryEntity;
import com.bringspring.extend.entity.DocumentShareEntity;
import com.bringspring.extend.model.document.*;
import com.bringspring.extend.service.DocumentHistoryService;
import com.bringspring.extend.service.DocumentService;
import com.bringspring.extend.service.DocumentShareService;
import com.bringspring.extend.utils.CommonUtil;
import com.bringspring.files.model.MergeChunkDto;
import com.bringspring.system.base.util.JsonUtilEx;
import com.bringspring.system.base.util.OptimizeUtil;
import com.bringspring.system.permission.entity.OrganizeEntity;
import com.bringspring.system.permission.service.OrganizeService;
import com.bringspring.system.permission.service.UserService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.Cleanup;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.FileUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.validation.Valid;
import java.io.*;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Collectors;

/**
 * 文档管理
 *
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2017年9月26日 上午9:18
 */
@Slf4j
@Api(tags = "知识管理", value = "Document")
@RestController
@RequestMapping("/api/extend/Document")
public class DocumentController {

    @Autowired
    private DocumentService documentService;
    @Autowired
    private ConfigValueUtil configValueUtil;
    @Autowired
    private UserService userService;
    @Autowired
    private UserProvider userProvider;
    @Autowired
    private DocumentHistoryService documentHistoryService;
    @Autowired
    private DocumentShareService documentShareService;
    @Autowired
    private OrganizeService organizeService;


    /**
     * 信息
     *
     * @param id 主键值
     * @return
     */
    @ApiOperation("获取文件/文件夹信息")
    @GetMapping("/{id}")
    public ActionResult<DocumentInfoVO> info(@PathVariable("id") String id) throws DataException {
        DocumentEntity entity = documentService.getInfo(id);
        DocumentInfoVO vo = JsonUtilEx.getJsonToBeanEx(entity, DocumentInfoVO.class);
        //设置所属公司
        OrganizeEntity org = organizeService.getInfo(vo.getCompanyId());
        if(ObjectUtil.isNotEmpty(org)){
            vo.setCompanyName(org.getFullName());
        }
        //截取后缀
        String[] fullName = vo.getFullName().split("\\.");
        if (fullName.length > 1) {
            String fullNames = "";
            for (int i = 0; i < fullName.length - 1; i++) {
                if (i > 0) {
                    fullNames += "." + fullName[i];
                } else {
                    fullNames += fullName[i];
                }
            }
            vo.setFullName(fullNames);
        }
        return ActionResult.success(vo);
    }

    /**
     * 新建
     *
     * @param documentCrForm 实体对象
     * @return
     */
    @ApiOperation("添加文件夹")
    @PostMapping
    public ActionResult create(@RequestBody @Valid DocumentCrForm documentCrForm) {
        UserInfo userInfo = userProvider.get();
        DocumentEntity entity = JsonUtil.getJsonToBean(documentCrForm, DocumentEntity.class);

        if ("0".equals(entity.getParentId())) {
            entity.setFilePath(userInfo.getUserId()+File.separator+ DateUtil.getNowYear() + File.separator+entity.getFullName());
        }else{
            String absoluteRouteById = documentService.getAbsoluteRouteById(entity.getParentId());
            absoluteRouteById=absoluteRouteById+entity.getFullName();
            entity.setFilePath(absoluteRouteById.substring(configValueUtil.getDocumentFilePath().length()));
        }
        if (documentService.isExistByFullName(entity)) {
            return ActionResult.fail(MsgCode.EXIST004.get());
        }
        entity.setCompanyId(userInfo.getOrganizeId());
        documentService.create(entity);
        return ActionResult.success(MsgCode.SU001.get());
    }

    /**
     * 更新
     *
     * @param id             主键值
     * @param documentUpForm 实体对象
     * @return
     */
    @ApiOperation("修改文件名/文件夹名")
    @PutMapping("/{id}")
    public ActionResult update(@PathVariable("id") String id, @RequestBody @Valid DocumentUpForm documentUpForm) {
        documentUpForm.setFilePath(null);
        DocumentEntity entity = JsonUtil.getJsonToBean(documentUpForm, DocumentEntity.class);
        if (documentService.isExistByFullName(entity)) {
            return ActionResult.fail(MsgCode.EXIST004.get());
        }
        DocumentEntity info = documentService.getInfo(id);
        //获取后缀名
        String[] fullName = info.getFullName().split("\\.");
        if (fullName.length > 1) {
            entity.setFullName(entity.getFullName() + "." + fullName[fullName.length - 1]);
        }
        //如修改的为文件夹名称，则相对应的路径也做修改
        if(0==info.getType()){
            if(StringUtils.isNotEmpty(info.getFilePath())){
                //原数据
                String source = info.getFilePath();
                //需要替换的名称
                String target = info.getFullName();
                //最后一次出现的位置
                int lastIndex= source.lastIndexOf(target); // 技到目标字符串最后一次出现的位置
                //替换最后一个目标
                String result = source.substring(0, lastIndex)+ documentUpForm.getFullName() + source.substring(lastIndex + target.length());
                entity.setFilePath(result);
                UserInfo userInfo = userProvider.get();
                //处理当前记录下的所有数据，修改文件夹路径，暂时未改，修改后无法下载，需要同步更改服务器文件夹名称
//                getChildNodes(info.getId(),source,result);
            }
        }
        boolean flag = documentService.update(id, entity);
        if (flag == false) {
            return ActionResult.fail(MsgCode.FA002.get());
        }
        return ActionResult.success(MsgCode.SU004.get());
    }

    /**
     * 递归查询所有子节点
     * @param nodes 所有节点的列表
     * @param parentId 父节点ID
     * @return 子节点列表
     */
    public void getChildNodes(String parentId,String oldPath,String newPath ) {
        QueryWrapper<DocumentEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(DocumentEntity::getParentId, parentId);
        List<DocumentEntity> nodes = documentService.list(queryWrapper);
        for (DocumentEntity node : nodes) {
            DocumentEntity entity = new DocumentEntity();
            entity.setId(node.getId());
            entity.setFilePath(node.getFilePath().replace(oldPath,newPath));
            documentService.update(node.getId(), entity);
            getChildNodes(node.getId(),oldPath,newPath);
        }
    }

    /**
     * 删除
     *
     * @param id 主键值
     * @return
     */
    @ApiOperation("删除知识管理")
    @DeleteMapping("/{id}")
    public ActionResult delete(@PathVariable("id") String id) {
        DocumentEntity entity = documentService.getInfo(id);
        if (entity != null) {
            List<DocumentEntity> allList = documentService.getAllList(entity.getId(), "");
            if (allList.size() > 0) {
                return ActionResult.fail(MsgCode.FA016.get());
            }
            if (entity.getIsShare() != null && 0 != entity.getIsShare()) {
                entity.setIsShare(0);
                documentService.updateById(entity);
            }
            documentService.delete(entity);
            return ActionResult.success(MsgCode.SU003.get());
        }
        return ActionResult.fail(MsgCode.FA003.get());
    }

    /**
     * 列表（文件夹树）
     *
     * @return
     */
    @ApiOperation("获取知识管理列表（文件夹树）")
    @GetMapping("/FolderTree/{id}")
    public ActionResult<ListVO<DocumentFolderTreeVO>> folderTree(@PathVariable("id") String id) {
        List<DocumentEntity> data = documentService.getFolderList();
        if (!"0".equals(id)) {
            data.remove(documentService.getInfo(id));
        }
        List<DocumentFolderTreeModel> treeList = new ArrayList<>();
        DocumentFolderTreeModel model = new DocumentFolderTreeModel();
        model.setId("-1");
        model.setFullName("全部文档");
        model.setParentId("0");
        model.setIcon("0");
        treeList.add(model);
        for (DocumentEntity entity : data) {
            DocumentFolderTreeModel treeModel = new DocumentFolderTreeModel();
            treeModel.setId(entity.getId());
            treeModel.setFullName(entity.getFullName());
            treeModel.setParentId(entity.getParentId());
            treeModel.setIcon("fa fa-folder");
            treeList.add(treeModel);
        }
        List<SumTree<DocumentFolderTreeModel>> trees = TreeDotUtils.convertListToTreeDotFilter(treeList);
        List<DocumentFolderTreeVO> listVO = JsonUtil.getJsonToList(trees, DocumentFolderTreeVO.class);
        ListVO vo = new ListVO();
        vo.setList(listVO);
        return ActionResult.success(vo);
    }

    /**
     * 列表（全部文档）
     *
     * @param page
     * @return
     */
    @ApiOperation("获取知识管理列表（全部文档）")
    @GetMapping
    public ActionResult<ListVO<DocumentListVO>> allList(PageDocument page) {
        List<DocumentEntity> data = documentService.getAllList(page.getParentId(), page.getIsPublic());
        //所有数据
        List<DocumentEntity> allData =documentService.list();
        //设置路径
        for(DocumentEntity dd : data){
            dd.setFilePath(getFilePath(dd,allData,""));
        }
        List<DocumentEntity> returnList = new ArrayList<>();
        if (!StringUtils.isEmpty(page.getKeyword())) {
            //只查询文件
            page.setType(1);
            page.setParentId(null);
            returnList = documentService.getAllList(page);
            returnList = returnList.stream().filter(t -> t.getFullName().contains(page.getKeyword())).collect(Collectors.toList());
        }
        if(ObjectUtil.isNotEmpty(page)){
            String fileLable = page.getFileLable();
            if(StringUtils.isNotEmpty(fileLable)){
                //只查询文件
                page.setType(1);
                page.setParentId(null);

                List<DocumentEntity> listLable = new ArrayList<>();
                if(ObjectUtil.isEmpty(returnList)){
                    listLable = documentService.getAllList(page);
                }else{
                    listLable = returnList;
                }
                String[] aa = fileLable.split(",");
                for(String lable : aa){
                    List<DocumentEntity> ff = listLable.stream().filter(t -> ObjectUtil.isNotEmpty(t.getFileLable()) && t.getFileLable().contains(lable)).collect(Collectors.toList());
                    returnList.addAll(ff);
                }
            }
        }
        if (ObjectUtil.isNotEmpty(page)) {
            // 关键字和标签都是null 则为列表查询
            if (StringUtils.isEmpty(page.getKeyword()) && StringUtils.isEmpty(page.getFileLable())) {
                returnList = data;
            }
        }

        List<DocumentEntity> listReturn = returnList.stream().collect(
                Collectors.toMap(DocumentEntity::getId,person ->person,(p1,p2)-> p1))
                .values().stream().collect(Collectors.toList());
        List<DocumentListVO> list = JsonUtil.getJsonToList(listReturn, DocumentListVO.class);
        for (DocumentListVO documentListVO : list) {
            if (documentListVO.getType() == 1) {
                List<DocumentHistoryEntity> listByOriginalFileId = documentHistoryService.getListByOriginalFileId(documentListVO.getId());
                documentListVO.setDocumentHistory(listByOriginalFileId);
//                //文件则取消最后文件的名称
//                if(StringUtils.isNotEmpty(documentListVO.getFilePath())){
//                    String[] parts = documentListVO.getFilePath().split("\\\\");
//                    StringBuilder combined = new StringBuilder();
//                    //从第三个文件夹开始，取消最后一个
//                    for (int i = 2; i < parts.length - 1; i++) {
//                        combined.append(parts[i]);
//                        if (i < parts.length - 2) { // 避免在最后一个分隔符后添加多余的反斜杠
//                            combined.append("\\"); // 注意这里的双反斜杠，因为在字符串中表示一个反斜杠
//                        }
//                    }
//                    documentListVO.setFilePath(combined.toString());
//                }
            }
//            else{
//                //文件只取消前两级
//                if(StringUtils.isNotEmpty(documentListVO.getFilePath())){
//                    String[] parts = documentListVO.getFilePath().split("\\\\");
//                    StringBuilder combined = new StringBuilder();
//                    //从第三个文件夹开始，取消最后一个
//                    for (int i = 2; i < parts.length - 1; i++) {
//                        combined.append(parts[i]);
//                        if (i < parts.length - 1) { // 避免在最后一个分隔符后添加多余的反斜杠
//                            combined.append("\\"); // 注意这里的双反斜杠，因为在字符串中表示一个反斜杠
//                        }
//                    }
//                    documentListVO.setFilePath(combined.toString());
//                }
//            }
        }
        ListVO vo = new ListVO();
        vo.setList(list);
        return ActionResult.success(vo);
    }

    //拼接文件路径
    private String getFilePath(DocumentEntity doc ,List<DocumentEntity> list,String filePath){
        for (DocumentEntity dd : list){
            if(dd.getId().equals(doc.getParentId())){
                filePath = dd.getFullName()+"\\"+filePath;
                if("0".equals(dd.getParentId())){
                    return filePath;
                }else{
                    filePath = getFilePath(dd,list,filePath);
                }
                break;
            }
        }
        return filePath;
    }


    /**
     * 列表（我的分享）
     *
     * @param page
     * @return
     */
    @ApiOperation("知识管理（我的共享列表）")
    @GetMapping("/Share")
    public ActionResult<ListVO<DocumentSoutListVO>> shareOutList(Page page) {
        List<DocumentEntity> data = documentService.getShareOutList();
        if (!StringUtils.isEmpty(page.getKeyword())) {
            data = data.stream().filter(t -> t.getFullName().contains(page.getKeyword())).collect(Collectors.toList());
        }
        ListVO vo = new ListVO();
        vo.setList(data);
        return ActionResult.success(vo);
    }

    /**
     * 列表（共享给我）
     *
     * @param page
     * @return
     */
    @ApiOperation("获取知识管理列表（共享给我）")
    @GetMapping("/ShareTome")
    public ActionResult<ListVO<DocumentStomeListVO>> shareTomeList(Page page) {
        List<DocumentEntity> list = documentService.getListByShareUserId();
        List<DocumentStomeListVO> vos = JsonUtil.getJsonToList(list, DocumentStomeListVO.class);
        ListVO vo = new ListVO();
        vo.setList(vos);
        return ActionResult.success(vo);
    }

    /**
     * 列表（回收站）
     *
     * @param page
     * @return
     */
    @ApiOperation("获取知识管理列表（回收站）")
    @GetMapping("/Trash")
    public ActionResult<ListVO<DocumentEntity>> trashList(Page page) {
        List<DocumentEntity> data = documentService.getTrashList();
        if (!StringUtils.isEmpty(page.getKeyword())) {
            data = data.stream().filter(t -> t.getFullName().contains(page.getKeyword())).collect(Collectors.toList());
        }
        ListVO<DocumentEntity> vo = new ListVO<>();
        vo.setList(data);
        return ActionResult.success(vo);
    }

    /**
     * 列表（共享人员）
     *
     * @param documentId 文档主键
     * @return
     */
    @ApiOperation("获取知识管理列表（共享人员）")
    @GetMapping("/ShareUser/{documentId}")
    public ActionResult<ListVO<DocumentShareUserListVO>> shareUserList(@PathVariable("documentId") String documentId) {
        List<DocumentShareEntity> data = documentService.getShareUserList(documentId);
        List<DocumentShareUserListVO> list = JsonUtil.getJsonToList(data, DocumentShareUserListVO.class);
        for (DocumentShareUserListVO documentShareUserListVO : list) {
            if ("1".equals(documentShareUserListVO.getShareType())) {
                DocumentShareForm shareEntityById = documentService.getShareEntityById(documentShareUserListVO.getId());
                documentShareUserListVO.setShareUserName(shareEntityById.getShareUserName());
            }
        }
        ListVO vo = new ListVO();
        vo.setList(list);
        return ActionResult.success(vo);
    }


    /**
     * 上传文件
     *
     * @return
     */
    @ApiOperation("知识管理上传文件")
    @PostMapping("/Uploader")
    public ActionResult uploader(DocumentUploader documentUploader) {
        String fileName = documentUploader.getFile().getOriginalFilename();
        List<DocumentEntity> data = documentService.getAllList(documentUploader.getParentId(), "");
        String finalFileName = fileName;
        data = data.stream().filter(t -> finalFileName.equals(t.getFullName())).collect(Collectors.toList());
        if (data.size() > 0) {
            fileName = DateUtil.getNow("+8") + "-" + fileName;
        }
        String fileType = UpUtil.getFileType(documentUploader.getFile());
        String name = RandomUtil.uuId();
        String filePath = configValueUtil.getDocumentFilePath();
        //验证类型
        if (!OptimizeUtil.fileType(configValueUtil.getAllowUploadFileType(), fileType)) {
            return ActionResult.fail(MsgCode.FA017.get());
        }
        //上传
        UploadUtil.uploadFile(configValueUtil.getFileType(), FileTypeEnum.DOCUMENT, name + "." + fileType, documentUploader.getFile(), filePath);
        DocumentEntity entity = new DocumentEntity();
        entity.setType(1);
        entity.setFullName(fileName);
        entity.setParentId(documentUploader.getParentId());
        entity.setFileExtension(fileType);

        entity.setFilePath(name + "." + fileType);
        entity.setFileSize(String.valueOf(documentUploader.getFile().getSize()));
        entity.setDeleteMark(0);
        documentService.create(entity);
        return ActionResult.success(MsgCode.SU015.get());
    }

    @NoDataSourceBind()
    @ApiOperation("分片组装")
    @PostMapping("/merge")
    public ActionResult merge(MergeChunkDto mergeChunkDto) {
        String identifier = mergeChunkDto.getIdentifier();
        String path = FilePathUtil.getFilePath(FileTypeEnum.TEMPORARY);
        String filePath = XSSEscape.escapePath(path + identifier);
        String partFile = XSSEscape.escapePath(path + mergeChunkDto.getFileName());
        try {
            List<File> mergeFileList = FileUtil.getFile(new File(filePath));
            @Cleanup FileOutputStream destTempfos = new FileOutputStream(partFile, true);
            for (int i = 0; i < mergeFileList.size(); i++) {
                String chunkName = identifier.concat("-") + (i + 1);
                File files = new File(filePath, chunkName);
                if (files.exists()) {
                    FileUtils.copyFile(files, destTempfos);
                }
            }
            File partFiles = new File(partFile);
            if (partFiles.exists()) {
                MultipartFile multipartFile = FileUtil.createFileItem(partFiles);
                uploaderVO(multipartFile, mergeChunkDto.getParentId(), mergeChunkDto.getIsPublic());
                FileUtil.deleteTmp(multipartFile);
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            FileUtils.deleteQuietly(new File(filePath));
            FileUtils.deleteQuietly(new File(partFile));
        }
        return ActionResult.success(MsgCode.SU015.get());
    }

    /**
     * 合并文件
     *
     * @param guid
     * @return
     */
    /*@ApiOperation("（未找到）知识管理合并文件")
    @PostMapping("/Merge/{guid}")
    public ActionResult Merge(@PathVariable("guid") String guid, String fileName, String folderId) {
        //临时文件
        String temp = configValueUtil.getTemporaryFilePath() + guid;
        File file = new File(temp);
        //保存文件
        UserInfo userInfo = userProvider.get();
        String userId = userInfo.getUserId();
        String tenantId = userInfo.getTenantId();
        String time = DateTimeFormatter.ofPattern("yyyyMMdd").format(LocalDateTime.now());
        String path = configValueUtil.getDocumentFilePath() + "\\" + tenantId + "\\" + userId + "\\" + time;
        String fileType = "";
        String name = RandomUtil.uuId();
        File partFile = null;
        try {
            if (file.isDirectory()) {
                File[] files = file.listFiles();
                if (files != null && files.length > 0) {
                    fileType = fileName.substring(fileName.lastIndexOf(".") + 1, fileName.length());
                    partFile = new File(path);
                    if (!partFile.exists()) {
                        partFile.mkdirs();
                    }
                    partFile = new File(path + "\\" + name + "." + fileType);
                    for (int i = 0; i < files.length; i++) {
                        File s = new File(temp, i + ".part");
                        FileOutputStream destTempfos = new FileOutputStream(partFile, true);
                        FileUtils.copyFile(s, destTempfos);
                        destTempfos.close();
                    }
                    FileUtils.deleteDirectory(file);
                }
            }
            DocumentEntity entity = new DocumentEntity();
            entity.setFType(1);
            entity.setFParentId(folderId);
            entity.setFFullName(fileName);
            entity.setFFileExtension(fileType);
            entity.setFFilePath(tenantId + "\\" + userId + "\\" + time + "\\" + name + "." + fileType);
            entity.setFFileSize(String.valueOf(partFile.length()));
            entity.setFDeleteMark(0);
            documentService.create(entity);
            return ActionResult.success("合并成功");
        } catch (Exception e) {
            try {
                FileUtils.deleteDirectory(file);
            } catch (Exception e1) {
                System.out.println(e1.getMessage());
            }
            System.out.println(e.getMessage());
            return ActionResult.fail("上传失败");
        }
    }*/

    /**
     * 下载文件
     *
     * @param id 主键值
     * @return
     */
    @ApiOperation("获取下载文件链接")
    @PostMapping("/Download/{id}")
    public ActionResult<DownloadVO> download(@PathVariable("id") String id) {
        UserInfo userInfo = userProvider.get();
        DocumentEntity entity = documentService.getInfo(id);
        if (entity != null) {
            String name = entity.getFilePath();
//            String fileName = userInfo.getId() + "#" + name + "#" + "document#" + entity.getFullName() + "." + entity.getFileExtension();
            String fileName = userInfo.getId() + "#" + name + "#" + "document";
            DownloadVO vo = DownloadVO.builder().name(entity.getFullName()).url(UploaderUtil.uploaderFile(fileName)).build();
            return ActionResult.success(vo);
        } else {
            DocumentHistoryEntity info = documentHistoryService.getInfo(id);
            if (info != null) {
                String name = info.getFilePath();
                String fileName = userInfo.getId() + "#" + name + "#" + "document#" + info.getFullName() + "." + info.getFileExtension();
                DownloadVO vo = DownloadVO.builder().name(info.getFullName()).url(UploaderUtil.uploaderFile(fileName)).build();
                return ActionResult.success(vo);
            }
        }
        return ActionResult.fail(MsgCode.FA018.get());
    }

    /**
     * 回收站（彻底删除）
     *
     * @param id 主键值
     * @return
     */
    @ApiOperation("回收站（彻底删除）")
    @DeleteMapping("/Trash/{id}")
    public ActionResult trashDelete(@PathVariable("id") String id) {
        documentService.trashDelete(id);
        return ActionResult.success(MsgCode.SU003.get());
    }

    /**
     * 回收站（还原文件）
     *
     * @param id 主键值
     * @return
     */
    @ApiOperation("回收站（还原文件）")
    @PostMapping("/Trash/{id}/Actions/Recovery")
    public ActionResult trashRecovery(@PathVariable("id") String id) {
        boolean flag = documentService.trashRecovery(id);
        if (flag == false) {
            return ActionResult.fail(MsgCode.FA010.get());
        }
        return ActionResult.success(MsgCode.SU010.get());
    }

    /**
     * 共享文件（创建）
     *
     * @return
     */
    @ApiOperation("分享文件/文件夹")
    @PostMapping("/{id}/Actions/Share")
    public ActionResult shareCreate(@PathVariable("id") String id, @RequestBody DocumentShareForm documentShareForm) {
        if ("2".equals(documentShareForm.getPassword())) {
            documentShareForm.setPassword(CommonUtil.RandomGenerator());
        }
        boolean flag = documentService.shareCreate(id, documentShareForm);
        if (flag == false) {
            return ActionResult.fail("操作失败，原文件不存在");
        }
        return ActionResult.success(MsgCode.SU005.get());
    }

    /**
     * 共享文件（修改）
     *
     * @return
     */
    @ApiOperation("分享文件/文件夹")
    @PostMapping("/{id}/Actions/updateShare")
    public ActionResult shareUpdate(@PathVariable("id") String id, @RequestBody DocumentShareForm documentShareForm) {
        boolean flag = documentService.shareUpdate(id, documentShareForm);
        if (flag == false) {
            return ActionResult.fail("操作失败，原文件不存在");
        }
        return ActionResult.success(MsgCode.SU005.get());
    }

    /**
     * 根据id获取文件分享实体
     *
     * @return
     */
    @ApiOperation("根据id获取文件分享实体")
    @GetMapping("/getShareEntityById/{id}")
    @NotCheckLogin
    public ActionResult getShareEntityById(@PathVariable("id") String id) {
        DocumentShareForm shareEntityById = documentService.getShareEntityById(id);
        return ActionResult.success(shareEntityById);
    }

    /**
     * 共享文件（取消）
     *
     * @param id 主键值
     * @return
     */
    @ApiOperation("取消分享文件/文件夹")
    @DeleteMapping("/{id}/Actions/Share")
    public ActionResult shareCancel(@PathVariable("id") String id) {
        boolean flag = documentService.shareCancel(id);
        if (flag == false) {
            return ActionResult.fail("操作失败，原文件不存在");
        }
        return ActionResult.success(MsgCode.SU005.get());
    }

    /**
     * 删除共享文件链接（）
     *
     * @param id 主键值
     * @return
     */
    @ApiOperation("取消分享文件/文件夹")
    @DeleteMapping("/{id}/Actions/handleDelShareLink")
    public ActionResult handleDelShareLink(@PathVariable("id") String id) {
        boolean flag = documentService.handleDelShareLink(id);
        if (flag == false) {
            return ActionResult.fail("操作失败，原文件不存在");
        }
        return ActionResult.success(MsgCode.SU003.get());
    }

    /**
     * 文件/夹移动到
     *
     * @param id   主键值
     * @param toId 将要移动到Id
     * @return
     */
    @ApiOperation("移动文件/文件夹")
    @PutMapping("/{id}/Actions/MoveTo/{toId}")
    public ActionResult moveTo(@PathVariable("id") String id, @PathVariable("toId") String toId) {
        if (id.equals(toId)) {
            return ActionResult.fail("不能移动到自己的文件夹");
        }
        boolean flag = documentService.moveTo(id, toId);
        if (flag == false) {
            return ActionResult.fail(MsgCode.FA002.get());
        }
        return ActionResult.success(MsgCode.SU004.get());
    }

    /**
     * 封装上传附件
     *
     * @param file
     * @param parentId
     * @return
     * @throws IOException
     */
    private void uploaderVO(MultipartFile file, String parentId, String isPublic) {
        UserInfo userInfo = userProvider.get();
        String fileType = UpUtil.getFileType(file);
        String filePath = configValueUtil.getDocumentFilePath();
        filePath += userInfo.getUserId() + File.separator+ DateUtil.getNowYear()+File.separator;
        String name = RandomUtil.uuId();
        String fPath=name;
        if (!"0".equals(parentId)) {
            String absoluteRouteById = documentService.getAbsoluteRouteById(parentId);
            filePath =absoluteRouteById;
            fPath=absoluteRouteById.substring(configValueUtil.getDocumentFilePath().length())+name;
        }else{
            fPath=userInfo.getUserId() + File.separator+ DateUtil.getNowYear()+File.separator+name;
        }

        String fileName = file.getOriginalFilename();
        List<DocumentEntity> data = documentService.getAllList(parentId, isPublic);
        String finalFileName = fileName;
        data = data.stream().filter(t -> finalFileName.equals(t.getFullName()) && isPublic.equals(t.getIsPublic())).collect(Collectors.toList());
        if (data.size() > 0) {
            //说明已有该文件，将源文件删除，并向历史表添加数据
            documentService.removeBatchByIds(data);

        }
        //上传
        UploadUtil.uploadFile(configValueUtil.getFileType(), FileTypeEnum.DOCUMENT, name + "." + fileType, file, filePath);
        DocumentEntity entity = new DocumentEntity();
        if (data.size() > 0) {
            entity.setId(data.get(0).getId());
            List<DocumentHistoryEntity> listByOriginalFileId = documentHistoryService.getListByOriginalFileId(entity.getId());
            entity.setFileVersion("V" + (listByOriginalFileId.size() + 2));
        } else {
            entity.setId(RandomUtil.uuId());
            entity.setFileVersion("V1");
        }
        entity.setType(1);
        entity.setFullName(fileName);
        entity.setParentId(parentId);
        entity.setFileExtension(fileType);
        entity.setFilePath(fPath + "." + fileType);
        entity.setFileSize(String.valueOf(file.getSize()));
        entity.setDeleteMark(0);
        entity.setCompanyId(userInfo.getOrganizeId());
        entity.setIsPublic(isPublic);
        documentService.create(entity);

        //向历史版本表添加相同数据
        if (data.size() > 0) {
            DocumentHistoryEntity documentHistoryEntity = JsonUtil.getJsonToBean(data.get(0), DocumentHistoryEntity.class);
            documentHistoryEntity.setId(RandomUtil.uuId());
            documentHistoryEntity.setOriginalFileId(data.get(0).getId());
            documentHistoryService.create(documentHistoryEntity);
        }
    }

    /**
     * 压缩文件夹并下载
     * @param id  文件夹id
     * @return
     */
    @ApiOperation("压缩文件夹并下载")
    @GetMapping("/toZip/{id}")
    public ActionResult toZip(@PathVariable("id") String id) {
        DownloadVO allDocumentByParentId = documentService.getAllDocumentByParentId(id);
        return ActionResult.success(allDocumentByParentId);
    }

}
