package com.bringspring.extend.controller;

import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.vo.ListVO;
import com.bringspring.common.base.vo.PageListVO;
import com.bringspring.common.base.vo.PaginationVO;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.exception.DataException;
import com.bringspring.common.util.DateUtil;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.extend.utils.ScheduleTask;
import com.bringspring.system.base.util.JsonUtilEx;
import com.bringspring.extend.entity.ScheduleEntity;
import com.bringspring.extend.model.schedule.*;
import com.bringspring.extend.service.ScheduleService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.text.SimpleDateFormat;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 日程安排
 *
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2017年9月26日 上午9:18
 */
@Api(tags = "日程安排", value = "Schedule")
@RestController
@RequestMapping("/api/extend/Schedule")
public class ScheduleController {

    @Autowired
    private ScheduleService scheduleService;


    /**
     * 列表
     *
     * @param scheduleTime 时间对象
     * @return
     */
    @ApiOperation("获取日程安排列表")
    @GetMapping
    public ActionResult<ListVO<ScheduleListVO>> list(ScheduleTime scheduleTime) {
        List<ScheduleEntity> data = scheduleService.getList(scheduleTime.getStartTime(), scheduleTime.getEndTime());
        List<ScheduleListVO> list = JsonUtil.getJsonToList(data, ScheduleListVO.class);
        ListVO<ScheduleListVO> listVO = new ListVO();
        listVO.setList(list);
        return ActionResult.success(listVO);
    }



    /**
     * 智行首页展示工作签列表
     *
     * @param schedulePagination
     * @return
     */
    @ApiOperation("首页获取日程安排列表")
    @PostMapping("/getList")
    public ActionResult listPc(@RequestBody SchedulePagination schedulePagination) throws Exception {
        List<ScheduleEntity> data = scheduleService.getListPagination(schedulePagination);
        List<SchedulePcListVO> listVO = JsonUtil.getJsonToList(data, SchedulePcListVO.class);
        if (StringUtils.isNotEmpty(schedulePagination.getCurrentDate()) && !listVO.isEmpty() ){
            for (SchedulePcListVO schedulePcListVO:listVO){
//           获取时分
                String format = new SimpleDateFormat("yyyy-MM-dd HH:mm").format(schedulePcListVO.getStartTime());
                String afterComma = format.substring(11);
                schedulePcListVO.setTime(afterComma);
            }
        }
        PageListVO vo=new PageListVO();
        vo.setList(listVO);
        PaginationVO page=JsonUtil.getJsonToBean(schedulePagination,PaginationVO.class);
        vo.setPagination(page);
        return ActionResult.success(vo);
    }

    /**
     * 获取未完成的工作签日期 （用来展示标记）
     * 条件---日期
     */
    @PostMapping("/getDateList")
    public ActionResult<List<Date>> dateList(@RequestBody SchedulePagination schedulePagination) {
        List<ScheduleEntity> data = scheduleService.getDateList(schedulePagination);
        List<Date> collect = data.stream().map(ScheduleEntity::getStartTime).collect(Collectors.toList());
        return ActionResult.success(collect);
    }

    /**
     * 列表（app使用）
     *
     * @param scheduleTime 时间对象
     * @return
     */
    @ApiOperation("（待定）列表")
    @GetMapping("/AppList")
    public ActionResult appList(ScheduleTimes scheduleTime) {
        Map<String, Object> days = new LinkedHashMap<>();
        List<ScheduleEntity> scheduleList = scheduleService.getList(scheduleTime.getStartTime(), scheduleTime.getEndTime());
        Date start = DateUtil.stringToDates(scheduleTime.getStartTime());
        Date end = DateUtil.stringToDates(scheduleTime.getEndTime());
        List<Date> item = DateUtil.getAllDays(start, end);
        if (StringUtils.isEmpty(scheduleTime.getDateTime())) {
            scheduleTime.setDateTime(DateUtil.dateNow("yyyyMMdd"));
        } else {
            scheduleTime.setDateTime(scheduleTime.getDateTime().replaceAll("-", ""));
        }
        Map<String, List<ScheduleEntity>> dataList = new HashMap<>(16);
        for (int i = 0; i < item.size(); i++) {
            String startTime = DateUtil.daFormat(item.get(i)) + " 00:00";
            String endTime = DateUtil.daFormat(item.get(i)) + " 23:59";
            List<ScheduleEntity> count = scheduleList.stream().filter(m -> DateUtil.dateFormat(m.getStartTime()).compareTo(endTime) <= 0 && DateUtil.dateFormat(m.getEndTime()).compareTo(startTime) >= 0).collect(Collectors.toList());
            String time = DateUtil.daFormat(item.get(i)).replaceAll("-", "");
            days.put(time, count.size());
            dataList.put(time, count);
        }
        Map<String, Object> data = new HashMap<>(16);
        data.put("signList", days);
        List<ScheduleEntity> todayList = dataList.get(scheduleTime.getDateTime());
        data.put("todayList", JsonUtil.listToJsonField(todayList));
        return ActionResult.success(data);
    }

    /**
     * 列表(app使用）
     *
     * @param scheduleTimes 时间对象
     * @return
     */
    @ApiOperation("（待定）列表")
    @GetMapping("/AppDayList")
    public ActionResult list(ScheduleTimes scheduleTimes) {
        List<ScheduleEntity> data = scheduleService.getList(scheduleTimes.getStartTime(), scheduleTimes.getEndTime());
        List<ScheduleEntity> datas = new ArrayList<>();
        Date dateTimes = DateUtil.stringToDates(scheduleTimes.getDateTime());
        if (!StringUtils.isEmpty(scheduleTimes.getDateTime())) {
            for (ScheduleEntity entity : data) {
                Date startTimes = entity.getStartTime();
                Date endTimes = entity.getEndTime();
                if (DateUtil.isEffectiveDate(dateTimes, startTimes, endTimes)) {
                    datas.add(entity);
                }
            }
        }
        return ActionResult.success(JsonUtil.listToJsonField(datas));
    }


    /**
     * 信息
     *
     * @param id 主键值
     * @return
     */
    @ApiOperation("获取日程安排信息")
    @GetMapping("/{id}")
    public ActionResult<ScheduleInfoVO> info(@PathVariable("id") String id) throws DataException {
        ScheduleEntity entity = scheduleService.getInfo(id);
        ScheduleInfoVO vo = JsonUtilEx.getJsonToBeanEx(entity, ScheduleInfoVO.class);
        return ActionResult.success(vo);
    }

    /**
     * 新建
     *
     * @param scheduleCrForm 实体对象
     * @return
     */
    @ApiOperation("新建日程安排")
    @PostMapping
    public ActionResult create(@RequestBody @Valid ScheduleCrForm scheduleCrForm) {
        ScheduleEntity entity = JsonUtil.getJsonToBean(scheduleCrForm, ScheduleEntity.class);
        Boolean aBoolean = scheduleService.judgeScheduleTimeConflict(entity);
        if (aBoolean) {
            return ActionResult.success("创建失败,日程时间重复");
        } else {
            //创建--状态默认为未完成
            entity.setState("0");
            entity.setSendNumber(0);//提醒数量
            scheduleService.create(entity);
            return ActionResult.success(MsgCode.SU001.get());
        }
    }

    /**
     * 新建
     *
     * @param scheduleCrForm 实体对象
     * @return
     */
    @ApiOperation("新建日程安排")
    @PostMapping("/createRemind")
    public ActionResult createRemind(@RequestBody @Valid ScheduleCrForm scheduleCrForm) {
        ScheduleEntity entity = JsonUtil.getJsonToBean(scheduleCrForm, ScheduleEntity.class);
        //创建--状态默认为未完成
        entity.setState("0");
        entity.setSendNumber(0);//提醒数量
        scheduleService.create(entity);
        return ActionResult.success(MsgCode.SU001.get());
    }

    /**
     * 更新
     *
     * @param id             主键值
     * @param scheduleUpForm 实体对象
     * @return
     */
    @ApiOperation("更新日程安排")
    @PutMapping("/{id}")
    public ActionResult update(@PathVariable("id") String id, @RequestBody @Valid ScheduleUpForm scheduleUpForm) {
        ScheduleEntity entity = JsonUtil.getJsonToBean(scheduleUpForm, ScheduleEntity.class);
        boolean flag = scheduleService.update(id, entity);
        if (flag == false) {
            return ActionResult.fail(MsgCode.FA002.get());
        }
        return ActionResult.success(MsgCode.SU004.get());
    }

    /**
     * 删除
     *
     * @param id 主键值
     * @return
     */
    @ApiOperation("删除日程安排")
    @DeleteMapping("/{id}")
    public ActionResult delete(@PathVariable("id") String id) {
        ScheduleEntity entity = scheduleService.getInfo(id);
        if (entity != null) {
            scheduleService.delete(entity);
            return ActionResult.success(MsgCode.SU003.get());
        }
        return ActionResult.fail(MsgCode.FA003.get());
    }

    @GetMapping("/ceshi")
    public ActionResult ceshi() throws Exception {
        scheduleService.scheduledRemindeEvery();
        return ActionResult.success(MsgCode.SU001.get());
    }
}
