package com.bringspring.extend.controller;

import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.MailAccount;
import com.bringspring.common.base.systemconfig.SysConfigModel;
import com.bringspring.common.base.vo.PaginationVO;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.exception.DataException;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.extend.entity.EmailSendEntity;
import com.bringspring.extend.model.email.*;
import com.bringspring.extend.service.EmailReceiveService;
import com.bringspring.system.base.entity.EmailConfigEntity;
import com.bringspring.system.base.entity.EmailReceiveEntity;
import com.bringspring.system.base.entity.SysConfigEntity;
import com.bringspring.system.base.exception.BaseException;
import com.bringspring.system.base.service.SysConfigService;
import com.bringspring.system.base.util.JsonUtilEx;
import com.bringspring.system.base.util.Pop3Util;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.HashMap;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 邮件配置
 *
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2017年9月26日 上午9:18
 */
@Tag(name = "邮件收发", description = "邮件收发功能管理")
@RestController
@RequestMapping("/api/extend/Email")
public class EmailController {

    @Autowired
    private EmailReceiveService emailReceiveService;
    @Autowired
    private Pop3Util pop3Util;
    @Autowired
    private SysConfigService sysConfigService;

    /**
     * 获取邮件列表(收件箱、标星件、草稿箱、已发送)
     *
     * @param paginationEmail
     * @return
     */
    @Operation(summary = "获取邮件列表(收件箱、标星件、草稿箱、已发送)")
    @GetMapping
    public ActionResult receiveList(PaginationEmail paginationEmail) {
        String type = paginationEmail.getType() != null ? paginationEmail.getType() : "inBox";
        switch (type) {
            case "inBox":
                List<EmailReceiveEntity> entity = emailReceiveService.getReceiveList(paginationEmail);
                PaginationVO paginationVO = JsonUtil.getJsonToBean(paginationEmail, PaginationVO.class);
                List<EmailReceiveListVO> listVO = JsonUtil.getJsonToList(entity, EmailReceiveListVO.class);
                return ActionResult.page(listVO, paginationVO);
            case "star":
                List<EmailReceiveEntity> entity1 = emailReceiveService.getStarredList(paginationEmail);
                PaginationVO paginationVo1 = JsonUtil.getJsonToBean(paginationEmail, PaginationVO.class);
                List<EmailStarredListVO> listVo1 = JsonUtil.getJsonToList(entity1, EmailStarredListVO.class);
                return ActionResult.page(listVo1, paginationVo1);
            case "draft":
                List<EmailSendEntity> entity2 = emailReceiveService.getDraftList(paginationEmail);
                PaginationVO paginationVo2 = JsonUtil.getJsonToBean(paginationEmail, PaginationVO.class);
                List<EmailDraftListVO> listVo2 = JsonUtil.getJsonToList(entity2, EmailDraftListVO.class);
                return ActionResult.page(listVo2, paginationVo2);
            case "sent":
                List<EmailSendEntity> entity3 = emailReceiveService.getSentList(paginationEmail);
                PaginationVO paginationVo3 = JsonUtil.getJsonToBean(paginationEmail, PaginationVO.class);
                List<EmailSentListVO> listVo3 = JsonUtil.getJsonToList(entity3, EmailSentListVO.class);
                return ActionResult.page(listVo3, paginationVo3);
            default:
                return ActionResult.fail("获取失败");
        }
    }

    /**
     * 获取邮箱配置
     *
     * @return
     */
    @Operation(summary = "获取邮箱配置")
    @GetMapping("/Config")
    public ActionResult<EmailCofigInfoVO> configInfo() throws BaseException {
        EmailConfigEntity entity = emailReceiveService.getConfigInfo();
        EmailCofigInfoVO vo = JsonUtil.getJsonToBean(entity, EmailCofigInfoVO.class);
        if (vo == null) {
            vo = new EmailCofigInfoVO();
            List<SysConfigEntity> list = sysConfigService.getList();
            if (!list.isEmpty()) {
                List<SysConfigEntity> emailConfig = list.stream().filter(item -> item.getCategory().equals("EmailConfig")).collect(Collectors.toList());
                if (!emailConfig.isEmpty()) {
                    HashMap<String, String> map = new HashMap<>(16);
                    for (SysConfigEntity sys : list) {
                        map.put(sys.getKeyName(), sys.getKeyValue());
                    }
                    SysConfigModel sysConfigModel = JsonUtil.getJsonToBean(map, SysConfigModel.class);

                    vo.setPop3Host(sysConfigModel.getEmailPop3Host());
                    vo.setPop3Port(Integer.valueOf(sysConfigModel.getEmailPop3Port()));
                    vo.setSmtpHost(sysConfigModel.getEmailSmtpHost());
                    vo.setSmtpPort(Integer.valueOf(sysConfigModel.getEmailSmtpPort()));
                    vo.setSenderName(sysConfigModel.getEmailSenderName());
                    vo.setEmailSsl(sysConfigModel.getEmailSsl());
                }
            }

        }
        return ActionResult.success(vo);
    }

    /**
     * 获取邮件信息
     *
     * @param id 主键值
     * @return
     */
    @Operation(summary = "获取邮件信息")
    @GetMapping("/{id}")
    public ActionResult<EmailInfoVO> info(@PathVariable("id") String id) throws DataException {
        Object entity = emailReceiveService.getInfo(id);
        EmailInfoVO vo = JsonUtilEx.getJsonToBeanEx(entity, EmailInfoVO.class);
        return ActionResult.success(vo);
    }

    /**
     * 删除
     *
     * @param id 主键值
     * @return
     */
    @Operation(summary = "删除邮件")
    @DeleteMapping("/{id}")
    public ActionResult delete(@PathVariable("id") String id) {
        boolean flag = emailReceiveService.delete(id);
        if (flag == false) {
            return ActionResult.fail("删除失败，邮件不存在");
        }
        return ActionResult.success(MsgCode.SU003.get());
    }

    /**
     * 设置已读邮件
     *
     * @param id 主键值
     * @return
     */
    @Operation(summary = "设置已读邮件")
    @PutMapping("/{id}/Actions/Read")
    public ActionResult receiveRead(@PathVariable("id") String id) {
        boolean flag = emailReceiveService.receiveRead(id, 1);
        if (flag == false) {
            return ActionResult.fail("操作失败，邮件不存在");
        }
        return ActionResult.success(MsgCode.SU005.get());
    }

    /**
     * 设置未读邮件
     *
     * @param id 主键值
     * @return
     */
    @Operation(summary = "设置未读邮件")
    @PutMapping("/{id}/Actions/Unread")
    public ActionResult receiveUnread(@PathVariable("id") String id) {
        boolean flag = emailReceiveService.receiveRead(id, 0);
        if (flag == false) {
            return ActionResult.fail("操作失败，邮件不存在");
        }
        return ActionResult.success(MsgCode.SU005.get());
    }

    /**
     * 设置星标邮件
     *
     * @param id 主键值
     * @return
     */
    @Operation(summary = "设置星标邮件")
    @PutMapping("/{id}/Actions/Star")
    public ActionResult receiveYesStarred(@PathVariable("id") String id) {
        boolean flag = emailReceiveService.receiveStarred(id, 1);
        if (flag == false) {
            return ActionResult.fail("操作失败，邮件不存在");
        }
        return ActionResult.success(MsgCode.SU005.get());
    }

    /**
     * 设置取消星标
     *
     * @param id 主键值
     * @return
     */
    @Operation(summary = "设置取消星标")
    @PutMapping("/{id}/Actions/Unstar")
    public ActionResult receiveNoStarred(@PathVariable("id") String id) {
        boolean flag = emailReceiveService.receiveStarred(id, 0);
        if (flag == false) {
            return ActionResult.fail("操作失败，邮件不存在");
        }
        return ActionResult.success(MsgCode.SU005.get());
    }

    /**
     * 收邮件
     *
     * @return
     */
    @Operation(summary = "收邮件")
    @PostMapping("/Receive")
    public ActionResult receive() {
        EmailConfigEntity configEntity = emailReceiveService.getConfigInfo();
        if (configEntity != null) {
            MailAccount mailAccount = new MailAccount();
            mailAccount.setAccount(configEntity.getAccount());
            mailAccount.setPassword(configEntity.getPassword());
            mailAccount.setPop3Host(configEntity.getPop3Host());
            mailAccount.setPop3Port(configEntity.getPop3Port());
            mailAccount.setSmtpHost(configEntity.getSmtpHost());
            mailAccount.setSmtpPort(configEntity.getSmtpPort());
            if ("1".equals(String.valueOf(configEntity.getEmailSsl()))) {
                mailAccount.setSsl(true);
            } else {
                mailAccount.setSsl(false);
            }
            String checkResult = pop3Util.checkConnected(mailAccount);
            if ("true".equals(checkResult)) {
                int mailCount = emailReceiveService.receive(configEntity);
                return ActionResult.success(MsgCode.SU005.get(), mailCount);
            } else {
                return ActionResult.fail(checkResult);
            }
        } else {
            return ActionResult.fail("你还没有设置邮件的帐户");
        }
    }

    /**
     * 存草稿
     *
     * @return
     */
    @Operation(summary = "存草稿")
    @PostMapping("/Actions/SaveDraft")
    public ActionResult saveDraft(@RequestBody @Valid EmailSendCrForm emailSendCrForm) {
        EmailSendEntity entity = JsonUtil.getJsonToBean(emailSendCrForm, EmailSendEntity.class);
        emailReceiveService.saveDraft(entity);
        return ActionResult.success(MsgCode.SU002.get());
    }

    /**
     * 发邮件
     *
     * @return
     */
    @Operation(summary = "发邮件")
    @PostMapping
    public ActionResult saveSent(@RequestBody @Valid EmailCrForm emailCrForm) {
        EmailSendEntity entity = JsonUtil.getJsonToBean(emailCrForm, EmailSendEntity.class);
        EmailConfigEntity configEntity = emailReceiveService.getConfigInfo();
        if (configEntity != null) {
            MailAccount mailAccount = new MailAccount();
            mailAccount.setAccount(configEntity.getAccount());
            mailAccount.setPassword(configEntity.getPassword());
            mailAccount.setPop3Host(configEntity.getPop3Host());
            mailAccount.setPop3Port(configEntity.getPop3Port());
            mailAccount.setSmtpHost(configEntity.getSmtpHost());
            mailAccount.setSmtpPort(configEntity.getSmtpPort());
            if ("1".equals(String.valueOf(configEntity.getEmailSsl()))) {
                mailAccount.setSsl(true);
            } else {
                mailAccount.setSsl(false);
            }
            int flag = emailReceiveService.saveSent(entity, configEntity);
            if (flag == 0) {
                return ActionResult.success(MsgCode.SU012.get());
            } else {
                return ActionResult.fail("账户认证错误");
            }
        } else {
            return ActionResult.fail("你还没有设置邮件的帐户");
        }
    }

    /**
     * 更新邮件配置
     *
     * @return
     */
    @Operation(summary = "更新邮件配置")
    @PutMapping("/Config")
    public ActionResult saveConfig(@RequestBody @Valid EmailCheckForm emailCheckForm) throws DataException {
        EmailConfigEntity entity = JsonUtil.getJsonToBean(emailCheckForm, EmailConfigEntity.class);
        emailReceiveService.saveConfig(entity);
        return ActionResult.success(MsgCode.SU002.get());
    }

    /**
     * 邮箱配置-测试连接
     *
     * @return
     */
    @Operation(summary = "邮箱配置-测试连接")
    @PostMapping("/Config/Actions/CheckMail")
    public ActionResult checkLogin(@RequestBody @Valid EmailCheckForm emailCheckForm) {
        EmailConfigEntity entity = JsonUtil.getJsonToBean(emailCheckForm, EmailConfigEntity.class);
        String result = sysConfigService.checkMainLogin(entity);
        if ("true".equals(result)) {
            return ActionResult.success(MsgCode.SU017.get());
        } else {
            return ActionResult.fail(result);
        }
    }

}
