package com.bringspring.extend.controller;

import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.UserInfo;
import com.bringspring.common.base.vo.ListVO;
import com.bringspring.common.base.vo.PageListVO;
import com.bringspring.common.base.vo.PaginationVO;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.exception.DataException;
import com.bringspring.common.util.DateUtil;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.common.util.StringUtils;
import com.bringspring.common.auth.util.UserProvider;
import com.bringspring.common.util.treeutil.SumTree;
import com.bringspring.common.util.treeutil.newtreeutil.TreeDotUtils;
import com.bringspring.extend.entity.TableExampleEntity;
import com.bringspring.extend.model.tableexample.*;
import com.bringspring.extend.model.tableexample.postil.PostilInfoVO;
import com.bringspring.extend.model.tableexample.postil.PostilModel;
import com.bringspring.extend.model.tableexample.postil.PostilSendForm;
import com.bringspring.extend.service.TableExampleService;
import com.bringspring.system.base.entity.DictionaryDataEntity;
import com.bringspring.system.base.entity.ProvinceEntity;
import com.bringspring.system.base.service.DictionaryDataService;
import com.bringspring.system.base.service.ProvinceService;
import com.bringspring.system.base.util.JsonUtilEx;
import com.bringspring.system.permission.entity.UserEntity;
import com.bringspring.system.permission.service.UserService;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.Operation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 表格示例数据
 *
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2017年9月26日 上午9:18
 */
@Tag(name = "表格示例数据", description = "表格数据示例展示")
@RestController
@RequestMapping("/api/extend/TableExample")
public class TableExampleController {

    @Autowired
    private TableExampleService tableExampleService;
    @Autowired
    private ProvinceService provinceService;
    @Autowired
    private DictionaryDataService dictionaryDataService;
    @Autowired
    private UserProvider userProvider;
    @Autowired
    private UserService userService;

    /**
     * 列表
     *
     * @param paginationTableExample
     * @return
     */
    @Operation(summary="获取表格数据列表")
    @GetMapping
    public ActionResult<PageListVO<TableExampleListVO>> list(PaginationTableExample paginationTableExample) {
        List<TableExampleEntity> data = tableExampleService.getList(paginationTableExample);
        List<TableExampleListVO> list = JsonUtil.getJsonToList(data, TableExampleListVO.class);
        List<String> userId = list.stream().map(t -> t.getRegistrant()).collect(Collectors.toList());
        List<UserEntity> userList = userService.getUserName(userId);
        for (TableExampleListVO tableExampleListVO : list) {
            UserEntity user = userList.stream().filter(t -> t.getId().equals(tableExampleListVO.getRegistrant())).findFirst().orElse(null);
            tableExampleListVO.setRegistrant(user != null ? user.getRealName(): "");
        }
        PaginationVO paginationVO = JsonUtil.getJsonToBean(paginationTableExample, PaginationVO.class);
        return ActionResult.page(list, paginationVO);
    }

    /**
     * 列表（树形表格）
     *
     * @param typeId 主键值
     * @return
     */
    @Operation(summary="（树形表格）")
    @GetMapping("/ControlSample/{typeId}")
    public ActionResult list(@PathVariable("typeId") String typeId, PaginationTableExample paginationTableExample) {
        List<TableExampleEntity> data = tableExampleService.getList(typeId, paginationTableExample);
        List<TableExampleListVO> list = JsonUtil.getJsonToList(data, TableExampleListVO.class);
        List<String> userId = list.stream().map(t -> t.getRegistrant()).collect(Collectors.toList());
        List<UserEntity> userList = userService.getUserName(userId);
        for (TableExampleListVO tableExampleListVO : list) {
            UserEntity user = userList.stream().filter(t -> t.getId().equals(tableExampleListVO.getRegistrant())).findFirst().orElse(null);
            tableExampleListVO.setRegistrant(user != null ? user.getRealName(): "");
        }
        PaginationVO paginationVO = JsonUtil.getJsonToBean(paginationTableExample, PaginationVO.class);
        return ActionResult.page(list, paginationVO);
    }

    /**
     * 列表
     *
     * @return
     */
    @Operation(summary="获取表格分组列表")
    @GetMapping("/All")
    public ActionResult listAll() {
        List<TableExampleEntity> data = tableExampleService.getList();
        List<TableExampleListAllVO> list = JsonUtil.getJsonToList(data, TableExampleListAllVO.class);
        List<String> userId = list.stream().map(t -> t.getRegistrant()).collect(Collectors.toList());
        List<UserEntity> userList = userService.getUserName(userId);
        for (TableExampleListAllVO tableExampleListVO : list) {
            UserEntity user = userList.stream().filter(t -> t.getId().equals(tableExampleListVO.getRegistrant())).findFirst().orElse(null);
            tableExampleListVO.setRegistrant(user != null ? user.getRealName(): "");
        }
        ListVO<TableExampleListAllVO> vo = new ListVO<>();
        vo.setList(list);
        return ActionResult.success(vo);
    }

    /**
     * 列表
     *
     * @return
     */
    @Operation(summary="获取延伸扩展列表(行政区划)")
    @GetMapping("/IndustryList")
    public ActionResult industryList() {
        List<ProvinceEntity> data = provinceService.getListByParentId("-1");
        List<TableExampleIndustryListVO> listVos = JsonUtil.getJsonToList(data, TableExampleIndustryListVO.class);
        ListVO<TableExampleIndustryListVO> vo = new ListVO<>();
        vo.setList(listVos);
        return ActionResult.success(vo);
    }

    /**
     * 列表
     *
     * @param id 主键值
     * @return
     */
    @Operation(summary="获取城市信息列表(获取延伸扩展列表(行政区划))")
    @GetMapping("/CityList/{id}")
    public ActionResult cityList(@PathVariable("id") String id) {
        List<ProvinceEntity> data = provinceService.getListByParentId(id);
        List<TableExampleCityListVO> listVos = JsonUtil.getJsonToList(data, TableExampleCityListVO.class);
        ListVO<TableExampleCityListVO> vo = new ListVO<>();
        vo.setList(listVos);
        return ActionResult.success(vo);
    }

    /**
     * 列表（表格树形）
     *
     * @return
     */
    @Operation(summary="表格树形")
    @GetMapping("/ControlSample/TreeList")
    public ActionResult treeList(String isTree) {
        List<DictionaryDataEntity> data = dictionaryDataService.getList("d59a3cf65f9847dbb08be449e3feae16");
        List<TableExampleTreeModel> treeList = new ArrayList<>();
        for (DictionaryDataEntity entity : data) {
            TableExampleTreeModel treeModel = new TableExampleTreeModel();
            treeModel.setId(entity.getId());
            treeModel.setText(entity.getFullName());
            treeModel.setParentId(entity.getParentId());
            treeModel.setLoaded(true);
            treeModel.setExpanded(true);
            treeModel.setHt(JsonUtil.entityToMap(entity));
            treeList.add(treeModel);
        }
        if (isTree != null && "1".equals(isTree)) {
            List<SumTree<TableExampleTreeModel>> trees = TreeDotUtils.convertListToTreeDot(treeList);
            List<TableExampleTreeModel> listVO = JsonUtil.getJsonToList(trees, TableExampleTreeModel.class);
            ListVO vo = new ListVO();
            vo.setList(listVO);
            return ActionResult.success(vo);
        }
        ListVO vo = new ListVO();
        vo.setList(treeList);
        return ActionResult.success(vo);
    }

    /**
     * 信息
     *
     * @param id 主键值
     * @return
     */
    @Operation(summary="获取普通表格示例信息")
    @GetMapping("/{id}")
    public ActionResult<TableExampleInfoVO> info(@PathVariable("id") String id) throws DataException {
        TableExampleEntity entity = tableExampleService.getInfo(id);
        TableExampleInfoVO vo = JsonUtilEx.getJsonToBeanEx(entity, TableExampleInfoVO.class);
        return ActionResult.success(vo);
    }

    /**
     * 删除
     *
     * @param id 主键值
     * @return
     */
    @Operation(summary="删除项目")
    @DeleteMapping("/{id}")
    public ActionResult delete(@PathVariable("id") String id) {
        TableExampleEntity entity = tableExampleService.getInfo(id);
        if (entity != null) {
            tableExampleService.delete(entity);
            return ActionResult.success(MsgCode.SU003.get());
        }
        return ActionResult.fail(MsgCode.FA003.get());
    }

    /**
     * 创建
     *
     * @param tableExampleCrForm 实体对象
     * @return
     */
    @Operation(summary="新建项目")
    @PostMapping
    public ActionResult create(@RequestBody @Valid TableExampleCrForm tableExampleCrForm) {
        TableExampleEntity entity = JsonUtil.getJsonToBean(tableExampleCrForm, TableExampleEntity.class);
        entity.setCostAmount(entity.getCostAmount() == null ? new BigDecimal("0") : entity.getCostAmount());
        entity.setTunesAmount(entity.getTunesAmount() == null ? new BigDecimal("0") : entity.getTunesAmount());
        entity.setProjectedIncome(entity.getProjectedIncome() == null ? new BigDecimal("0") : entity.getProjectedIncome());
        entity.setSign("0000000");
        tableExampleService.create(entity);
        return ActionResult.success("创建成功");
    }

    /**
     * 更新
     *
     * @param id                 主键值
     * @param tableExampleUpForm 实体对象
     * @return
     */
    @Operation(summary="更新项目")
    @PutMapping("/{id}")
    public ActionResult update(@PathVariable("id") String id, @RequestBody @Valid TableExampleUpForm tableExampleUpForm) {
        TableExampleEntity entity = JsonUtil.getJsonToBean(tableExampleUpForm, TableExampleEntity.class);
        entity.setCostAmount(entity.getCostAmount() == null ? new BigDecimal("0") : entity.getCostAmount());
        entity.setTunesAmount(entity.getTunesAmount() == null ? new BigDecimal("0") : entity.getTunesAmount());
        entity.setProjectedIncome(entity.getProjectedIncome() == null ? new BigDecimal("0") : entity.getProjectedIncome());
        boolean flag = tableExampleService.update(id, entity);
        if (flag == false) {
            return ActionResult.fail(MsgCode.FA002.get());
        }
        return ActionResult.success(MsgCode.SU004.get());
    }

    /**
     * 更新标签
     *
     * @param id                     主键值
     * @param tableExampleSignUpForm 实体对象
     * @return
     */
    @Operation(summary="更新标记")
    @PutMapping("/UpdateSign/{id}")
    public ActionResult updateSign(@PathVariable("id") String id, @RequestBody @Valid TableExampleSignUpForm tableExampleSignUpForm) {
        TableExampleEntity entity = JsonUtil.getJsonToBean(tableExampleSignUpForm, TableExampleEntity.class);
        TableExampleEntity tableExampleEntity = tableExampleService.getInfo(id);
        if (tableExampleEntity == null) {
            return ActionResult.success(MsgCode.FA002.get());
        }
        tableExampleEntity.setSign(entity.getSign());
        tableExampleService.update(id, entity);
        return ActionResult.success(MsgCode.SU004.get());
    }

    /**
     * 行编辑
     *
     * @param tableExampleRowUpForm 实体对象
     * @return
     */
    @Operation(summary="行编辑")
    @PutMapping("/{id}/Actions/RowsEdit")
    public ActionResult rowEditing(@PathVariable("id") String id, @RequestBody @Valid TableExampleRowUpForm tableExampleRowUpForm) {
        TableExampleEntity entity = JsonUtil.getJsonToBean(tableExampleRowUpForm, TableExampleEntity.class);
        entity.setCostAmount(entity.getCostAmount() == null ? new BigDecimal("0") : entity.getCostAmount());
        entity.setTunesAmount(entity.getTunesAmount() == null ? new BigDecimal("0") : entity.getTunesAmount());
        entity.setProjectedIncome(entity.getProjectedIncome() == null ? new BigDecimal("0") : entity.getProjectedIncome());
        entity.setId(id);
        boolean falg = tableExampleService.rowEditing(entity);
        if (falg == false) {
            return ActionResult.fail(MsgCode.FA002.get());
        }
        return ActionResult.success(MsgCode.SU004.get());
    }

    /**
     * 发送
     *
     * @param id 主键值
     * @return
     */
    @Operation(summary="发送批注")
    @PostMapping("/{id}/Postil")
    public ActionResult sendPostil(@PathVariable("id") String id, @RequestBody PostilSendForm postilSendForm) {
        TableExampleEntity tableExampleEntity = tableExampleService.getInfo(id);
        if (tableExampleEntity == null) {
            return ActionResult.success(MsgCode.FA005.get());
        }
        UserInfo userInfo = userProvider.get();
        PostilModel model = new PostilModel();
        model.setCreatorTime(DateUtil.getNow("+8"));
        model.setText(postilSendForm.getText());
        model.setUserId(userInfo != null ? userInfo.getRealName() + "/" + userInfo.getUserAccount() : "");
        List<PostilModel> list = new ArrayList<>();
        list.add(model);
        if (!StringUtils.isEmpty(tableExampleEntity.getPostilJson())) {
            list.addAll(JsonUtil.getJsonToList(tableExampleEntity.getPostilJson(), PostilModel.class));
        }

        String postilJson = JsonUtil.getObjectToString(list);
        tableExampleEntity.setPostilJson(postilJson);
        tableExampleEntity.setPostilCount(list.size());
        tableExampleService.update(id, tableExampleEntity);
        return ActionResult.success(MsgCode.SU012.get());
    }


    /**
     * 发送
     *
     * @param id 主键值
     * @return
     */
    @Operation(summary="获取批注")
    @GetMapping("/{id}/Actions/Postil")
    public ActionResult getPostil(@PathVariable("id") String id) {
        TableExampleEntity tableExampleEntity = tableExampleService.getInfo(id);
        if (tableExampleEntity == null) {
            return ActionResult.success(MsgCode.FA012.get());
        }
        PostilInfoVO vo = new PostilInfoVO();
        vo.setPostilJson(tableExampleEntity.getPostilJson());
        return ActionResult.success(vo);
    }

    /**
     * 删除批注
     *
     * @param id 主键值
     * @return
     */
    @Operation(summary="删除批注")
    @DeleteMapping("/{id}/Postil/{index}")
    public ActionResult deletePostil(@PathVariable("id") String id, @PathVariable("index") int index) {
        TableExampleEntity tableExampleEntity = tableExampleService.getInfo(id);
        if (tableExampleEntity == null) {
            return ActionResult.success(MsgCode.FA003.get());
        }
        List<PostilModel> list = JsonUtil.getJsonToList(tableExampleEntity.getPostilJson(), PostilModel.class);
        list.remove(index);
        String postilJson = JsonUtil.getObjectToString(list);
        tableExampleEntity.setPostilJson(postilJson);
        tableExampleEntity.setPostilCount((list.size()));
        tableExampleService.update(id, tableExampleEntity);
        return ActionResult.success(MsgCode.SU003.get());
    }
}
