package com.bringspring.extend.controller;

import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.vo.ListVO;
import com.bringspring.common.base.vo.PageListVO;
import com.bringspring.common.base.vo.PaginationVO;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.extend.entity.ProductgoodsEntity;
import com.bringspring.extend.model.productgoods.*;
import com.bringspring.extend.service.ProductgoodsService;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.Operation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;

/**
 * 产品商品
 *
 * @版本： V1.0.0
 * @版权： 荣科科技股份有限公司
 * @作者： RKKJ开发平台组
 * @日期： 2021-07-10 15:57:50
 */
@Slf4j
@RestController
@Tag(name = "产品商品", description = "产品商品信息管理")
@RequestMapping("/api/extend/saleOrder/Goods")
public class ProductgoodsController {

    @Autowired
    private ProductgoodsService productgoodsService;

    /**
     * 列表
     *
     * @param type
     * @return
     */
    @GetMapping("/getGoodList")
    @Operation(summary="列表")
    public ActionResult list(String type) {
        List<ProductgoodsEntity> list = productgoodsService.getGoodList(type);
        List<ProductgoodsListVO> listVO = JsonUtil.getJsonToList(list, ProductgoodsListVO.class);
        ListVO vo = new ListVO();
        vo.setList(listVO);
        return ActionResult.success(vo);
    }

    /**
     * 列表
     *
     * @param goodsPagination
     * @return
     */
    @GetMapping
    @Operation(summary="列表")
    public ActionResult list(ProductgoodsPagination goodsPagination) {
        List<ProductgoodsEntity> list = productgoodsService.getList(goodsPagination);
        List<ProductgoodsListVO> listVO = JsonUtil.getJsonToList(list, ProductgoodsListVO.class);
        PageListVO vo = new PageListVO();
        vo.setList(listVO);
        PaginationVO page = JsonUtil.getJsonToBean(goodsPagination, PaginationVO.class);
        vo.setPagination(page);
        return ActionResult.success(vo);
    }

    /**
     * 创建
     *
     * @param goodsCrForm
     * @return
     */
    @PostMapping
    @Operation(summary="创建")
    public ActionResult create(@RequestBody @Valid ProductgoodsCrForm goodsCrForm) {
        ProductgoodsEntity entity = JsonUtil.getJsonToBean(goodsCrForm, ProductgoodsEntity.class);
        productgoodsService.create(entity);
        return ActionResult.success(MsgCode.SU001.get());
    }

    /**
     * 信息
     *
     * @param id
     * @return
     */
    @Operation(summary="信息")
    @GetMapping("/{id}")
    public ActionResult<ProductgoodsInfoVO> info(@PathVariable("id") String id) {
        ProductgoodsEntity entity = productgoodsService.getInfo(id);
        ProductgoodsInfoVO vo = JsonUtil.getJsonToBean(entity, ProductgoodsInfoVO.class);
        return ActionResult.success(vo);
    }

    /**
     * 更新
     *
     * @param id
     * @return
     */
    @PutMapping("/{id}")
    @Operation(summary="更新")
    public ActionResult update(@PathVariable("id") String id, @RequestBody @Valid ProductgoodsUpForm goodsCrFormUpForm) {
        ProductgoodsEntity entity = JsonUtil.getJsonToBean(goodsCrFormUpForm, ProductgoodsEntity.class);
        boolean ok = productgoodsService.update(id, entity);
        if (ok) {
            return ActionResult.success(MsgCode.SU004.get());
        }
        return ActionResult.fail(MsgCode.FA002.get());
    }

    /**
     * 删除
     *
     * @param id
     * @return
     */
    @DeleteMapping("/{id}")
    @Operation(summary="删除")
    public ActionResult delete(@PathVariable("id") String id) {
        ProductgoodsEntity entity = productgoodsService.getInfo(id);
        if (entity != null) {
            productgoodsService.delete(entity);
        }
        return ActionResult.success(MsgCode.SU003.get());
    }

    /**
     * 下拉
     *
     * @param goodsPagination
     * @return
     */
    @GetMapping("/Selector")
    @Operation(summary="下拉")
    public ActionResult listSelect(ProductgoodsPagination goodsPagination) {
        goodsPagination.setCurrentPage(1);
        goodsPagination.setPageSize(50);
        List<ProductgoodsEntity> list = productgoodsService.getList(goodsPagination);
        List<ProductgoodsListVO> listVO = JsonUtil.getJsonToList(list, ProductgoodsListVO.class);
        ListVO vo = new ListVO();
        vo.setList(listVO);
        return ActionResult.success(vo);
    }

}
