package com.bringspring.extend.controller;

import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.Pagination;
import com.bringspring.common.base.vo.PageListVO;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.extend.entity.CustomerEntity;
import com.bringspring.extend.model.customer.CustomerCrForm;
import com.bringspring.extend.model.customer.CustomerInfoVO;
import com.bringspring.extend.model.customer.CustomerListVO;
import com.bringspring.extend.model.customer.CustomerUpForm;
import com.bringspring.extend.service.CustomerService;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.Operation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;

/**
 * 客户信息
 *
 * @版本： V1.0.0
 * @版权： 荣科科技股份有限公司
 * @作者： RKKJ开发平台组
 * @日期： 2021-07-10 14:09:05
 */
@Slf4j
@RestController
@Tag(name = "客户信息", description = "客户信息管理")
@RequestMapping("/api/extend/saleOrder/Customer")
public class CustomerController {

    @Autowired
    private CustomerService customerService;

    /**
     * 列表
     *
     * @param pagination
     * @return
     */
    @GetMapping
    @Operation(summary="列表")
    public ActionResult list(Pagination pagination) {
        pagination.setPageSize(50);
        pagination.setCurrentPage(1);
        List<CustomerEntity> list = customerService.getList(pagination);
        List<CustomerListVO> listVO = JsonUtil.getJsonToList(list, CustomerListVO.class);
        PageListVO vo = new PageListVO();
        vo.setList(listVO);
        return ActionResult.success(vo);
    }

    /**
     * 创建
     *
     * @param customerCrForm
     * @return
     */
    @PostMapping
    @Operation(summary="创建")
    public ActionResult create(@RequestBody @Valid CustomerCrForm customerCrForm) {
        CustomerEntity entity = JsonUtil.getJsonToBean(customerCrForm, CustomerEntity.class);
        customerService.create(entity);
        return ActionResult.success(MsgCode.SU001.get());
    }

    /**
     * 信息
     *
     * @param id
     * @return
     */
    @GetMapping("/{id}")
    @Operation(summary="信息")
    public ActionResult<CustomerInfoVO> info(@PathVariable("id") String id) {
        CustomerEntity entity = customerService.getInfo(id);
        CustomerInfoVO vo = JsonUtil.getJsonToBean(entity, CustomerInfoVO.class);
        return ActionResult.success(vo);
    }

    /**
     * 更新
     *
     * @param id
     * @return
     */
    @PutMapping("/{id}")
    @Operation(summary="更新")
    public ActionResult update(@PathVariable("id") String id, @RequestBody @Valid CustomerUpForm customerUpForm) {
        CustomerEntity entity = JsonUtil.getJsonToBean(customerUpForm, CustomerEntity.class);
        boolean ok = customerService.update(id, entity);
        if (ok) {
            return ActionResult.success(MsgCode.SU004.get());
        }
        return ActionResult.fail(MsgCode.FA002.get());
    }

    /**
     * 删除
     *
     * @param id
     * @return
     */
    @DeleteMapping("/{id}")
    @Operation(summary="删除")
    public ActionResult delete(@PathVariable("id") String id) {
        CustomerEntity entity = customerService.getInfo(id);
        if (entity != null) {
            customerService.delete(entity);
        }
        return ActionResult.success(MsgCode.SU003.get());
    }


}
