package com.bringspring.files.utils;


import net.lingala.zip4j.ZipFile;
import net.lingala.zip4j.model.ZipParameters;
import net.lingala.zip4j.model.enums.EncryptionMethod;

import java.io.File;

public class ZipUtils {
    public static void main(String[] args) throws Exception {
        //压缩
//        zip("D:\\work\\jsbos-logistics\\jsbos-resources\\DataBackupFile\\演示环境数据库备份\\jsbos_logistics_standard",
//                "D:\\work\\jsbos-logistics\\jsbos-resources\\DataBackupFile\\演示环境数据库备份\\zip","rkkj88888888",
//                "D:\\work\\jsbos-logistics\\jsbos-resources\\DataBackupFile\\演示环境数据库备份\\zip\\test.zip");
        //解压
//        unZip("F:/wwwtest/zip/wwwtest-20231009154828.zip","F:/wwwtest/unZip/wwwtest-20231009154828","rkkj88888888");
    }

    /**
     * 压缩文件夹中的所有文件
     *
     * @param rootFolder        待压缩的文件夹根目录
     * @param zipFileTargetPath 最终生成的压缩文件路径
     * @throws Exception rootFolder不是文件夹或rootFolder下没有文件时抛出异常
     */
    public static void zip(String rootFolder, String zipFileTargetPath) throws Exception {
        zip(new File(rootFolder), new File(zipFileTargetPath));
    }

    /**
     * 压缩文件夹中的所有文件，并设置压缩文件密码
     *
     * @param rootFolder        待压缩的文件夹根目录
     * @param zipFileTargetPath 最终生成的压缩文件路径
     * @param password          压缩文件密码
     * @throws Exception rootFolder不是文件夹或rootFolder下没有文件时抛出异常
     */
    public static void zip(String rootFolder, String zipFileTargetPath, String password, String fileName) throws Exception {

        zip(new File(rootFolder), new File(zipFileTargetPath), password, fileName);
    }

    public static void zip(File rootFolder, File zipFileTargetPath) throws Exception {
        verifyZipRootFolder(rootFolder);
        try (ZipFile zipFile = new ZipFile(zipFileTargetPath)) {
            for (File f : rootFolder.listFiles()) {
                if (f.isFile()) {
                    zipFile.addFile(f);
                } else if (f.isDirectory()) {
                    zipFile.addFolder(f);
                }
            }
        }
    }

    public static void zip(File rootFolder, File zipFileTargetPath, String password, String fileName) throws Exception {
        if (password == null || password.isEmpty()) {
            zip(rootFolder, zipFileTargetPath);
            return;
        }
        //如果目的路径不存在，则新建
        if (!zipFileTargetPath.exists()) {
            zipFileTargetPath.mkdirs();
        }
        verifyZipRootFolder(rootFolder);

        // 压缩文件启用密码，加密算法采用AES
        ZipParameters zipParameters = new ZipParameters();
        zipParameters.setEncryptFiles(true);
        zipParameters.setEncryptionMethod(EncryptionMethod.AES);

        try (ZipFile zipFile = new ZipFile(fileName, password.toCharArray())) {
//            zipFile.addFolder(rootFolder,zipParameters);
            for (File f : rootFolder.listFiles()) {
                if (f.isFile()) {
                    zipFile.addFile(f, zipParameters);
                } else if (f.isDirectory()) {
                    zipFile.addFolder(f, zipParameters);
                }
            }
        }
    }

    /**
     * 压缩文件夹中的所有文件，并设置压缩文件密码
     *
     * @param rootFolder        待压缩的文件夹根目录
     * @param zipFileTargetPath 最终生成的压缩文件路径
     * @param password          压缩文件密码
     * @throws Exception rootFolder不是文件夹或rootFolder下没有文件时抛出异常
     */
    public static void zipProjectFiles(String rootFolder, String zipFileTargetPath, String password, String fileName) {

        try {
            zipProjectFiles(new File(rootFolder), new File(zipFileTargetPath), password, fileName);
        } catch (Exception e) {
            System.out.println("待压缩文件目录:" + rootFolder + "不存在");
        }
    }

    public static void zipProjectFiles(File rootFolder, File zipFileTargetPath, String password, String fileName) throws Exception {
        if (password == null || password.isEmpty()) {
            zip(rootFolder, zipFileTargetPath);
            return;
        }
        //如果目的路径不存在，则新建
        if (!zipFileTargetPath.exists()) {
            zipFileTargetPath.mkdirs();
        }
        verifyZipRootFolder(rootFolder);

        // 压缩文件启用密码，加密算法采用AES
        ZipParameters zipParameters = new ZipParameters();
        zipParameters.setEncryptFiles(true);
        zipParameters.setEncryptionMethod(EncryptionMethod.AES);

        try (ZipFile zipFile = new ZipFile(fileName, password.toCharArray())) {
            zipFile.addFolder(rootFolder, zipParameters);

        }
    }

    /**
     * 解压文件
     *
     * @param zipFilePath  待解压的zip文件所在路径
     * @param targetFolder 解压到指定的目录
     * @throws Exception
     */
    public static void unZip(String zipFilePath, String targetFolder) throws Exception {
        unZip(new File(zipFilePath), targetFolder);
    }

    public static void unZip(String zipFilePath, String targetFolder, String password) throws Exception {
        unZip(new File(zipFilePath), targetFolder, password);
    }

    public static void unZip(File zipFilePath, String targetFolder) throws Exception {
        try (ZipFile zipFile = new ZipFile(zipFilePath)) {
            zipFile.extractAll(targetFolder);
        }
    }

    public static void unZip(File zipFilePath, String targetFolder, String password) throws Exception {
        try (ZipFile zipFile = new ZipFile(zipFilePath)) {
            if (zipFile.isEncrypted()) {
                zipFile.setPassword(password.toCharArray());
            }
            zipFile.extractAll(targetFolder);
        }
    }

    private static void verifyZipRootFolder(File rootFolder) {
        if (!rootFolder.isDirectory()) {
            throw new RuntimeException("压缩路径根目录必须是文件夹");
        }
        if (rootFolder.listFiles() == null) {
            throw new NullPointerException("压缩路径根目录下没有文件");
        }
    }

}

