package com.bringspring.files.controller;

import cn.hutool.core.util.ObjectUtil;
import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.vo.PageListVO;
import com.bringspring.common.base.vo.PaginationVO;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.files.entity.FileEntity;
import com.bringspring.files.param.FilePageParam;
import com.bringspring.files.service.FileService;
import com.bringspring.system.permission.util.BaseDataUtil;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.tags.Tag;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.responses.ApiResponse;
import io.swagger.v3.oas.annotations.responses.ApiResponses;
import jakarta.annotation.Resource;
import jakarta.validation.constraints.NotEmpty;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.List;

@Slf4j
@Tag(name = "文件管理", description = "文件上传、下载、删除等操作")
@RestController
@RequestMapping("/api/files")
public class FileController {


    @Resource
    private FileService fileService;

    @Autowired
    private BaseDataUtil baseDataUtil;


    /**
     * 获取文件分页列表
     */
    @Operation(summary = "获取文件分页列表", description = "根据分页参数获取文件列表，支持按条件筛选")
    @ApiResponses({
        @ApiResponse(responseCode = "200", description = "成功获取文件列表"),
        @ApiResponse(responseCode = "400", description = "参数错误"),
        @ApiResponse(responseCode = "500", description = "服务器内部错误")
    })
    @GetMapping("/page")
    public ActionResult page(
        @Parameter(description = "文件分页查询参数") FilePageParam filePageParam) {
        List<FileEntity> list = fileService.getList(filePageParam);
        for (FileEntity file : list) {
            if (!ObjectUtil.isEmpty(file.getCreatorUserId())) {
                file.setCreatorUserName(baseDataUtil.userSelectValueNoAccount(file.getCreatorUserId()));
            }
        }
        PageListVO vo = new PageListVO();
        vo.setList(list);
        PaginationVO page = JsonUtil.getJsonToBean(filePageParam, PaginationVO.class);
        vo.setPagination(page);
        return ActionResult.success(vo);
    }

    /**
     * 删除文件
     */
    @Operation(summary = "删除文件", description = "批量删除指定ID的文件")
    @ApiResponses({
        @ApiResponse(responseCode = "200", description = "文件删除成功"),
        @ApiResponse(responseCode = "400", description = "参数错误，ID列表不能为空"),
        @ApiResponse(responseCode = "404", description = "文件不存在"),
        @ApiResponse(responseCode = "500", description = "服务器内部错误")
    })
    @PostMapping(value = "/delete")
    public ActionResult<String> delete(
        @Parameter(description = "文件ID列表", required = true) 
        @RequestBody @NotEmpty(message = "集合不能为空") List<String> ids) {
        ActionResult res = fileService.delete(ids);
        return res;
    }
}
