package com.bringspring.oauth.service.impl;

import com.bringspring.common.util.StringUtils;
import com.bringspring.oauth.config.ThirdPartyConfig;
import com.bringspring.oauth.model.ThirdPartyResponse;
import com.bringspring.oauth.service.ThirdPartyAuthService;
import com.bringspring.system.base.exception.LoginException;
import org.springframework.http.*;
import org.springframework.stereotype.Service;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

import java.util.HashMap;
import java.util.Map;

/**
 * 第三方认证服务接口实现
 */
@Service
public class ThirdPartyAuthServiceImpl implements ThirdPartyAuthService {

    private final ThirdPartyConfig thirdPartyConfig;

    private final RestTemplate restTemplate;

    public ThirdPartyAuthServiceImpl(ThirdPartyConfig thirdPartyConfig, RestTemplate restTemplate) {
        this.thirdPartyConfig = thirdPartyConfig;
        this.restTemplate = restTemplate;
    }

    @Override
    public boolean thirdPartyVerify(String thirdType, String productId, String code) {
        // thirdType 第三方类型
        if (StringUtils.isBlank(thirdType)) {
            throw new LoginException("第三方类型不能为空");
        }
        if (StringUtils.isBlank(productId)) {
            throw new LoginException("产品ID不能为空");
        }
        if (StringUtils.isBlank(code)) {
            throw new LoginException("验证码不能为空");
        }

        switch (thirdType) {
            case "rkkjWxwork":
                return rkkjWxworkVerifyCode(productId, code);
            default:
                return false;
        }
    }

    @Override
    public boolean rkkjWxworkVerifyCode(String productId, String code) {
        String url = thirdPartyConfig.getRkkjWxworkVerifyCodeUrl() + productId;
        // 构建请求体
        Map<String, String> requestBody = new HashMap<>();
        requestBody.put("code", code);

        // 设置请求头（根据接口要求）
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);

        // 创建请求实体
        HttpEntity<Map<String, String>> requestEntity = new HttpEntity<>(requestBody, headers);

        try {
            // 使用exchange方法发送PUT请求
            ResponseEntity<ThirdPartyResponse> response = restTemplate.exchange(
                    url,
                    HttpMethod.PUT,
                    requestEntity,
                    ThirdPartyResponse.class
            );

            return response.getStatusCode() == HttpStatus.OK
                    && response.getBody() != null
                    && response.getBody().isData();
        } catch (RestClientException e) {
            throw new LoginException(e.getMessage());
        }
    }
}
