package com.bringspring.oauth.method.detail;

import com.bringspring.common.util.CacheUtil;
import com.bringspring.oauth.config.MyRoleHierarchy;
import com.bringspring.oauth.config.detail.MoreMethodUserDetailService;
import com.bringspring.oauth.model.LoginBySignUpModel;
import com.bringspring.oauth.service.LoginService;
import com.bringspring.oauth.service.ThirdPartyAuthService;
import com.bringspring.system.base.exception.LoginException;
import com.bringspring.system.permission.entity.UserEntity;
import lombok.SneakyThrows;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.userdetails.User;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Objects;

/**
 * 多种登录方式执行实际登录
 *
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2021/3/16
 */
@Service
public class UserDetailsServiceImpl implements MoreMethodUserDetailService {

    @Autowired
    private LoginService loginService;
    @Autowired
    private CacheUtil cacheUtil;
    @Autowired
    private ThirdPartyAuthService thirdPartyAuthService;


    @SneakyThrows
    @Override
    public UserExtend loadUserByUsername(String username) throws UsernameNotFoundException {
        if (username.endsWith("@admin")) {
            /**
             * 模拟用户账户
             */
            UserEntity userEntity = loginService.isExistUser("admin");

            Collection<SimpleGrantedAuthority> list = new ArrayList();
            UserExtend user = new UserExtend(userEntity.getAccount(), userEntity.getSecretkey(), userEntity.getPassword(), list);
            return user;
        } else {
            /**
             * 验证用户是否可以使用
             */
            UserEntity userEntity = loginService.isExistUser(username);

            Collection<SimpleGrantedAuthority> list = new ArrayList();
            //管理员
            if (Objects.equals(1, userEntity.getIsAdministrator())) {
                //鉴权动态获取权限时判断是否管理员 MyRoleHierachy
                list.add(MyRoleHierarchy.adminAuth);
            }
            UserExtend user = new UserExtend(userEntity.getAccount(), userEntity.getSecretkey(), userEntity.getPassword(), list);
            return user;
        }
    }

    /**
     * 单点登录
     *
     * @return
     * @throws Exception
     */
    @Override
    public UserDetails loadUserByAccount(String account) throws Exception {
        /**
         * 验证用户是否可以使用
         */
        UserEntity userEntity = loginService.isExistUser(account);
        Collection<SimpleGrantedAuthority> list = new ArrayList<>();
        //管理员
        if (Objects.equals(1, userEntity.getIsAdministrator())) {
            //鉴权动态获取权限时判断是否管理员 MyRoleHierachy
            list.add(MyRoleHierarchy.adminAuth);
        }
        UserDetails user = new User(userEntity.getAccount(), userEntity.getPassword(), list);
        return user;

    }

    /**
     * 短信验证码登陆
     *
     * @param code
     * @return
     * @throws Exception
     */
    @Override
    public UserDetails loadUserBySmsCode(String account, String code) throws Exception {
        //获取验证码和输入得验证码
        String tempCode = String.valueOf(cacheUtil.query(account));
        if (!tempCode.equals(code)) {
            throw new LoginException("验证码错误");
        }
        /**
         * 验证用户是否可以使用
         */
        UserEntity userEntity = loginService.isExistUser(account);
        Collection<SimpleGrantedAuthority> list = new ArrayList<>();
        //管理员
        if (Objects.equals(1, userEntity.getIsAdministrator())) {
            //鉴权动态获取权限时判断是否管理员 MyRoleHierachy
            list.add(MyRoleHierarchy.adminAuth);
        }
        UserDetails user = new User(userEntity.getAccount(), userEntity.getPassword(), list);
        return user;
    }

    @Override
    public UserDetails loadUserByThirdAuth(String account, String thirdType, String productId, String code) throws Exception {
        // 验证第三方授权是否成功
        boolean verifyResult = false;
        try {
            verifyResult = thirdPartyAuthService.thirdPartyVerify(thirdType, productId, code);
        } catch (Exception e) {
            throw new LoginException(e.getMessage());
        }

        if (!verifyResult) {
            throw new LoginException("第三方授权验证未通过");
        }

        /**
         * 验证用户是否可以使用
         */
        UserEntity userEntity = loginService.isExistUser(account);
        Collection<SimpleGrantedAuthority> list = new ArrayList<>();
        //管理员
        if (Objects.equals(1, userEntity.getIsAdministrator())) {
            //鉴权动态获取权限时判断是否管理员 MyRoleHierachy
            list.add(MyRoleHierarchy.adminAuth);
        }
        UserDetails user = new User(userEntity.getAccount(), userEntity.getPassword(), list);
        return user;
    }

    /**
     * 第一步、使用授权码找到对应用户
     * 第二步、验证用户是否能登录
     * 第三步、执行登录过程处理
     *
     * @param code
     * @return
     * @throws Exception
     */
    @Override
    public UserDetails loadUserByWeComCode(String code, String cropId, Integer agentId) throws Exception {
        /**
         * 根据企业微信授权码code获取用户名
         */
        String username = loginService.getAccountByWxCode(code, cropId);
        /**
         * 验证用户是否可以使用
         */
        UserEntity userEntity = loginService.isExistUser(username);
        Collection<SimpleGrantedAuthority> list = new ArrayList<>();
        //管理员
        if (Objects.equals(1, userEntity.getIsAdministrator())) {
            //鉴权动态获取权限时判断是否管理员 MyRoleHierachy
            list.add(MyRoleHierarchy.adminAuth);
        }
        UserDetails user = new User(userEntity.getAccount(), userEntity.getPassword(), list);
        return user;
    }

    @Override
    public UserDetails loadUserByDingCode(String code) throws Exception {
        /**
         * 根据钉钉授权码code获取用户名
         */
        String username = loginService.getAccountByDingCode(code);
        /**
         * 验证用户是否可以使用
         */
        UserEntity userEntity = loginService.isExistUser(username);
        Collection<SimpleGrantedAuthority> list = new ArrayList<>();
        //管理员
        if (Objects.equals(1, userEntity.getIsAdministrator())) {
            //鉴权动态获取权限时判断是否管理员 MyRoleHierachy
            list.add(MyRoleHierarchy.adminAuth);
        }
        UserDetails user = new User(userEntity.getAccount(), userEntity.getPassword(), list);
        return user;
    }

    @Override
    public UserDetails loadUserByDingCodeH5(String code) throws Exception {
        /**
         * 根据企业微信授权码code获取用户名
         */
        String username = loginService.getAccountByDingCodeH5(code);
        /**
         * 验证用户是否可以使用
         */
        UserEntity userEntity = loginService.isExistUser(username);
        Collection<SimpleGrantedAuthority> list = new ArrayList<>();
        //管理员
        if (Objects.equals(1, userEntity.getIsAdministrator())) {
            //鉴权动态获取权限时判断是否管理员 MyRoleHierachy
            list.add(MyRoleHierarchy.adminAuth);
        }
        UserDetails user = new User(userEntity.getAccount(), userEntity.getPassword(), list);
        return user;
    }

    /**
     * 根据微信小程序授权码code获取用户名
     */
    @Override
    public UserDetails loadUserByWxMiniappCode(LoginBySignUpModel loginBySignUpModel) throws Exception {
        String username = loginService.getAccountByWxMiniappCode(loginBySignUpModel);
        /**
         * 验证用户是否可以使用
         */
        UserEntity userEntity = loginService.isExistUser(username);
        Collection<SimpleGrantedAuthority> list = new ArrayList<>();
        // 管理员
        if (Objects.equals(1, userEntity.getIsAdministrator())) {
            //鉴权动态获取权限时判断是否管理员 MyRoleHierachy
            list.add(MyRoleHierarchy.adminAuth);
        }
        UserDetails user = new User(userEntity.getAccount(), userEntity.getPassword(), list);
        return user;
    }
}
