package com.bringspring.oauth.config.cas;

import org.jasig.cas.client.authentication.AuthenticationFilter;
import org.jasig.cas.client.session.SingleSignOutFilter;
import org.jasig.cas.client.session.SingleSignOutHttpSessionListener;
import org.jasig.cas.client.util.HttpServletRequestWrapperFilter;
import org.jasig.cas.client.validation.Cas30ProxyReceivingTicketValidationFilter;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.web.servlet.FilterRegistrationBean;
import org.springframework.boot.web.servlet.ServletListenerRegistrationBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import java.util.EventListener;
import java.util.HashMap;
import java.util.Map;

/**
 *
 * @author zhujunjie
 * @since 2024/1/5 19:39
 */
@Configuration
@ConditionalOnProperty(prefix = "config",name ="cas",havingValue = "true", matchIfMissing = false)
public class CasConfig {


    //cas认证服务中心地址
    @Value("${cas.server-url-prefix:}")
    private String CAS_SERVER_URL_PREFIX;

    //cas认证服务中心登陆地址
    @Value("${cas.server-login-url:}")
    private String CAS_SERVER_URL_LOGIN ;

    //客户端的地址
    @Value("${cas.client-host-url:}")
    private String SERVER_NAME;


    /**
     * 【注销请求】
     * 该过滤器主要用于拦截 CAS 服务器发送的注销请求，实现单点注销功能。其实现原理如下：
     * 1、当用户在 CAS 服务器上注销登录时，CAS 服务器会向所有已登录的客户端发送注销请求。
     * 2、SingleSignOutFilter 过滤器会拦截这个注销请求，并将其解析为一个票据（Ticket）。
     * 3、SingleSignOutFilter 过滤器会遍历所有已登录的用户会话（Session），查找包含与该票据相关联的 CAS 令牌（Token）的会话。
     * 4、对于包含该令牌的会话，SingleSignOutFilter 过滤器会调用 Session.invalidate() 方法，使该会话失效并注销用户登录。
     * 5、当用户再次访问应用程序时，由于会话已经失效，用户需要重新登录并获取新的会话。
     * @return
     */
    @Bean
    public FilterRegistrationBean filterSingleRegistration() {
        FilterRegistrationBean registration = new FilterRegistrationBean();
        registration.setFilter(new SingleSignOutFilter());
        // 设定匹配的路径
        registration.addUrlPatterns("/api/cas/*");
        Map<String,String> initParameters = new HashMap<String, String>();
        initParameters.put("casServerUrlPrefix", CAS_SERVER_URL_PREFIX);
        registration.setInitParameters(initParameters);
        // 设定加载的顺序
        registration.setOrder(1);
        return registration;
    }

    /**
     * 【验证 CAS 服务器发放的票据（Ticket）】
     *  Cas30ProxyReceivingTicketValidationFilter
     *  该过滤器用于验证 CAS 服务器发放的票据（Ticket），其实现原理如下：
     *  1、客户端携带票据访问应用程序时，Cas30ProxyReceivingTicketValidationFilter 过滤器会拦截该请求，并将票据信息发送给 CAS 服务器进行验证。
     *  2、CAS 服务器验证该票据的合法性，并返回用户信息给客户端。
     *  3、客户端将用户信息保存在会话中，并对该请求放行。
     * @return
     */
    @Bean
    public FilterRegistrationBean filterValidationRegistration() {
        FilterRegistrationBean registration = new FilterRegistrationBean();
        registration.setFilter(new Cas30ProxyReceivingTicketValidationFilter());
        // 设定匹配的路径
        registration.addUrlPatterns("/api/cas/*");
        Map<String,String>  initParameters = new HashMap<String, String>();
        initParameters.put("casServerUrlPrefix", CAS_SERVER_URL_PREFIX);
        initParameters.put("serverName", SERVER_NAME);
        initParameters.put("useSession", "true");   //是否将令牌存在会话中
        registration.setInitParameters(initParameters);
        // 设定加载的顺序
        registration.setOrder(1);
        return registration;
    }



    @Bean
    public FilterRegistrationBean filterAuthenticationRegistration() {
        FilterRegistrationBean registration = new FilterRegistrationBean();
        registration.setFilter(new AuthenticationFilter());
        // 设定匹配的路径
        registration.addUrlPatterns("/api/cas/*");

        Map<String,String>  initParameters = new HashMap<String, String>();
        initParameters.put("casServerLoginUrl", CAS_SERVER_URL_LOGIN);
        initParameters.put("serverName", SERVER_NAME);
        //设置忽略  退出登录不用登录
        initParameters.put("ignorePattern", "/system/*");

        registration.setInitParameters(initParameters);
        // 设定加载的顺序
        registration.setOrder(1);
        return registration;
    }


    /**
     *  HttpServletRequestWrapperFilter
     *
     *  该过滤器主要用于将 ServletRequest 包装为 HttpServletRequest，提供了一些新的方法，使得获取请求参数、请求头等更加方便，比如：
     *  1、在 ServletRequest 的基础上，HttpServletRequest 提供了更加丰富的请求信息获取方法，如获取请求参数、请求头、请求 URI 等等。
     *  2、在实际应用场景中，我们可能需要对请求进行更加细致的处理，比如通过请求参数来进行业务处理、通过请求头来进行请求授权、通过请求 URI 来进行请求路由等等。
     *  而 ServletRequest 并没有提供这些方便的方法，这时候我们就需要将其包装为 HttpServletRequest，从而获得更多的处理能力。
     *  3、此外，HttpServletRequest 还提供了一些方便的方法，如获取 Session、设置 Cookie 等等，这些方法也能够极大地帮助我们进行请求处理。
     *  因此，将 ServletRequest 包装为 HttpServletRequest，可以提供更加丰富的请求处理能力，同时也能够提高代码的可读性和可维护性。
     * @return
     */
    @Bean
    public FilterRegistrationBean filterWrapperRegistration() {
        FilterRegistrationBean registration = new FilterRegistrationBean();
        registration.setFilter(new HttpServletRequestWrapperFilter());
        // 设定匹配的路径
        registration.addUrlPatterns("/api/cas/*");
        // 设定加载的顺序
        registration.setOrder(1);
        return registration;
    }



    /**
     * 该过滤器用于监听 Session 的销毁事件，当用户在一个应用程序中注销登录或 Session 超时时，应用程序会销毁对应的 Session，
     * 但此时 CAS 服务器并不会知道该 Session 已经被销毁，因此该票据仍然可以在其他应用程序中被使用，这就导致了单点注销的问题。
     *
     * 为了解决这个问题，SingleSignOutHttpSessionListener 会监听 Session 的销毁事件，
     * 在 Session 销毁时，获取该 Session 中保存的票据信息，通过 CAS 客户端向 CAS 服务器发送注销请求。
     * 在接收到注销请求后，CAS 服务器会将该票据对应的 Session 注销，并通知其他应用程序将该票据对应的 Session 也进行注销。
     * 这样，即使用户在一个应用程序中注销登录或 Session 超时，该票据也会被注销，从而保证了单点注销的正确性。
     * @return
     */
    @Bean
    public ServletListenerRegistrationBean<EventListener> singleSignOutListenerRegistration(){
        ServletListenerRegistrationBean<EventListener> registrationBean = new ServletListenerRegistrationBean<EventListener>();
        registrationBean.setListener(new SingleSignOutHttpSessionListener());
        registrationBean.setOrder(1);
        return registrationBean;
    }
}
