package com.bringspring.oauth.util;

import com.bringspring.common.util.StringUtils;
import com.google.zxing.BarcodeFormat;
import com.google.zxing.EncodeHintType;
import com.google.zxing.MultiFormatWriter;
import com.google.zxing.common.BitMatrix;
import org.apache.hc.client5.http.classic.methods.*;
import org.apache.hc.client5.http.config.RequestConfig;
import org.apache.hc.client5.http.impl.classic.CloseableHttpClient;
import org.apache.hc.client5.http.impl.classic.CloseableHttpResponse;
import org.apache.hc.client5.http.impl.classic.HttpClients;
import org.apache.hc.client5.http.impl.io.PoolingHttpClientConnectionManagerBuilder;
import org.apache.hc.client5.http.io.HttpClientConnectionManager;
import org.apache.hc.client5.http.ssl.SSLConnectionSocketFactory;
import org.apache.hc.client5.http.ssl.SSLConnectionSocketFactoryBuilder;
import org.apache.hc.core5.http.ContentType;
import org.apache.hc.core5.http.HttpEntity;
import org.apache.hc.core5.http.io.entity.ByteArrayEntity;
import org.apache.hc.core5.http.io.entity.EntityUtils;
import org.apache.hc.core5.http.io.entity.StringEntity;
import org.apache.hc.core5.net.URIBuilder;
import org.apache.hc.core5.ssl.SSLContextBuilder;
import org.apache.hc.core5.util.Timeout;

import javax.net.ssl.SSLContext;
import java.io.*;
import java.net.MalformedURLException;
import java.net.URI;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.Hashtable;
import java.util.Map;
import java.util.concurrent.TimeUnit;

public class HttpUtils {

    private static final RequestConfig DEFAULT_REQUEST_CONFIG = RequestConfig.custom()
            .setResponseTimeout(Timeout.of(30, TimeUnit.SECONDS))
            .setConnectTimeout(Timeout.of(30, TimeUnit.SECONDS))
            .setConnectionRequestTimeout(Timeout.of(30, TimeUnit.SECONDS))
            .build();

    /**
     * get
     */
    public static CloseableHttpResponse doGet(String host, String path, String method,
                                              Map<String, String> headers, Map<String, String> querys) throws Exception {
        try (CloseableHttpClient httpClient = createHttpClient(host)) {
            HttpGet request = new HttpGet(buildUri(host, path, querys));
            addHeaders(request, headers);
            return httpClient.execute(request);
        }
    }

    /**
     * post form
     */
    public static CloseableHttpResponse doPost(String host, String path, String method,
                                               Map<String, String> headers, Map<String, String> querys,
                                               Map<String, String> bodys) throws Exception {
        try (CloseableHttpClient httpClient = createHttpClient(host)) {
            HttpPost request = new HttpPost(buildUri(host, path, querys));
            addHeaders(request, headers);

            if (bodys != null && !bodys.isEmpty()) {
                // 手动构建表单数据
                StringBuilder formData = new StringBuilder();
                for (Map.Entry<String, String> entry : bodys.entrySet()) {
                    if (formData.length() > 0) {
                        formData.append("&");
                    }
                    formData.append(URLEncoder.encode(entry.getKey(), StandardCharsets.UTF_8.name()));
                    formData.append("=");
                    formData.append(URLEncoder.encode(entry.getValue(), StandardCharsets.UTF_8.name()));
                }
                StringEntity entity = new StringEntity(formData.toString(), ContentType.APPLICATION_FORM_URLENCODED);
                request.setEntity(entity);
            }

            return httpClient.execute(request);
        }
    }

    /**
     * Post String
     */
    public static CloseableHttpResponse doPost(String host, String path, String method,
                                               Map<String, String> headers, Map<String, String> querys,
                                               String body) throws Exception {
        try (CloseableHttpClient httpClient = createHttpClient(host)) {
            HttpPost request = new HttpPost(buildUri(host, path, querys));
            addHeaders(request, headers);

            if (StringUtils.isNotEmpty(body)) {
                StringEntity entity = new StringEntity(body, ContentType.APPLICATION_JSON);
                request.setEntity(entity);
            }

            return httpClient.execute(request);
        }
    }

    /**
     * Post stream
     */
    public static CloseableHttpResponse doPost(String host, String path, String method,
                                               Map<String, String> headers, Map<String, String> querys,
                                               byte[] body) throws Exception {
        try (CloseableHttpClient httpClient = createHttpClient(host)) {
            HttpPost request = new HttpPost(buildUri(host, path, querys));
            addHeaders(request, headers);

            if (body != null) {
                ByteArrayEntity entity = new ByteArrayEntity(body, ContentType.APPLICATION_OCTET_STREAM);
                request.setEntity(entity);
            }

            return httpClient.execute(request);
        }
    }

    /**
     * Put String
     */
    public static CloseableHttpResponse doPut(String host, String path, String method,
                                              Map<String, String> headers, Map<String, String> querys,
                                              String body) throws Exception {
        try (CloseableHttpClient httpClient = createHttpClient(host)) {
            HttpPut request = new HttpPut(buildUri(host, path, querys));
            addHeaders(request, headers);

            if (StringUtils.isNotEmpty(body)) {
                StringEntity entity = new StringEntity(body, ContentType.APPLICATION_JSON);
                request.setEntity(entity);
            }

            return httpClient.execute(request);
        }
    }

    /**
     * Put stream
     */
    public static CloseableHttpResponse doPut(String host, String path, String method,
                                              Map<String, String> headers, Map<String, String> querys,
                                              byte[] body) throws Exception {
        try (CloseableHttpClient httpClient = createHttpClient(host)) {
            HttpPut request = new HttpPut(buildUri(host, path, querys));
            addHeaders(request, headers);

            if (body != null) {
                ByteArrayEntity entity = new ByteArrayEntity(body, ContentType.APPLICATION_OCTET_STREAM);
                request.setEntity(entity);
            }

            return httpClient.execute(request);
        }
    }

    /**
     * Delete
     */
    public static CloseableHttpResponse doDelete(String host, String path, String method,
                                                 Map<String, String> headers, Map<String, String> querys) throws Exception {
        try (CloseableHttpClient httpClient = createHttpClient(host)) {
            HttpDelete request = new HttpDelete(buildUri(host, path, querys));
            addHeaders(request, headers);
            return httpClient.execute(request);
        }
    }

    /**
     * 构建 URI
     */
    private static URI buildUri(String host, String path, Map<String, String> querys) throws Exception {
        URIBuilder uriBuilder = new URIBuilder(host + path);

        if (querys != null) {
            for (Map.Entry<String, String> query : querys.entrySet()) {
                if (StringUtils.isNotEmpty(query.getKey()) && StringUtils.isNotEmpty(query.getValue())) {
                    uriBuilder.addParameter(query.getKey(), query.getValue());
                }
            }
        }

        return uriBuilder.build();
    }

    /**
     * 创建 HttpClient 实例
     */
    private static CloseableHttpClient createHttpClient(String host) throws Exception {
        if (host.startsWith("https://")) {
            return createSSLHttpClient();
        } else {
            return HttpClients.custom()
                    .setDefaultRequestConfig(DEFAULT_REQUEST_CONFIG)
                    .build();
        }
    }

    /**
     * 创建支持 SSL 的 HttpClient - 修复版本
     */
    private static CloseableHttpClient createSSLHttpClient() throws Exception {
        // 创建信任所有证书的 SSLContext
        SSLContext sslContext = SSLContextBuilder.create()
                .loadTrustMaterial((chain, authType) -> true)
                .build();

        // 创建 SSL socket factory
        SSLConnectionSocketFactory sslSocketFactory = SSLConnectionSocketFactoryBuilder.create()
                .setSslContext(sslContext).build();
        HttpClientConnectionManager cm = PoolingHttpClientConnectionManagerBuilder.create()
                .setSSLSocketFactory(sslSocketFactory).build();

        return HttpClients.custom()
                .setDefaultRequestConfig(DEFAULT_REQUEST_CONFIG)
                .setConnectionManager(cm)
                .build();
    }

    /**
     * 添加请求头
     */
    private static void addHeaders(HttpUriRequestBase request, Map<String, String> headers) {
        if (headers != null) {
            for (Map.Entry<String, String> header : headers.entrySet()) {
                request.addHeader(header.getKey(), header.getValue());
            }
        }
    }

    /**
     * 从 HttpResponse 中获取响应内容字符串
     */
    public static String getResponseContent(CloseableHttpResponse response) throws IOException {
        if (response == null) {
            return null;
        }

        HttpEntity entity = response.getEntity();
        if (entity == null) {
            return null;
        }

        try {
            return EntityUtils.toString(entity);
        } catch (org.apache.hc.core5.http.ParseException e) {
            throw new IOException("Failed to parse response entity", e);
        } finally {
            EntityUtils.consume(entity);
            response.close();
        }
    }

    /**
     * 获取URL内容长度
     */
    public static Integer getUrlLength(String urlStr) {
        Integer length = 0;

        try (InputStream is = new URL(urlStr).openStream();
             InputStreamReader isr = new InputStreamReader(is, StandardCharsets.UTF_8);
             BufferedReader br = new BufferedReader(isr)) {

            StringBuffer sb = new StringBuffer();
            char[] c = new char[10240];
            int charsRead;

            while ((charsRead = br.read(c)) != -1) {
                sb.append(c, 0, charsRead);
            }
            length = sb.toString().length();

        } catch (MalformedURLException e) {
            e.printStackTrace();
        } catch (IOException e) {
            e.printStackTrace();
        }

        return length;
    }

    public static void main(String[] args) throws Exception {
        String text = "https://open.work.weixin.qq.com/wwopen/sso/qrConnect?";

        int width = 400;
        int height = 400;
        String format = "jpg";

        Hashtable<EncodeHintType, String> hints = new Hashtable<>();
        hints.put(EncodeHintType.CHARACTER_SET, "utf-8");

        BitMatrix bitMatrix = new MultiFormatWriter().encode(text, BarcodeFormat.QR_CODE, width, height, hints);

        File outputFile = new File("D:\\邀请二维码" + File.separator + "20210013.jpg");
    }
}