package com.bringspring.oauth.service.impl;

import cn.hutool.core.net.url.UrlBuilder;
import cn.hutool.core.util.BooleanUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.http.useragent.UserAgent;
import cn.hutool.http.useragent.UserAgentUtil;
import com.bringspring.common.auth.granter.UserDetailsServiceBuilder;
import com.bringspring.common.auth.service.LoginSatokenService;
import com.bringspring.common.auth.service.ThirdPartyAuthService;
import com.bringspring.common.auth.util.UserProvider;
import com.bringspring.common.base.UserInfo;
import com.bringspring.common.config.ConfigValueUtil;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.database.model.TenantVO;
import com.bringspring.common.database.util.TenantDataSourceUtil;
import com.bringspring.common.exception.LoginException;
import com.bringspring.common.exception.TenantDatabaseException;
import com.bringspring.common.model.login.BaseSystemInfo;
import com.bringspring.common.model.login.PcUserVO;
import com.bringspring.common.util.*;
import com.bringspring.common.util.context.RequestContext;
import com.bringspring.oauth.util.LoginHolder;
import com.bringspring.system.base.entity.SysConfigEntity;
import com.bringspring.system.base.exception.BaseException;
import com.bringspring.system.base.service.LogService;
import com.bringspring.system.base.service.SysConfigService;
import com.bringspring.system.permission.entity.UserEntity;
import com.bringspring.system.permission.service.UserService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.*;

import static com.bringspring.common.util.Constants.ADMIN_KEY;

/**
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2021/3/16
 */
@Slf4j
@Service
public class LoginSatokenServiceImpl implements LoginSatokenService {

    @Autowired
    private ConfigValueUtil configValueUtil;
    @Autowired
    private UserService userService;
    @Autowired
    private SysConfigService sysConfigService;
    @Autowired
    private RedisUtil redisUtil;
    @Autowired
    private CacheKeyUtil cacheKeyUtil;
    @Autowired
    private UserDetailsServiceBuilder userDetailsServiceBuilder;
    @Autowired
    private ThirdPartyAuthService thirdPartyAuthService;

    @Autowired
    private LogService logService;

    @Override
    public UserInfo getTenantAccount(UserInfo userInfo) throws LoginException {
        String tenantId = null;
        if (configValueUtil.isMultiTenancy()) {
            String[] tenantAccount = userInfo.getUserAccount().split("\\@");
            if (tenantAccount.length == 1) {
                //只输入账号, 1:配置的二级域名下只输入账号, 2:主域名下输入了租户号
                String referer = ServletUtil.getHeader("Referer");
                if (StringUtils.isNotEmpty(referer)) {
                    String remoteHost = UrlBuilder.of(referer).getHost();
                    String apiHost = UrlBuilder.of(RequestContext.isOrignPc() ? configValueUtil.getFrontDomain() : configValueUtil.getAppDomain()).getHost();
                    if (!ObjectUtil.equals(remoteHost, apiHost)
                            && remoteHost.endsWith(apiHost)) {
                        //二级域名访问, 输入的是账号
                        tenantId = remoteHost.split("\\.")[0];
                        userInfo.setUserAccount(tenantAccount[0]);
                    }
                }
                if (tenantId == null) {
                    //主域名访问, 输入的是租户号
                    tenantId = tenantAccount[0];
                    userInfo.setUserAccount(ADMIN_KEY);
                }
            } else {
                //租户号@账号
                tenantId = tenantAccount[0];
                userInfo.setUserAccount(tenantAccount[1]);
            }
            if (StringUtils.isEmpty(tenantId) || tenantAccount.length > 2 || StringUtils.isEmpty(userInfo.getUserAccount())) {
                throw new LoginException(MsgCode.LOG102.get());
            }
            TenantVO tenantVO = TenantDataSourceUtil.getRemoteTenantInfo(tenantId);
            TenantDataSourceUtil.switchTenant(tenantId, tenantVO);
            //切换成租户库
            userInfo.setTenantId(tenantId);
            userInfo.setTenantDbConnectionString(tenantVO.getDbName());
            userInfo.setTenantDbType(tenantVO.getType());
            //查库测试
            BaseSystemInfo baseSystemInfo = null;
            try {
                baseSystemInfo = getBaseSystemConfig(userInfo.getTenantId());
            } catch (Exception e) {
                log.error("登录获取系统配置失败: {}", e.getMessage());
            }
            if (baseSystemInfo == null || baseSystemInfo.getSingleLogin() == null) {
                throw new TenantDatabaseException();
            }
        }
        return userInfo;
    }

    @Override
    public UserInfo userInfo(UserInfo userInfo, BaseSystemInfo sysConfigInfo) throws LoginException {
        //获取账号信息
        UserEntity userEntity = LoginHolder.getUserEntity();
        if (userEntity == null) {
            userEntity = userDetailsServiceBuilder.getUserDetailService(userInfo.getUserDetailKey()).loadUserEntity(userInfo);
            LoginHolder.setUserEntity(userEntity);
        }

        checkUser(userEntity, userInfo, sysConfigInfo);

        userInfo.setUserId(userEntity.getId());
        userInfo.setUserAccount(userEntity.getAccount());
        userInfo.setUserName(userEntity.getRealName());
        userInfo.setUserIcon(userEntity.getHeadIcon());
        userInfo.setTheme(userEntity.getTheme());
        userInfo.setOrganizeId(userEntity.getOrganizeId());
        userInfo.setPortalId(userEntity.getPortalId());
        userInfo.setIsAdministrator(BooleanUtil.toBoolean(String.valueOf((userEntity.getIsAdministrator()))));
        if (!ADMIN_KEY.equals(userInfo.getUserAccount())) {
//            if (ObjectUtil.isNotEmpty(userEntity.getStanding())) {
//                userInfo.setIsAdministrator(Objects.equals(userEntity.getStanding(), 1));
//            }
        }
        // 添加过期时间
        String time = sysConfigInfo.getTokenTimeout();
        if (StringUtils.isNotEmpty(time)) {
            Integer minu = Integer.valueOf(time);
            userInfo.setOverdueTime(DateUtil.dateAddMinutes(null, minu));
            userInfo.setTokenTimeout(minu);
        }

        String ipAddr = IpUtil.getIpAddr();
        userInfo.setLoginIpAddress(ipAddr);
        userInfo.setLoginIpAddressName(IpUtil.getIpCity(ipAddr));
        userInfo.setLoginTime(DateUtil.getmmNow());
        UserAgent userAgent = UserAgentUtil.parse(ServletUtil.getUserAgent());
        if (userAgent != null) {
            userInfo.setLoginPlatForm(userAgent.getPlatform().getName() + " " + userAgent.getOsVersion());
            userInfo.setBrowser(userAgent.getBrowser().getName() + " " + userAgent.getVersion());
        }
        userInfo.setPrevLoginTime(userEntity.getPrevLogTime());
        userInfo.setPrevLoginIpAddress(userEntity.getPrevLogIp());
        userInfo.setPrevLoginIpAddressName(IpUtil.getIpCity(userEntity.getPrevLogIp()));
        // 生成id
        String token = RandomUtil.uuId();
        userInfo.setId(cacheKeyUtil.getLoginToken(userInfo.getTenantId()) + token);

        createUserOnline(userInfo);
        return userInfo;
    }

    @Override
    public PcUserVO getCurrentUser(String type, String systemCode, Integer isBackend) {
        return null;
    }

    /**
     * 创建用户在线信息
     *
     * @param userInfo
     */
    private void createUserOnline(UserInfo userInfo) {
        String userId = userInfo.getUserId();
//        long time= DateUtil.getTime(userInfo.getOverdueTime()) - DateUtil.getTime(new Date());

        String authorize = String.valueOf(redisUtil.getString(cacheKeyUtil.getUserAuthorize() + userId));
//        String loginOnlineKey=cacheKeyUtil.getLoginOnline() + userId;
        redisUtil.remove(authorize);
        //记录Token
//        redisUtil.insert(userInfo.getId(), userInfo,time);
        //记录在线
        if (ServletUtil.getIsMobileDevice()) {
//            redisUtil.insert(cacheKeyUtil.getMobileLoginOnline() + userId, userInfo.getId(), time);
            //记录移动设备CID,用于消息推送
            if (ServletUtil.getHeader("clientId") != null) {
                String clientId = ServletUtil.getHeader("clientId");
                Map<String, String> map = new HashMap<>(16);
                map.put(userInfo.getUserId(), clientId);
                redisUtil.insert(cacheKeyUtil.getMobileDeviceList(), map);
            }
        } else {
//            redisUtil.insert(loginOnlineKey, userInfo.getId(), time);
        }
    }

    public UserEntity checkUser(UserEntity userEntity, UserInfo userInfo, BaseSystemInfo sysConfigInfo) throws LoginException {
        if (userEntity == null) {
            throw new LoginException(MsgCode.LOG101.get());
        }
        //判断是否组织、岗位、角色、部门主管是否为空，为空则抛出异常
        //判断是否为管理员，是否为Admin(Admin为最高账号，不受限制)
        if (!ADMIN_KEY.equals(userEntity.getAccount()) || userEntity.getIsAdministrator() != 1) {
            List<String> posAndRole = new ArrayList<>();
            //没岗位，且没用户角色时直接提示没权限
//            List<UserRelationEntity> userPos = userRelationApi.getListByUserIdAndObjType(userEntity.getId(), PermissionConst.POSITION);
//            List<String> userPosIds = userPos.stream().map(t -> t.getObjectId()).collect(Collectors.toList());
//            userPosIds.add(userEntity.getId());
//            List<RoleRelationEntity> userRole = roleRelationApi.getListByObjectId(userPosIds, null);
//            posAndRole.addAll(userPosIds);
//            posAndRole.addAll(userRole.stream().map(t -> t.getRoleId()).collect(Collectors.toList()));
            //有岗位角色但是没有权限
//            if (CollectionUtil.isEmpty(posAndRole) || CollectionUtil.isEmpty(authorizeApi.getListByObjectId(posAndRole))) {
//                throw new LoginException(MsgCode.LOG004.get());
//            }

        }
        if (userEntity.getIsAdministrator() == 0) {
            if (userEntity.getEnabledMark() == null) {
                throw new LoginException(MsgCode.LOG005.get());
            }
            if (userEntity.getEnabledMark() == 0) {
                throw new LoginException(MsgCode.LOG006.get());
            }
        }
        if (userEntity.getDeleteMark() != null && userEntity.getDeleteMark() == 1) {
            throw new LoginException(MsgCode.LOG007.get());
        }
        //安全验证
        String ipAddr = IpUtil.getIpAddr();
        userInfo.setLoginIpAddress(IpUtil.getIpAddr());
        // 判断白名单
        if (!ADMIN_KEY.equals(userEntity.getAccount()) && "1".equals(sysConfigInfo.getWhitelistSwitch())) {
            List<String> ipList = Arrays.asList(sysConfigInfo.getWhitelistIp().split(","));
            if (!ipList.contains(ipAddr)) {
                throw new LoginException(MsgCode.LOG010.get());
            }
        }
        // 判断当前账号是否被锁定
        Integer lockMark = userEntity.getEnabledMark();
        if (Objects.nonNull(lockMark) && lockMark == 2) {
            // 获取解锁时间
            Date unlockTime = userEntity.getUnlockTime();
            // 账号锁定
            if (sysConfigInfo.getLockType() == 1 || Objects.isNull(unlockTime)) {
                throw new LoginException(MsgCode.LOG012.get());
            }
            // 延迟登陆锁定
            long millis = System.currentTimeMillis();
            // 系统设置的错误次数
            int passwordErrorsNumber = sysConfigInfo.getPasswordErrorsNumber() != null ? sysConfigInfo.getPasswordErrorsNumber() : 0;
            // 用户登录错误次数
            int logErrorCount = userEntity.getLogErrorCount() != null ? userEntity.getLogErrorCount() : 0;
            if (unlockTime.getTime() > millis) {
                // 转成分钟
                int time = (int) ((unlockTime.getTime() - millis) / (1000 * 60));
                throw new LoginException(MsgCode.LOG108.get(time + 1));
            } else if (unlockTime.getTime() < millis && logErrorCount >= passwordErrorsNumber) {
                // 已经接触错误时间锁定的话就重置错误次数
                userEntity.setLogErrorCount(0);
                userEntity.setEnabledMark(1);
                userService.updateById(userEntity);
            }
        }
        return userEntity;
    }

    @Override
    public void updatePasswordMessage() {
        UserInfo userInfo = UserProvider.getUser();
        UserEntity userEntity = userService.getInfo(userInfo.getUserId());
        BaseSystemInfo baseSystemInfo = sysConfigService.getSysInfo();
        if (baseSystemInfo.getPasswordIsUpdatedRegularly() == 1) {
            Date changePasswordDate = userEntity.getCreatorTime();
            if (userEntity.getChangePasswordDate() != null) {
                changePasswordDate = userEntity.getChangePasswordDate();
            }
            //当前时间
            Date nowDate = DateUtil.getNowDate();
            //更新周期
            Integer updateCycle = baseSystemInfo.getUpdateCycle();
            //提前N天提醒
            Integer updateInAdvance = baseSystemInfo.getUpdateInAdvance();
            Integer day = DateUtil.getDiffDays(changePasswordDate, nowDate);
            if (day >= (updateCycle - updateInAdvance)) {
//                MessageTemplateConfigEntity entity = messageTemplateApi.getInfoByEnCode("XTXXTX001", "1");
//                if (entity != null) {
//                    List<String> toUserIds = new ArrayList<>();
//                    toUserIds.add(userInfo.getUserId());
//                    sentMessageApi.sentMessage(toUserIds, entity.getTitle(), entity.getContent(), userInfo, Integer.parseInt(entity.getMessageSource()), Integer.parseInt(entity.getMessageType()));
//                }
            }
        }
    }

    @Override
    public BaseSystemInfo getBaseSystemConfig(String tenantId) {
        if (tenantId != null) {
            TenantDataSourceUtil.switchTenant(tenantId);
        }
        return sysConfigService.getSysInfo();
    }

    @Override
    public String getConfigByKeyName(String keyName) {
        // 入参合法性校验：提前拦截空值，避免下游无效调用
        if (keyName == null || keyName.trim().isEmpty()) {
            throw new LoginException("获取SysConfig配置失败：配置键名不能为空");
        }
        SysConfigEntity sysConfigEntity;
        try {
            // 调用服务层方法，捕获并细化业务异常
            sysConfigEntity = sysConfigService.getConfigByKeyName(keyName);
        } catch (BaseException e) {
            log.error("根据键名[{}]查询SysConfig配置失败", keyName, e); // 保留异常栈，便于排查根因
            throw new LoginException("获取SysConfig配置失败：" + e.getMessage(), e); // 传递异常根因
        } catch (Exception e) { // 兜底捕获服务层未声明的异常
            log.error("根据键名[{}]查询SysConfig配置时发生未知异常", keyName, e);
            throw new LoginException("获取SysConfig配置失败：系统异常", e);
        }

        // 校验服务层返回结果是否为空
        if (ObjectUtil.isEmpty(sysConfigEntity)) {
            log.warn("根据键名[{}]未查询到对应的SysConfig配置", keyName);
            throw new LoginException("获取SysConfig配置失败：未找到键名为[" + keyName + "]的配置");
        }

        // 校验配置值是否为空，避免JSON解析"null"字符串
        String keyValue = sysConfigEntity.getKeyValue();
        if (keyValue == null || keyValue.trim().isEmpty()) {
            log.warn("键名[{}]对应的SysConfig配置值为空", keyName);
            throw new LoginException("获取SysConfig配置失败：键名为[" + keyName + "]的配置值为空");
        }

        return keyValue;
    }

}
