package com.bringspring.oauth.service.impl;

import com.bringspring.common.auth.granter.TokenGranter;
import com.bringspring.common.auth.granter.TokenGranterBuilder;
import com.bringspring.common.auth.util.TenantProvider;
import com.bringspring.common.auth.util.UserProvider;
import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.UserInfo;
import com.bringspring.common.config.ConfigValueUtil;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.exception.LoginException;
import com.bringspring.common.exception.TenantDatabaseException;
import com.bringspring.common.exception.TenantInvalidException;
import com.bringspring.common.model.login.LoginVO;
import com.bringspring.common.util.RedisUtil;
import com.bringspring.common.util.StringUtils;
import com.bringspring.oauth.service.AuthService;
import com.bringspring.oauth.util.LoginHolder;
import com.bringspring.system.base.service.LogService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Map;

/**
 * 登录与退出服务 其他服务调用
 */
@Service
@Slf4j
public class AuthServiceImpl implements AuthService {

    @Autowired
    private TokenGranterBuilder tokenGranterBuilder;
    @Autowired
    private LogService logApi;
    @Autowired
    private ConfigValueUtil configValueUtil;
    @Autowired
    private RedisUtil redisUtil;

    /**
     * 登录
     * @param parameters {grant_type}
     * @return
     * @throws LoginException
     */
    @Override
    public ActionResult<LoginVO> login(Map<String, String> parameters) throws LoginException {
        long millis = System.currentTimeMillis();
        String grantType = parameters.getOrDefault("grant_type", "");
        TokenGranter tokenGranter = tokenGranterBuilder.getGranter(grantType);
        ActionResult<LoginVO> result;
        UserInfo userInfo = new UserInfo();
        try {
            String account = parameters.get("account");
            userInfo.setUserAccount(account);
            UserProvider.setLocalLoginUser(userInfo);
            result = tokenGranter.granter(parameters);
            //写入日志
            if (userInfo.getUserId() != null && !UserProvider.isTempUser(userInfo)) {
                logApi.writeLogAsync(userInfo.getUserId(), userInfo.getUserName() + "/" + userInfo.getUserAccount(), MsgCode.OA015.get(), (System.currentTimeMillis() - millis));
            }
        } catch (TenantDatabaseException tdex){
            throw tdex;
        } catch (Exception e){
            if(!(e instanceof LoginException || e instanceof TenantInvalidException)){
                String msg = e.getMessage();
                if(msg == null){
                    msg = MsgCode.OA007.get();
                }
                log.error("{}, Account: {}, Error: {}", MsgCode.OA007.get(), parameters.getOrDefault("account", ""), e.getMessage(), e);
                throw new LoginException(msg);
            }
            String userName = StringUtils.isNotEmpty(userInfo.getUserName()) ? userInfo.getUserName()+"/"+userInfo.getUserAccount() : userInfo.getUserAccount();
            logApi.writeLogAsync(userInfo.getUserId(), userName, e.getMessage(), userInfo, 0, null, (System.currentTimeMillis()-millis));
            throw e;
        }finally{
            LoginHolder.clearUserEntity();
            TenantProvider.clearBaseSystemIfo();
            // 请求之后就删除验证码 不论结果
            String imgCode = parameters.get("timestamp");
            if(StringUtils.isNotEmpty(imgCode)) {
                redisUtil.remove(imgCode);
            }
        }
        return result;
    }


    /**
     * 踢出用户, 用户将收到Websocket下线通知
     * 执行流程：认证服务退出用户->用户踢出监听->消息服务发送Websocket推送退出消息
     * @param tokens
     */
    @Override
    public ActionResult kickoutByToken(String... tokens){
        UserProvider.kickoutByToken(tokens);
        return ActionResult.success();
    }

    /**
     * 踢出用户, 用户将收到Websocket下线通知
     * 执行流程：认证服务退出用户->用户踢出监听->消息服务发送Websocket推送退出消息
     * @param userId
     * @param tenantId
     */
    @Override
    public ActionResult kickoutByUserId(String userId, String tenantId){
        UserProvider.kickoutByUserId(userId, tenantId);
        return ActionResult.success();
    }
}
