package com.bringspring.system.msgCenter.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.dynamic.datasource.annotation.DSTransactional;
import com.baomidou.mybatisplus.annotation.TableField;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bringspring.common.util.*;
import com.bringspring.system.external.config.mutil.WxCpConfiguration;
import com.bringspring.system.message.entity.SynThirdInfoEntity;
import com.bringspring.system.message.service.SynThirdInfoService;
import com.bringspring.system.msgCenter.entity.*;
import com.bringspring.system.msgCenter.enums.ChannelTypeEnum;
import com.bringspring.system.msgCenter.exception.MsgCenterException;
import com.bringspring.system.msgCenter.mapper.McTaskMsgMapper;
import com.bringspring.system.msgCenter.model.mcBusiness.McBusinessInfoVO;
import com.bringspring.system.msgCenter.model.mcMsgSend.McMsgSendInfoVO;
import com.bringspring.system.msgCenter.model.mcMsgSendTemplate.McMsgSendTemplateModel;
import com.bringspring.system.msgCenter.model.mcTaskMsg.McTaskMsgCrForm;
import com.bringspring.system.msgCenter.model.mcTaskMsg.McTaskMsgPagination;
import com.bringspring.system.msgCenter.model.mcTaskMsg.McTaskMsgUpForm;
import com.bringspring.system.msgCenter.service.*;
import com.bringspring.system.msgCenter.util.EmailUtil;
import com.bringspring.system.msgCenter.util.SensitiveWordUtil;
import com.bringspring.system.msgCenter.util.UrlUtils;
import com.bringspring.system.permission.entity.UserEntity;
import com.bringspring.system.permission.service.UserService;
import lombok.val;
import org.apache.commons.collections4.SetUtils;
import org.apache.commons.text.StringSubstitutor;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.lang.reflect.Field;
import java.util.*;
import java.util.concurrent.Executor;
import java.util.stream.Collectors;

import static com.bringspring.system.message.util.SynThirdConsts.*;
import static com.bringspring.system.msgCenter.constant.CommonConsts.*;
import static com.bringspring.system.msgCenter.enums.ChannelTypeEnum.getByCode;

/**
 * 业务场景消息表
 * 版本： V1.0.0
 * 版权： 荣科科技股份有限公司
 * 作者： RKKJ开发平台组
 * 日期： 2024-06-14
 */
@Service
public class McTaskMsgServiceImpl extends ServiceImpl<McTaskMsgMapper, McTaskMsgEntity> implements McTaskMsgService {


    @Autowired
    private UserProvider userProvider;
    @Autowired
    private UserService userService;

    @Autowired
    private McMsgSendService mcMsgSendService;
    @Autowired
    private McBusinessService mcBusinessService;
    @Autowired
    private McMsgTemplateService mcMsgTemplateService;
    @Autowired
    private McMsgAccountService mcMsgAccountService;

    @Autowired
    private SynThirdInfoService synThirdInfoService;
    @Autowired
    private WxCpConfiguration wxCpConfiguration;

    @Autowired
    private McTaskMsgContentService mcTaskMsgContentService;
    @Autowired
    private McTaskMsgReceiveService mcTaskMsgReceiveService;
    @Autowired
    private SendMessageService sendMessageService;
    @Autowired
    private McTaskMsgMapper mcTaskMsgMapper;

    /**
     * 取出线程池
     */
    @Autowired
    private Executor threadPoolExecutor;


    @Override
    public List<McTaskMsgEntity> getList(McTaskMsgPagination pagination) {
        String userId = userProvider.get().getUserId();
        List<String> AllIdList = new ArrayList();
        int total = 0;
        int mcTaskMsgNum = 0;
        QueryWrapper<McTaskMsgEntity> queryWrapper = new QueryWrapper<>();

        if (StringUtils.isNotEmpty(pagination.getKeyword())) {
            queryWrapper.lambda().and(t -> t.like(McTaskMsgEntity::getId, pagination.getKeyword()).or().like(McTaskMsgEntity::getTaskCode, pagination.getKeyword()).or().like(McTaskMsgEntity::getSendCode, pagination.getKeyword()));
            mcTaskMsgNum++;
        }
        if (StringUtils.isNotEmpty(pagination.getTaskCode())) {
            mcTaskMsgNum++;
            queryWrapper.lambda().like(McTaskMsgEntity::getTaskCode, pagination.getTaskCode());
        }

        if (StringUtils.isNotEmpty(pagination.getSendCode())) {
            mcTaskMsgNum++;
            queryWrapper.lambda().like(McTaskMsgEntity::getSendCode, pagination.getSendCode());
        }
        if (StringUtils.isNotEmpty(pagination.getEnabledMark())) {
            mcTaskMsgNum++;
            queryWrapper.lambda().eq(McTaskMsgEntity::getEnabledMark, Integer.parseInt(pagination.getEnabledMark()));
        }

        if (StringUtils.isNotEmpty(pagination.getToUserIds())) {
            mcTaskMsgNum++;
            queryWrapper.lambda().like(McTaskMsgEntity::getToUserIds, pagination.getToUserIds());
        }

        if (AllIdList.size() > 0) {
            queryWrapper.lambda().in(McTaskMsgEntity::getId, AllIdList);
        }
        //排序
        if (StringUtils.isEmpty(pagination.getSidx())) {
            queryWrapper.lambda().orderByDesc(McTaskMsgEntity::getCreatorTime);
        } else {
            try {
                String sidx = pagination.getSidx();
                McTaskMsgEntity mcTaskMsgEntity = new McTaskMsgEntity();
                Field declaredField = mcTaskMsgEntity.getClass().getDeclaredField(sidx);
                declaredField.setAccessible(true);
                String value = declaredField.getAnnotation(TableField.class).value();
                queryWrapper = "asc".equals(pagination.getSort().toLowerCase()) ?
                        queryWrapper.orderByAsc(value) : queryWrapper.orderByDesc(value);
            } catch (NoSuchFieldException e) {
                e.printStackTrace();
            }
        }
        if ((total > 0 && AllIdList.size() > 0) || total == 0) {
            Page<McTaskMsgEntity> page = new Page<>(pagination.getCurrentPage(), pagination.getPageSize());
            IPage<McTaskMsgEntity> userIPage = this.page(page, queryWrapper);
            return pagination.setData(userIPage.getRecords(), userIPage.getTotal());
        } else {
            List<McTaskMsgEntity> list = new ArrayList();
            return pagination.setData(list, list.size());
        }
    }

    @Override
    public List<McTaskMsgEntity> getTypeList(McTaskMsgPagination pagination, String dataType) {
        String userId = userProvider.get().getUserId();
        List<String> AllIdList = new ArrayList();
        int total = 0;
        int mcTaskMsgNum = 0;
        QueryWrapper<McTaskMsgEntity> queryWrapper = new QueryWrapper<>();
        if (StringUtils.isNotEmpty(pagination.getTaskCode())) {
            mcTaskMsgNum++;
            queryWrapper.lambda().like(McTaskMsgEntity::getTaskCode, pagination.getTaskCode());
        }

        if (StringUtils.isNotEmpty(pagination.getSendCode())) {
            mcTaskMsgNum++;
            queryWrapper.lambda().like(McTaskMsgEntity::getSendCode, pagination.getSendCode());
        }

        if (StringUtils.isNotEmpty(pagination.getToUserIds())) {
            mcTaskMsgNum++;
            queryWrapper.lambda().like(McTaskMsgEntity::getToUserIds, pagination.getToUserIds());
        }

        if (AllIdList.size() > 0) {
            queryWrapper.lambda().in(McTaskMsgEntity::getId, AllIdList);
        }
        //排序
        if (StringUtils.isEmpty(pagination.getSidx())) {
            queryWrapper.lambda().orderByDesc(McTaskMsgEntity::getCreatorTime);
        } else {
            try {
                String sidx = pagination.getSidx();
                McTaskMsgEntity mcTaskMsgEntity = new McTaskMsgEntity();
                Field declaredField = mcTaskMsgEntity.getClass().getDeclaredField(sidx);
                declaredField.setAccessible(true);
                String value = declaredField.getAnnotation(TableField.class).value();
                queryWrapper = "asc".equals(pagination.getSort().toLowerCase()) ?
                        queryWrapper.orderByAsc(value) : queryWrapper.orderByDesc(value);
            } catch (NoSuchFieldException e) {
                e.printStackTrace();
            }
        }
        if ("0".equals(dataType)) {
            if ((total > 0 && AllIdList.size() > 0) || total == 0) {
                Page<McTaskMsgEntity> page = new Page<>(pagination.getCurrentPage(), pagination.getPageSize());
                IPage<McTaskMsgEntity> userIPage = this.page(page, queryWrapper);
                return pagination.setData(userIPage.getRecords(), userIPage.getTotal());
            } else {
                List<McTaskMsgEntity> list = new ArrayList();
                return pagination.setData(list, list.size());
            }
        } else {
            return this.list(queryWrapper);
        }
    }

    @Override
    public McTaskMsgEntity getInfo(String id) {
        QueryWrapper<McTaskMsgEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(McTaskMsgEntity::getId, id);
        return this.getOne(queryWrapper);
    }

    @Override
    @DSTransactional
    public void create(McTaskMsgCrForm mcTaskMsgCrForm) {
        String mainId = RandomUtil.uuId();
        McTaskMsgEntity entity = JsonUtil.getJsonToBean(mcTaskMsgCrForm, McTaskMsgEntity.class);
        entity.setId(mainId);
        this.save(entity);
    }

    @Override
    @DSTransactional
    public boolean update(String id, McTaskMsgUpForm mcTaskMsgUpForm) {
        McTaskMsgEntity entity = JsonUtil.getJsonToBean(mcTaskMsgUpForm, McTaskMsgEntity.class);
        entity.setId(id);
        return this.updateById(entity);
    }

    @Override
    @DSTransactional
    public void delete(McTaskMsgEntity entity) {
        if (entity != null) {
            this.removeById(entity.getId());
        }
    }


    /**
     * 创建业务消息并下发
     *
     * @param mcTaskMsgCrForm
     */
    @Override
    public void addAndSendMessage(McTaskMsgCrForm mcTaskMsgCrForm) throws Exception {
        // 创建消息
        McTaskMsgEntity entity = this.addTaskMsg(mcTaskMsgCrForm);
        // 下发消息
        this.sendTaskMsgReceive(entity);
    }

    /**
     * 业务消息创建
     * 1.数据验证
     * 2.持久化保存
     *
     * @param mcTaskMsgCrForm
     */
    @Override
    public McTaskMsgEntity addTaskMsg(McTaskMsgCrForm mcTaskMsgCrForm) throws MsgCenterException {

        /**
         * 数据验证
         */
        if (ObjectUtil.isNotEmpty(mcTaskMsgCrForm)) {
            // 业务系统的code 验证
            String taskCode = mcTaskMsgCrForm.getTaskCode();
            if (StringUtils.isEmpty(taskCode)) throw new MsgCenterException("taskCode不能为空");
            // 发送配置 验证
            String sendCode = mcTaskMsgCrForm.getSendCode();
            if (StringUtils.isEmpty(sendCode)) throw new MsgCenterException("sendCode不能为空");

            McMsgSendEntity msgSendEntity = mcMsgSendService.getValidInfo(sendCode);
            // 验证发送配置code有效性
            if (ObjectUtil.isEmpty(msgSendEntity)) {
                throw new MsgCenterException("sendCode不存在或禁用，请确定发送配置编码");
            }

            if (StringUtils.isEmpty(mcTaskMsgCrForm.getBusinessKey())) {
                throw new MsgCenterException("使用模板，businessKey不能为空");
            } else if (!JsonUtil.isJSONString(mcTaskMsgCrForm.getBusinessKey())) {
                throw new MsgCenterException("parameterMap非json格式");
            }
            // 接收人或接收部门 验证
            final String toUserIds = mcTaskMsgCrForm.getToUserIds();
            final String toDeptIds = mcTaskMsgCrForm.getToDeptIds();
            if (StringUtils.isEmpty(toUserIds) && StringUtils.isEmpty(toDeptIds))
                throw new MsgCenterException("无效的接收人");

            mcTaskMsgCrForm.setEnabledMark("1"); // 待发送

            McTaskMsgEntity entity = JsonUtil.getJsonToBean(mcTaskMsgCrForm, McTaskMsgEntity.class);
            entity.setId(RandomUtil.uuId()); // 设置消息内容表主键id
            this.save(entity);
            return entity;
        } else {
            throw new MsgCenterException("请求参数不能为空");
        }
    }

    /**
     * 下发消息
     * <p>
     * 1.根据task_msg的发送配置，解释消息模板及消息内容，解释接收人：下发渠道+接收人=>task_msg_receive接收人记录
     * 接收人：user列表、部门列表或@ALL，@ALL时部门列表不生效
     * 2.根据task_msg_receive数据，渠道消息下发：首次全量下发
     * 根据消息内容主键Id,按渠道类型分别获取受接收人记录进行渠道消息封装下发。
     *
     * @param entity 业务消息
     * @return
     */
    @Override
    @Transactional
    public void sendTaskMsgReceive(McTaskMsgEntity entity) {
        // 发送配置
        String sendCode = entity.getSendCode();
        McMsgSendInfoVO sendInfoVO = mcMsgSendService.getInfoByEnCode(sendCode);
        List<McMsgSendTemplateModel> sendTemplateList = sendInfoVO.getSendTemplateList();
        if (CollectionUtil.isNotEmpty(sendTemplateList)) {
            /**
             * 解释消息模板及消息内容，解释接收人
             */
            this.saveContentAndReceive(entity, sendTemplateList);
            /**
             * 设置消息发送状态
             */
            entity.setEnabledMark(2); // 消息状态，1待发送、2已发送
            this.updateById(entity);

            threadPoolExecutor.execute(() -> {
                try {
                    /**
                     * 根据task_msg_receive数据，渠道消息下发：首次全量下发
                     */
                    mcTaskMsgReceiveService.postSendReceive(entity, sendTemplateList, 0);
                } catch (Exception e) {
                    log.error("消息中心-根据task_msg_receive数据，渠道消息下发失败，异常：" + e.getMessage());
                }
            });
        } else {
            entity.setDescription("发送配置-未配置渠道消息模板");
            this.updateById(entity);
        }
    }

    /**
     * 解释消息模板及消息内容，解释接收人
     *
     * @param taskMsg          原始消息
     * @param sendTemplateList 发送配置 消息模板组合列表
     */
    private void saveContentAndReceive(McTaskMsgEntity taskMsg, List<McMsgSendTemplateModel> sendTemplateList) {
        if (ObjectUtil.isEmpty(taskMsg) || CollectionUtil.isEmpty(sendTemplateList)) {
            return;
        }
        String taskMsgId = taskMsg.getId();
        // 定义接收人set集合
        boolean allUsers = false; // 特殊情况：是否向全部成员发送的。
        Set<String> userSet = new HashSet<>();
        String toUserIds = taskMsg.getToUserIds();
        if (StringUtils.isNotEmpty(toUserIds)) {
            if (ALL_USERS.equals(toUserIds)) {
                // 触发特殊情况：指定为"@ALL"，则向全部成员发送
                allUsers = true;
                // 所有用户
                List<String> listId = userService.getListId();
                userSet.addAll(listId);
            } else {
                // 指定用户
                String[] toUserArr = toUserIds.split(PART_COMMA);
                Collections.addAll(userSet, toUserArr); // 通过Collections将数组转换为Set
            }
        }
        String toDeptIds = taskMsg.getToDeptIds();
        if (!allUsers && StringUtils.isNotEmpty(toDeptIds)) { // 当toUserIds为"@ALL"时忽略本参数
            // 指定部门
            List<String> orgIds = Arrays.asList(toDeptIds.split(PART_COMMA));
            // 根据部门ID批量获取用户列表
            List<UserEntity> userList = userService.getUserByOrgIds(orgIds);
            if (CollectionUtil.isNotEmpty(userList)) {
                final Set<String> deptUserSet = userList.stream().map(UserEntity::getId).collect(Collectors.toSet());
                userSet = SetUtils.union(userSet, deptUserSet);
            }
        }
        List<UserEntity> receiveUsers = userService.getUserByIds(new ArrayList<>(userSet));

        // 不同渠道消息接收人
        Map<String, List<McTaskMsgReceiveEntity>> receiveMap = new HashMap<>();
        // 不同应用消息内容
        Map<String, McTaskMsgContentEntity> contentMap = new HashMap<>();
        /**
         * 根据task_msg的发送配置，解释消息模板及消息内容，解释接收人：下发渠道+接收人=>task_msg_receive接收人记录
         * 接收人：user列表、部门列表或@ALL，@ALL时部门列表不生效
         */
        for (McMsgSendTemplateModel sendTemplate : sendTemplateList) {
            String templateId = sendTemplate.getId(); // 发送配置消息模板
            String channelType = sendTemplate.getMessageType(); // 消息渠道
            String msgTemplateId = sendTemplate.getTemplateId(); // 消息模板
            String msgAccountId = sendTemplate.getAccountConfigId(); // 授权应用配置

            // 授权应用
            McMsgAccountEntity account = mcMsgAccountService.getInfo(sendTemplate.getAccountConfigId());
            // 消息模板
            McMsgTemplateEntity msgTemplate = mcMsgTemplateService.getInfo(sendTemplate.getTemplateId());
            // 消息场景
            // McBusinessInfoVO businessInfoVO = mcBusinessService.getInfoByEnCode(msgTemplate.getMessageSource());
            // 消息渠道
            ChannelTypeEnum typeEnum = getByCode(channelType);
            switch (typeEnum) {
                case SYS_MSG: // 站内消息
                    String sysKey = channelType + msgTemplateId;
                    // 消息内容
                    McTaskMsgContentEntity sysContent;
                    if (!contentMap.containsKey(sysKey)) {
                        sysContent = this.getTaskMsgContent(templateId, channelType, taskMsg, msgTemplate, account);
                        contentMap.put(sysKey, sysContent); // 渠道模板应用消息内容
                    } else {
                        sysContent = contentMap.get(sysKey);
                    }

                    // 消息接收人
                    if (!receiveMap.containsKey(sysKey)) {
                        List<McTaskMsgReceiveEntity> sysReceiveList = new ArrayList<>();
                        // 接收人
                        receiveUsers.stream().forEach(s -> {
                            val receiveEntity = new McTaskMsgReceiveEntity();
                            receiveEntity.setId(RandomUtil.uuId());
                            receiveEntity.setMsgId(taskMsgId);
                            // 发送配置模板id
                            receiveEntity.setTemplateId(templateId);
                            receiveEntity.setMsgTemplateId(msgTemplateId);
                            receiveEntity.setMsgAccountId(msgAccountId);
                            receiveEntity.setContentId(sysContent.getId());
                            // 站内信
                            receiveEntity.setChannelType(channelType);
                            // 下发渠道接收人：base_user主键id
                            receiveEntity.setReceiveUserId(s.getId());
                            // 冗余字段 接收人系统用户表id
                            receiveEntity.setSysUserId(s.getId());
                            receiveEntity.setEnabledMark(0); // 渠道消息下发状态，0待下发、1下/重发成功、2下/重发失败
                            receiveEntity.setIsRead(NOT_READ); // 是否阅读，0否、1是
                            receiveEntity.setDescription(null);
                            sysReceiveList.add(receiveEntity);
                        });
                        receiveMap.put(sysKey, sysReceiveList);
                    }
                    break;
                case MAIL_MSG: // 邮件
                    String mailKey = channelType + msgTemplateId + msgAccountId;
                    // 消息内容
                    McTaskMsgContentEntity mailContent;
                    if (!contentMap.containsKey(mailKey)) {
                        mailContent = this.getTaskMsgContent(templateId, channelType, taskMsg, msgTemplate, account);
                        contentMap.put(mailKey, mailContent); // 渠道模板应用消息内容
                    } else {
                        mailContent = contentMap.get(mailKey);
                    }

                    // 消息接收人
                    if (!receiveMap.containsKey(mailKey)) {
                        List<McTaskMsgReceiveEntity> mailReceiveList = new ArrayList<>();
                        // 接收人
                        receiveUsers.stream().forEach(s -> {
                            val receiveEntity = new McTaskMsgReceiveEntity();
                            receiveEntity.setId(RandomUtil.uuId());
                            receiveEntity.setMsgId(taskMsgId);
                            // 发送配置模板id
                            receiveEntity.setTemplateId(templateId);
                            receiveEntity.setMsgTemplateId(msgTemplateId);
                            receiveEntity.setMsgAccountId(msgAccountId);
                            receiveEntity.setContentId(mailContent.getId());
                            // 电子邮箱
                            receiveEntity.setChannelType(channelType);
                            // 下发渠道接收人：base_user主键email，系统账户邮箱
                            if (StringUtils.isNotEmpty(s.getEmail()) && EmailUtil.isEmail(s.getEmail())) {
                                receiveEntity.setReceiveUserId(s.getRealName() + "<" + s.getEmail() + ">");
                            }
                            // 冗余字段 接收人系统用户表id
                            receiveEntity.setSysUserId(s.getId());
                            receiveEntity.setEnabledMark(0); // 渠道消息下发状态，0待下发、1下/重发成功、2下/重发失败
                            receiveEntity.setIsRead(NOT_READ); // 是否阅读，0否、1是
                            receiveEntity.setDescription(null);
                            mailReceiveList.add(receiveEntity);
                        });
                        receiveMap.put(mailKey, mailReceiveList);
                    }
                    break;
                case SMS_MSG: // 发送短信
                    String smsKey = channelType + msgTemplateId + msgAccountId;
                    // 消息内容
                    McTaskMsgContentEntity smsContent;
                    if (!contentMap.containsKey(smsKey)) {
                        smsContent = this.getTaskMsgContentKey(templateId, channelType, taskMsg, msgTemplate, account);
                        contentMap.put(smsKey, smsContent); // 渠道模板应用消息内容
                    } else {
                        smsContent = contentMap.get(smsKey);
                    }

                    // 消息接收人
                    if (!receiveMap.containsKey(smsKey)) {
                        List<McTaskMsgReceiveEntity> smsReceiveList = new ArrayList<>();
                        // 接收人
                        receiveUsers.stream().forEach(s -> {
                            val receiveEntity = new McTaskMsgReceiveEntity();
                            receiveEntity.setId(RandomUtil.uuId());
                            receiveEntity.setMsgId(taskMsgId);
                            // 发送配置模板id
                            receiveEntity.setTemplateId(templateId);
                            receiveEntity.setMsgTemplateId(msgTemplateId);
                            receiveEntity.setMsgAccountId(msgAccountId);
                            receiveEntity.setContentId(smsContent.getId());
                            // 短信
                            receiveEntity.setChannelType(channelType);
                            // 下发渠道接收人：base_user主键mobilePhone，系统账户手机号
                            receiveEntity.setReceiveUserId(s.getMobilePhone());
                            // 冗余字段 接收人系统用户表id
                            receiveEntity.setSysUserId(s.getId());
                            receiveEntity.setEnabledMark(0); // 渠道消息下发状态，0待下发、1下/重发成功、2下/重发失败
                            receiveEntity.setIsRead(NOT_READ); // 是否阅读，0否、1是
                            receiveEntity.setDescription(null);
                            smsReceiveList.add(receiveEntity);
                        });
                        receiveMap.put(smsKey, smsReceiveList);
                    }
                    break;
                case DING_MSG: // 钉钉
                    String dingContentKey = channelType + msgTemplateId + msgAccountId;
                    String dingReceiveKey = channelType + msgTemplateId;
                    // 消息内容
                    McTaskMsgContentEntity ddContent;
                    if (!contentMap.containsKey(dingContentKey)) {
                        ddContent = this.getTaskMsgContent(templateId, channelType, taskMsg, msgTemplate, account);
                        contentMap.put(dingContentKey, ddContent); // 渠道模板应用消息内容
                    } else {
                        ddContent = contentMap.get(dingContentKey);
                    }

                    String ddCrop = ObjectUtil.isEmpty(account) ? null : account.getEnterpriseId();
                    if (receiveMap.containsKey(dingReceiveKey)) {
                        // 钉钉 钉钉用户关联base_sync_third_info获取
                        List<SynThirdInfoEntity> ddThirdList =
                                synThirdInfoService.getThirdInfoList(THIRD_TYPE_DING, DATA_TYPE_USER, ddCrop, userSet);
                        List<McTaskMsgReceiveEntity> ddReceiveList =
                                packThirdReceive(receiveUsers, ddThirdList, taskMsgId, templateId, channelType,
                                        msgTemplateId, msgAccountId, ddContent.getId(), receiveMap.get(dingReceiveKey));
                        receiveMap.put(dingReceiveKey, ddReceiveList);
                    } else {
                        // 钉钉 钉钉用户关联base_sync_third_info获取
                        List<SynThirdInfoEntity> ddThirdList =
                                synThirdInfoService.getThirdInfoList(THIRD_TYPE_DING, DATA_TYPE_USER, ddCrop, userSet);
                        List<McTaskMsgReceiveEntity> ddReceiveList =
                                packThirdReceive(receiveUsers, ddThirdList, taskMsgId, templateId, channelType,
                                        msgTemplateId, msgAccountId, ddContent.getId(), null);
                        receiveMap.put(dingReceiveKey, ddReceiveList);
                    }
                    break;
                case QY_MSG:
                    String qyContentKey = channelType + msgTemplateId + msgAccountId;
                    String qyReceiveKey = channelType + msgTemplateId;
                    // 消息内容
                    McTaskMsgContentEntity qyContent;
                    if (!contentMap.containsKey(qyContentKey)) {
                        qyContent = this.getTaskMsgContent(templateId, channelType, taskMsg, msgTemplate, account);
                        contentMap.put(qyContentKey, qyContent); // 渠道模板应用消息内容
                    } else {
                        qyContent = contentMap.get(qyContentKey);
                    }

                    // 是否互联企业配置
                    String isLinkedCorp = ObjectUtil.isEmpty(account) ? null : account.getProgramState();
                    String qyCrop = ObjectUtil.isEmpty(account) ? null : account.getEnterpriseId();
                    if (receiveMap.containsKey(qyReceiveKey)) {
                        // 企业微信 企业微信用户关联base_sync_third_info获取
                        List<SynThirdInfoEntity> qyThirdList =
                                synThirdInfoService.getThirdInfoList(THIRD_TYPE_QY, DATA_TYPE_USER, qyCrop, userSet
                                        , Boolean.parseBoolean(isLinkedCorp));
                        List<McTaskMsgReceiveEntity> qyReceiveList =
                                packThirdReceive(receiveUsers, qyThirdList, taskMsgId, templateId, channelType,
                                        msgTemplateId, msgAccountId, qyContent.getId(), receiveMap.get(qyReceiveKey));
                        receiveMap.put(qyReceiveKey, qyReceiveList);
                    } else {
                        // 企业微信 企业微信用户关联base_sync_third_info获取
                        List<SynThirdInfoEntity> qyThirdList =
                                synThirdInfoService.getThirdInfoList(THIRD_TYPE_QY, DATA_TYPE_USER, qyCrop, userSet
                                        , Boolean.parseBoolean(isLinkedCorp));
                        List<McTaskMsgReceiveEntity> qyReceiveList =
                                packThirdReceive(receiveUsers, qyThirdList, taskMsgId, templateId, channelType,
                                        msgTemplateId, msgAccountId, qyContent.getId(), null);
                        receiveMap.put(qyReceiveKey, qyReceiveList);
                    }
                    break;
                case WX_MINIAPP:
                    String maContentKey = channelType + msgTemplateId + msgAccountId;
                    String maReceiveKey = channelType + msgTemplateId;
                    // 消息内容
                    McTaskMsgContentEntity maContent;
                    if (!contentMap.containsKey(maContentKey)) {
                        maContent = this.getTaskMsgContentKey(templateId, channelType, taskMsg, msgTemplate, account);
                        contentMap.put(maContentKey, maContent); // 渠道模板应用消息内容
                    } else {
                        maContent = contentMap.get(maContentKey);
                    }

                    String maCrop = ObjectUtil.isEmpty(account) ? null : account.getAppId();
                    if (receiveMap.containsKey(maReceiveKey)) {
                        // 微信小程序用户关联base_sync_third_info获取
                        List<SynThirdInfoEntity> maThirdList =
                                synThirdInfoService.getThirdInfoList(THIRD_TYPE_MINIAPP, DATA_TYPE_USER, maCrop,
                                        userSet);
                        List<McTaskMsgReceiveEntity> maReceiveList =
                                packThirdReceive(receiveUsers, maThirdList, taskMsgId, templateId, channelType,
                                        msgTemplateId, msgAccountId, maContent.getId(), receiveMap.get(maReceiveKey));
                        receiveMap.put(maReceiveKey, maReceiveList);
                    } else {
                        // 微信小程序用户关联base_sync_third_info获取
                        List<SynThirdInfoEntity> maThirdList =
                                synThirdInfoService.getThirdInfoList(THIRD_TYPE_MINIAPP, DATA_TYPE_USER, maCrop,
                                        userSet);
                        List<McTaskMsgReceiveEntity> maReceiveList =
                                packThirdReceive(receiveUsers, maThirdList, taskMsgId, templateId, channelType,
                                        msgTemplateId, msgAccountId, maContent.getId(), null);
                        receiveMap.put(maReceiveKey, maReceiveList);
                    }
                    break;
                case WX_MP:
                    String mpContentKey = channelType + msgTemplateId + msgAccountId;
                    String mpReceiveKey = channelType + msgTemplateId;
                    // 消息内容
                    McTaskMsgContentEntity mpContent;
                    if (!contentMap.containsKey(mpContentKey)) {
                        mpContent = this.getTaskMsgContentKey(templateId, channelType, taskMsg, msgTemplate, account);
                        contentMap.put(mpContentKey, mpContent); // 渠道模板应用消息内容
                    } else {
                        mpContent = contentMap.get(mpContentKey);
                    }

                    String mpCrop = ObjectUtil.isEmpty(account) ? null : account.getAppId();
                    if (receiveMap.containsKey(mpReceiveKey)) {
                        // 微信公众号用户关联base_sync_third_info获取
                        List<SynThirdInfoEntity> mpThirdList =
                                synThirdInfoService.getThirdInfoList(THIRD_TYPE_MP, DATA_TYPE_USER, mpCrop, userSet);
                        List<McTaskMsgReceiveEntity> mpReceiveList =
                                packThirdReceive(receiveUsers, mpThirdList, taskMsgId, templateId, channelType,
                                        msgTemplateId, msgAccountId, mpContent.getId(), receiveMap.get(mpReceiveKey));
                        receiveMap.put(mpReceiveKey, mpReceiveList);
                    } else {
                        // 微信公众号用户关联base_sync_third_info获取
                        List<SynThirdInfoEntity> mpThirdList =
                                synThirdInfoService.getThirdInfoList(THIRD_TYPE_MP, DATA_TYPE_USER, mpCrop, userSet);
                        List<McTaskMsgReceiveEntity> mpReceiveList =
                                packThirdReceive(receiveUsers, mpThirdList, taskMsgId, templateId, channelType,
                                        msgTemplateId, msgAccountId, mpContent.getId(), null);
                        receiveMap.put(mpReceiveKey, mpReceiveList);
                    }
                    break;
                case WEBHOOK:
                    String webhookKey = channelType + msgTemplateId + msgAccountId;
                    // 消息内容
                    McTaskMsgContentEntity webhookContent;
                    if (!contentMap.containsKey(webhookKey)) {
                        webhookContent = this.getTaskMsgContent(templateId, channelType, taskMsg, msgTemplate, account);
                        contentMap.put(webhookKey, webhookContent); // 渠道模板应用消息内容
                    } else {
                        webhookContent = contentMap.get(webhookKey);
                    }

                    // 消息接收人
                    if (!receiveMap.containsKey(webhookKey)) {
                        // 接收者: 应用授权webhook配置信息,mc_msg_account获取。
                        McTaskMsgReceiveEntity webhookReceiveEntity = new McTaskMsgReceiveEntity();
                        webhookReceiveEntity.setId(RandomUtil.uuId());
                        webhookReceiveEntity.setMsgId(taskMsgId);
                        // 发送配置模板id
                        webhookReceiveEntity.setTemplateId(templateId);
                        webhookReceiveEntity.setMsgTemplateId(msgTemplateId);
                        webhookReceiveEntity.setMsgAccountId(msgAccountId);
                        webhookReceiveEntity.setContentId(webhookContent.getId());
                        // webhook
                        webhookReceiveEntity.setChannelType(channelType);
                        // 下发渠道接收者：发送配置渠道消息模板关联应用账号sendTemplate.getAccountConfigId()
                        webhookReceiveEntity.setReceiveUserId(sendTemplate.getAccountConfigId());
                        // 冗余字段 接收人系统用户表无账号，设为空
                        webhookReceiveEntity.setSysUserId(null);
                        webhookReceiveEntity.setEnabledMark(0); // 渠道消息下发状态，0待下发、1下/重发成功、2下/重发失败
                        webhookReceiveEntity.setIsRead(NOT_READ); // 是否阅读，0否、1是
                        webhookReceiveEntity.setDescription(null);
                        receiveMap.put(webhookKey, new ArrayList<>(Arrays.asList(webhookReceiveEntity)));
                    }
                    break;
                default:
                    break;
            }
        }
        // 接收人不为空
        List<McTaskMsgContentEntity> contentEntityList = new ArrayList<>();
        for (McTaskMsgContentEntity value : contentMap.values()) {
            contentEntityList.add(value);
        }
        List<McTaskMsgReceiveEntity> receiveEntityList = new ArrayList<>();
        for (List<McTaskMsgReceiveEntity> value : receiveMap.values()) {
            receiveEntityList.addAll(value);
        }
        // 设置 渠道消息未发送的尝试重发最大限制次数
        Integer autoAgainMaxNum = ObjectUtil.isEmpty(taskMsg.getAutoAgainMaxNum()) ? AUTO_AGAIN_MAX_NUM :
                taskMsg.getAutoAgainMaxNum();
        receiveEntityList.stream().forEach(item -> item.setAutoAgainMaxNum(autoAgainMaxNum));

        // 消息内容持久化
        mcTaskMsgContentService.saveBatch(contentEntityList);
        // 消息接收人持久化
        mcTaskMsgReceiveService.saveBatch(receiveEntityList);
    }

    private McTaskMsgContentEntity getTaskMsgContent(String templateId, String channelType, McTaskMsgEntity taskMsg,
                                                     McMsgTemplateEntity msgTemplate, McMsgAccountEntity account) {
        String taskMsgId = taskMsg.getId();
        // 消息场景
        McBusinessInfoVO businessInfoVO = mcBusinessService.getInfoByEnCode(msgTemplate.getMessageSource());
        // 业务消息关键词
        String businessKey = taskMsg.getBusinessKey();
        Map<String, Object> businessKeyMap = JsonUtil.stringToMap(businessKey);

        // 标题（带有占位符的）
        String title = msgTemplate.getTitle();
        // 内容（带有占位符的）
        String content = msgTemplate.getContent();
        // 摘要
        String description = null;
        // 消息链接
        String linkUrl = null;
        // 图片链接
        String picUrl = null;
        // 替换参数
        if (StringUtils.isNotEmpty(businessKeyMap)) {
            // 标题
            if (StringUtils.isNotEmpty(title)) {
                StringSubstitutor strSubstitutor = new StringSubstitutor(businessKeyMap, "{", "}");
                title = strSubstitutor.replace(title);
            } else {
                title = String.valueOf(businessKeyMap.get("title"));
            }
            // 内容
            if (StringUtils.isNotEmpty(content)) {
                StringSubstitutor strSubstitutor = new StringSubstitutor(businessKeyMap, "{", "}");
                content = strSubstitutor.replace(content);
            } else {
                content = String.valueOf(businessKeyMap.get("content"));
            }
            // 摘要
            if (businessKeyMap.containsKey("desc") && StringUtils.isNotEmpty(businessKeyMap.get("desc").toString())) {
                description = businessKeyMap.get("desc").toString();
            } else {
                description = StringUtils.isEmpty(content) ? null : StringUtils.substring(content, 0, 200);
            }
            // 消息链接
            linkUrl = this.packLinkUrl(channelType, businessKeyMap, account, taskMsgId);
            // 图片链接
            if (businessKeyMap.containsKey("picUrl") && ObjectUtil.isNotEmpty(businessKeyMap.get("picUrl"))) {
                picUrl = String.valueOf(businessKeyMap.get("picUrl"));
            }
        }

        // 封装业务消息内容实体
        McTaskMsgContentEntity taskMsgContentEntity = new McTaskMsgContentEntity();
        // 标题
        if (StringUtils.isNotEmpty(title)) {
//            title = StringUtils.stringFilter(title);
            title = SensitiveWordUtil.sensitiveFilter(title, true);
            title = StringUtils.substring(title, 0, 50);
            taskMsgContentEntity.setTitle(title);
        }
        // 内容
        if (StringUtils.isNotEmpty(content)) {
            content = SensitiveWordUtil.sensitiveFilter(content, true);
            taskMsgContentEntity.setContent(content);
        }
        // 摘要
        if (StringUtils.isNotEmpty(description)) {
            description = SensitiveWordUtil.sensitiveFilter(description, true);
            taskMsgContentEntity.setDescription(description);
        }
        // 消息链接
        taskMsgContentEntity.setLinkUrl(linkUrl);
        // 图片链接
        if (StringUtils.isNotEmpty(picUrl)) {
            taskMsgContentEntity.setPicUrl(picUrl);
        }

        taskMsgContentEntity.setId(RandomUtil.uuId());
        taskMsgContentEntity.setMsgId(taskMsgId);
        taskMsgContentEntity.setChannelType(channelType);
        taskMsgContentEntity.setTemplateId(templateId);
        taskMsgContentEntity.setMsgTemplateId(msgTemplate.getId());
        taskMsgContentEntity.setMsgAccountId(ObjectUtil.isEmpty(account) ? null : account.getId());
        taskMsgContentEntity.setContentType(businessInfoVO.getCategory());
        return taskMsgContentEntity;
    }

    private McTaskMsgContentEntity getTaskMsgContentKey(String templateId, String channelType, McTaskMsgEntity taskMsg,
                                                        McMsgTemplateEntity msgTemplate, McMsgAccountEntity account) {
        String taskMsgId = taskMsg.getId();
        // 消息场景
        McBusinessInfoVO businessInfoVO = mcBusinessService.getInfoByEnCode(msgTemplate.getMessageSource());
        // 业务消息关键词
        String businessKey = taskMsg.getBusinessKey();
        Map<String, Object> businessKeyMap = JsonUtil.stringToMap(businessKey);

        McTaskMsgContentEntity taskMsgContentEntity = new McTaskMsgContentEntity();
        // 模板变量
        List<McMsgTemplateFieldEntity> fieldList = mcMsgTemplateService.getMcMsgTemplateFieldList(msgTemplate.getId());
        if (CollectionUtil.isNotEmpty(fieldList) && StringUtils.isNotEmpty(businessKeyMap)) {
            HashMap<Object, Object> fieldMap = new HashMap<>();
            for (McMsgTemplateFieldEntity field : fieldList) {
                fieldMap.put(field.getThirdField(), businessKeyMap.get(field.getKeywordCode()));
                if (Integer.valueOf(field.getIsTitle()) == 1) {
                    taskMsgContentEntity.setTitle(businessKeyMap.get(field.getKeywordCode()).toString());
                }
            }
            // 内容
            String content = JSONObject.toJSONString(fieldMap);
            if (StringUtils.isNotEmpty(content)) {
                content = SensitiveWordUtil.sensitiveFilter(content, true);
                taskMsgContentEntity.setContent(content);
            }

            // 消息链接
            taskMsgContentEntity.setLinkUrl(this.packLinkUrl(channelType, businessKeyMap, account, taskMsgId));
            // 图片链接
            if (businessKeyMap.containsKey("picUrl") && ObjectUtil.isNotEmpty(businessKeyMap.get("picUrl"))) {
                taskMsgContentEntity.setLinkUrl(String.valueOf(businessKeyMap.get("picUrl")));
            }
        }
        taskMsgContentEntity.setId(RandomUtil.uuId());
        taskMsgContentEntity.setMsgId(taskMsgId);
        taskMsgContentEntity.setChannelType(channelType);
        taskMsgContentEntity.setTemplateId(templateId);
        taskMsgContentEntity.setMsgTemplateId(msgTemplate.getId());
        taskMsgContentEntity.setMsgAccountId(ObjectUtil.isEmpty(account) ? null : account.getId());
        taskMsgContentEntity.setContentType(businessInfoVO.getCategory());
        return taskMsgContentEntity;
    }

    private String packLinkUrl(String channelType, Map<String, Object> businessKeyMap,
                               McMsgAccountEntity account, String taskMsgId) {
        // 消息渠道
        ChannelTypeEnum typeEnum = getByCode(channelType);
        Object linkUrl = null;
        switch (typeEnum) {
            case SYS_MSG: // 站内消息
                if (businessKeyMap.containsKey("linkUrl")) linkUrl = businessKeyMap.get("linkUrl");
            case MAIL_MSG: // 邮件
                if (businessKeyMap.containsKey("linkUrl2")) linkUrl = businessKeyMap.get("linkUrl2");
            case SMS_MSG: // 发送短信
                if (businessKeyMap.containsKey("linkUrl3")) linkUrl = businessKeyMap.get("linkUrl3");
                break;
            case DING_MSG: // 钉钉
                if (businessKeyMap.containsKey("linkUrl4")) linkUrl = businessKeyMap.get("linkUrl4");
                break;
            case QY_MSG:
                if (businessKeyMap.containsKey("linkUrl5")) linkUrl = businessKeyMap.get("linkUrl5");
                break;
            case WX_MINIAPP:
                if (businessKeyMap.containsKey("linkUrl6")) linkUrl = businessKeyMap.get("linkUrl6");
                break;
            case WX_MP:
                if (businessKeyMap.containsKey("linkUrl7")) linkUrl = businessKeyMap.get("linkUrl7");
                break;
            case WEBHOOK:
                if (businessKeyMap.containsKey("linkUrl8")) linkUrl = businessKeyMap.get("linkUrl8");
                break;
            default:

                break;
        }

        if (ObjectUtil.isEmpty(linkUrl) && ObjectUtil.isNotEmpty(account)) {
            linkUrl = account.getAddress();
        }

        if (ObjectUtil.isNotEmpty(linkUrl)) {
            Map<String, Object> map = new HashMap<>();
            map.put(MSG_ID, taskMsgId);
            return UrlUtils.addParameter(String.valueOf(linkUrl), map);
        }
        return null;
    }

    /**
     * @param sysList
     * @param thirdList
     * @param channelType
     * @param contentId   消息内容Id
     * @return
     */
    private List<McTaskMsgReceiveEntity> packThirdReceive(List<UserEntity> sysList,
                                                          List<SynThirdInfoEntity> thirdList, String taskMsgId,
                                                          String templateId, String channelType,
                                                          String msgTemplateId, String msgAccountId, String contentId,
                                                          List<McTaskMsgReceiveEntity> receiveList) {
        if (CollectionUtil.isEmpty(receiveList)) {
            List<McTaskMsgReceiveEntity> receiveEntities = new ArrayList<>();
            for (UserEntity user : sysList) {
                McTaskMsgReceiveEntity thirdReceive = new McTaskMsgReceiveEntity();
                thirdReceive.setMsgId(taskMsgId);
                // 发送配置模板id
                thirdReceive.setTemplateId(templateId);
                // 消息模板Id
                thirdReceive.setMsgTemplateId(msgTemplateId);
                // 授权应用Id
                thirdReceive.setMsgAccountId(msgAccountId);
                // 消息内容Id
                thirdReceive.setContentId(contentId);
                // 消息接收人渠道
                thirdReceive.setChannelType(channelType);
                // 下发渠道接收人：base_user主键id
                thirdReceive.setSysUserId(user.getId());
                thirdReceive.setIsRead(NOT_READ); // 是否阅读，0否、1是
                List<SynThirdInfoEntity> tempList =
                        thirdList.stream().filter(s -> user.getId().equals(s.getSystemObjectId())).collect(Collectors.toList());
                if (CollectionUtil.isNotEmpty(tempList)) {
                    tempList.stream().forEach(s -> {
                        McTaskMsgReceiveEntity thirdReceiveNew = new McTaskMsgReceiveEntity();
                        BeanUtils.copyProperties(thirdReceive, thirdReceiveNew);
                        // 下发渠道接收人：base_sync_third_info主键id
                        thirdReceiveNew.setReceiveUserId(s.getId());
                        thirdReceiveNew.setEnabledMark(0); // 渠道消息下发状态，0待下发、1下/重发成功、2下/重发失败
                        thirdReceiveNew.setDescription("创建，设置下发状态为‘待下发’");
                        thirdReceiveNew.setId(RandomUtil.uuId());
                        receiveEntities.add(thirdReceiveNew);
                    });
                } else {
                    thirdReceive.setEnabledMark(0); // 渠道消息下发状态，0待下发、1下/重发成功、2下/重发失败
                    thirdReceive.setDescription("创建，设置下发状态为‘待下发’");
                    thirdReceive.setId(RandomUtil.uuId());
                    receiveEntities.add(thirdReceive);
                }
            }
            return receiveEntities;
        } else {
            for (SynThirdInfoEntity thirdInfo : thirdList) {
                List<McTaskMsgReceiveEntity> rTempList =
                        receiveList.stream().filter(r -> thirdInfo.getSystemObjectId().equals(r.getSysUserId())).collect(Collectors.toList());
                if (CollectionUtil.isNotEmpty(rTempList)) {
                    McTaskMsgReceiveEntity mcTaskMsgReceiveEntity = rTempList.get(0);
                    if (StringUtils.isEmpty(mcTaskMsgReceiveEntity.getReceiveUserId())) {
                        // 下发渠道接收人：base_sync_third_info主键id
                        mcTaskMsgReceiveEntity.setReceiveUserId(thirdInfo.getId());
                        // 发送配置模板id
                        mcTaskMsgReceiveEntity.setTemplateId(templateId);
                        // 消息模板Id
                        mcTaskMsgReceiveEntity.setMsgTemplateId(msgTemplateId);
                        // 授权应用Id
                        mcTaskMsgReceiveEntity.setMsgAccountId(msgAccountId);
                        // 消息内容Id
                        mcTaskMsgReceiveEntity.setContentId(contentId);
                    } else {
                        McTaskMsgReceiveEntity thirdReceiveNew = new McTaskMsgReceiveEntity();
                        BeanUtils.copyProperties(mcTaskMsgReceiveEntity, thirdReceiveNew);
                        thirdReceiveNew.setId(RandomUtil.uuId());
                        // 下发渠道接收人：base_sync_third_info主键id
                        thirdReceiveNew.setReceiveUserId(thirdInfo.getId());
                        // 发送配置模板id
                        thirdReceiveNew.setTemplateId(templateId);
                        // 消息模板Id
                        thirdReceiveNew.setMsgTemplateId(msgTemplateId);
                        // 授权应用Id
                        thirdReceiveNew.setMsgAccountId(msgAccountId);
                        // 消息内容Id
                        thirdReceiveNew.setContentId(contentId);
                        receiveList.add(thirdReceiveNew);
                    }
                }
            }
            return receiveList;
        }
    }

    @Override
    public List<Map<String, Object>> selectGroupByEnabledMark(McTaskMsgPagination pagination) {
        QueryWrapper<McTaskMsgEntity> queryWrapper = new QueryWrapper<>();
        // 此处根据menuId 开启数据权限
        // 创建时间
        if (StringUtils.isNotEmpty(pagination.getCreatorTime())) {
            List<String> SendTimeList = pagination.getCreatorTime();
            Long fir = Long.valueOf(SendTimeList.get(0));
            Long sec = Long.valueOf(SendTimeList.get(1));

            queryWrapper.lambda().ge(McTaskMsgEntity::getCreatorTime, new Date(fir))
                    .le(McTaskMsgEntity::getCreatorTime, DateUtil.stringToDate(DateUtil.daFormatYmd(sec) + " " +
                            "23:59:59"));
        }
        return mcTaskMsgMapper.selectGroupByEnabledMark(queryWrapper);
    }
}