package com.bringspring.system.permission.service.strategy;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.bringspring.system.permission.model.authorize.ConditionModel;
import org.apache.poi.ss.formula.functions.T;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * 具体策略类:
 * 定义：实现策略接口的具体算法类。
 * 角色：提供策略接口中定义的算法的具体实现。
 * 职责：根据条件模型ConditionModel中的条件信息，生成QueryWrapper的查询条件。
 * 适用性：该策略类适用于datetime类型的条件。
 * <p>
 * datetime类型的条件策略
 */
@Component
public class DateConditionStrategy implements ConditionStrategy {

    @Autowired
    private ConditionStrategyFactory conditionStrategyFactory;

    @Autowired
    private ContentPickerDate contentPickerDate;

    @Override
    public void generateCondition(QueryWrapper<T> tw, String logic, ConditionModel.ConditionItemModel fieldItem) {
        String fieldSymbol = fieldItem.getOp();
        String fieldContent = fieldItem.getValue();
        String valueLabel = fieldItem.getValueLabel();
        String dateFormat = fieldItem.getDateFormat();
        /**
         * 字段类型为datetime
         * 字段内容：datePicker日期选择 （dateFormat：yyyy、yyyy-MM、yyyy-MM-dd、yyyy-MM-dd HH:mm、yyyy-MM-dd HH:mm:ss）
         * 符号：==等于、<>不等于、>大于、<小于、>=大于等于、<=小于等于、null为空、notNull不为空、between介于
         * 处理方式：
         * 1. 日期格式化
         *    a. 符号为==、<>、>、<、>=、<=时，字段内容直接格式化
         *    b. 符号为null、notNull时，字段内容不做处理
         *    c. 符号为between时，字段内容应为两个日期，特殊处理
         *    d. 其他符号，不做处理
         * 2. 根据符号设置tw的查询条件
         * 3. 调用tw.apply()方法添加查询条件
         */

        Object value;
        if (fieldContent.startsWith("@")) {
            value = conditionStrategyFactory.getFieldContentDate(fieldContent, fieldSymbol, dateFormat);
        } else {
            value = contentPickerDate.convertFieldContent(fieldContent, fieldSymbol, dateFormat);
        }

        // 设置tw的查询条件
        SymbolCommon.compare(logic, fieldItem.getField(), fieldSymbol, value, tw);
    }

    @Override
    public Integer key() {
        return 1;
    }

}
