package com.bringspring.system.permission.service.strategy;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.bringspring.system.permission.model.authorize.ConditionModel;
import org.apache.poi.ss.formula.functions.T;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

/**
 * 具体策略类:
 * 定义：实现策略接口的具体算法类。
 * 角色：提供策略接口中定义的算法的具体实现。
 * 职责：根据条件模型ConditionModel中的条件信息，生成QueryWrapper的查询条件。
 * <p>
 * int、bigint、double类型的条件策略
 */
@Component
public class NumberConditionStrategy implements ConditionStrategy {

    @Autowired
    private ConditionStrategyFactory conditionStrategyFactory;

    @Autowired
    private ContentInputNumber contentInputNumber;

    @Override
    public void generateCondition(QueryWrapper<T> tw, String logic, ConditionModel.ConditionItemModel fieldItem) {
        String fieldSymbol = fieldItem.getOp();
        String fieldContent = fieldItem.getValue();
        /**
         * 字段类型为int、bigint、double时，处理方式如下：
         * 字段内容：inputNumber数字输入
         * 符号：==等于、<>不等于、>大于、<小于、>=大于等于、<=小于等于、null为空、notNull不为空、between介于
         * 处理方式：
         * 1. 字段内容格式化
         *    a. 符号为==、<>、>、<、>=、<=时，字段内容直接用
         *    b. 符号为null、notNull时，字段内容不做处理
         *    c. 符号为between时，字段内容应为数字范围，格式为[数字1,数字2]，用JSON.parseArray()方法解析，特殊处理
         *    d. 其他符号，不做处理
         * 2. 根据符号设置tw的查询条件
         * 3. 调用tw.apply()方法添加查询条件
         */

        Object value;
        if (fieldContent.startsWith("@")) {
            value = conditionStrategyFactory.getFieldContentNumber(fieldContent, fieldSymbol);
        } else {
            value = contentInputNumber.convertFieldContent(fieldContent, fieldSymbol);
        }

        // 设置tw的查询条件
        SymbolCommon.compare(logic, fieldItem.getField(), fieldSymbol, value, tw);
    }

    @Override
    public Integer key() {
        return 3;
    }

}
