package com.bringspring.system.base.controller;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.Page;
import com.bringspring.common.base.UserInfo;
import com.bringspring.common.base.vo.ListVO;
import com.bringspring.common.util.*;
import com.bringspring.system.base.model.cachemanage.CacheManageInfoVO;
import com.bringspring.system.base.model.cachemanage.CacheManageListVO;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 缓存管理
 *
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2017年9月27日 上午9:18
 */
@Api(tags = "缓存管理", value = "CacheManage")
@RestController
@RequestMapping("/api/system/CacheManage")
public class CacheManageController {

    @Autowired
    private RedisUtil redisUtil;
    @Autowired
    private CacheUtil cacheUtil;
    @Autowired
    private UserProvider userProvider;

    /**
     * 获取缓存列表
     *
     * @param page 分页参数
     * @return ignore
     */
    @ApiOperation("获取缓存列表")
    @GetMapping
    public ActionResult<ListVO<CacheManageListVO>> getList(Page page) {
        String tenantId = userProvider.get().getTenantId();
        List<CacheManageListVO> list = new ArrayList<>();
        Set<String> data = cacheUtil.getAllKeys();
        for (String key : data) {
            try {
                int cacheSize = 0;
                Object value = cacheUtil.query(key);
                if (ObjectUtil.isNotEmpty(value)) {
                    cacheSize = String.valueOf(value).getBytes().length;
                }
                if (!StringUtils.isEmpty(tenantId) && key.contains(tenantId)) {
                    CacheManageListVO model = new CacheManageListVO();
                    model.setName(key);
                    model.setCacheSize(cacheSize);
                    model.setOverdueTime(cacheUtil.getLiveTime(key));
                    list.add(model);
                } else if (StringUtils.isEmpty(tenantId)) {
                    CacheManageListVO model = new CacheManageListVO();
                    model.setName(key);
                    model.setCacheSize(cacheSize);
                    model.setOverdueTime(cacheUtil.getLiveTime(key));
                    list.add(model);
                }
            }catch (Exception e){
                e.printStackTrace();
            }
        }
        list = list.stream().sorted(Comparator.comparing(CacheManageListVO::getOverdueTime)).collect(Collectors.toList());
        if (StringUtils.isNotEmpty(page.getKeyword())) {
            list = list.stream().filter(t -> t.getName().contains(page.getKeyword())).collect(Collectors.toList());
        }
        ListVO<CacheManageListVO> vo = new ListVO<>();
        vo.setList(list);
        return ActionResult.success(vo);
    }

    /**
     * 获取缓存信息
     *
     * @param name 主键值
     * @return ignore
     */
    @ApiOperation("获取缓存信息")
    @GetMapping("/{name}")
    public ActionResult info(@PathVariable("name") String name) {
        name = XSSEscape.escape(name);
        String json = String.valueOf(cacheUtil.query(name));
        CacheManageInfoVO vo = new CacheManageInfoVO();
        vo.setName(name);
        vo.setValue(json);
        return ActionResult.success(vo);
    }

    /**
     * 清空所有缓存
     *
     * @return ignore
     */
    @ApiOperation("清空所有缓存")
    @PostMapping("/Actions/ClearAll")
    public ActionResult clearAll() {
        String tenantId = userProvider.get().getTenantId();
        if ("".equals(tenantId)) {
            Set<String> keys = cacheUtil.getAllKeys();
            for (String key : keys) {
                cacheUtil.remove(key);
            }
        } else {
            Set<String> data = cacheUtil.getAllKeys();
            for (String key : data) {
                if (key.contains(tenantId)) {
                    cacheUtil.remove(key);
                }
            }
        }
        return ActionResult.success("清理成功");
    }

    /**
     * 清空单个缓存
     *
     * @param name 主键值
     * @return ignore
     */
    @ApiOperation("清空单个缓存")
    @DeleteMapping("/{name}")
    public ActionResult clear(@PathVariable("name") String name) {
        cacheUtil.remove(name);
        return ActionResult.success("清空成功");
    }


    /**
     * 批量删除缓存
     *
     * @return ignore
     */
    @ApiOperation("批量删除缓存")
    @PutMapping("/Actions/batchDelete/{names}")
    public ActionResult batchDelete(@PathVariable String[] names) {
        // 当前登录用户信息
        UserInfo userInfo = userProvider.get();
        if (ObjectUtil.isEmpty(userInfo)) {
            return ActionResult.fail("登录信息已失效，请重新登录");
        }
        List<String> nameList = Arrays.asList(names);
        if (CollectionUtil.isNotEmpty(nameList)) {
            nameList.stream().forEach(name -> cacheUtil.remove(name));
        }
        return ActionResult.success("批量删除成功");
    }

}
