package com.bringspring.system.msgCenter.service;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.common.util.StringUtils;
import com.bringspring.system.external.bean.WeComModel;
import com.bringspring.system.external.config.mutil.WxCpConfiguration;
import com.bringspring.system.message.entity.SynThirdInfoEntity;
import com.bringspring.system.message.service.SynThirdInfoService;
import com.bringspring.system.msgCenter.entity.McTaskMsgContentEntity;
import com.bringspring.system.msgCenter.entity.McTaskMsgReceiveEntity;
import com.bringspring.system.msgCenter.util.BlacklistUtil;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import me.chanjar.weixin.common.api.WxConsts;
import me.chanjar.weixin.common.error.WxErrorException;
import me.chanjar.weixin.cp.api.WxCpService;
import me.chanjar.weixin.cp.bean.article.NewArticle;
import me.chanjar.weixin.cp.bean.message.WxCpLinkedCorpMessage;
import me.chanjar.weixin.cp.bean.message.WxCpLinkedCorpMessageSendResult;
import me.chanjar.weixin.cp.bean.message.WxCpMessage;
import me.chanjar.weixin.cp.bean.message.WxCpMessageSendResult;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.stream.Collectors;

import static com.bringspring.system.msgCenter.constant.CommonConsts.PART_VERTICAL;
import static com.bringspring.system.msgCenter.enums.ContentTypeEnum.*;
import static com.bringspring.system.msgCenter.enums.ErrCodeEnum.SEND_EXCEPTION;

/**
 * 消息中心-企业微信消息下发工具类
 * 使用WxJava（微信Java开发工具包）
 */
@Component
@Slf4j
public class MsgToWxCpService {

    @Autowired
    WxCpConfiguration wxCpConfiguration;
    @Autowired
    private SynThirdInfoService synThirdInfoService;
    @Autowired
    private McTaskMsgReceiveService mcTaskMsgReceiveService;

    /**
     * 企业微信消息下发
     * 发送应用消息
     *
     * @param isAgain        是否重发
     * @param taskMsgContent 消息
     * @param receiveListS   接收人source
     */
    public void sendMessage(boolean isAgain,
                            McTaskMsgContentEntity taskMsgContent, List<McTaskMsgReceiveEntity> receiveListS) {
        if (CollectionUtil.isEmpty(receiveListS)) {
            log.error("~·~·~·~调用了 企业微信消息下发，接收人列表为空~·~·~·~");
        }
        List<McTaskMsgReceiveEntity> receiveList = JsonUtil.getJsonToList(receiveListS, McTaskMsgReceiveEntity.class);

        /**
         * 判断是否全员发送
         */
        Boolean toUserIdAll = false; // true为全部成员，false为非全部成员

        // 黑名单成员
        List<McTaskMsgReceiveEntity> receiveBlacklist = BlacklistUtil.receiveListFilter(receiveList);
        if (CollectionUtil.isNotEmpty(receiveList)) {
            // 无third绑定数据
            List<McTaskMsgReceiveEntity> thirdNullList =
                    receiveList.stream().filter(r -> StringUtils.isEmpty(r.getReceiveUserId())).collect(Collectors.toList());
            // 有third绑定数据
            List<McTaskMsgReceiveEntity> thirdNotNullList =
                    receiveList.stream().filter(r -> StringUtils.isNotEmpty(r.getReceiveUserId())).collect(Collectors.toList());
            List<String> thirdInfoIds =
                    thirdNotNullList.stream().map(r -> r.getReceiveUserId()).collect(Collectors.toList());
            if (CollectionUtil.isNotEmpty(thirdInfoIds)) {
                List<SynThirdInfoEntity> thirdInfoList = synThirdInfoService.listByIds(thirdInfoIds);
                // third无userid
                List<SynThirdInfoEntity> thirdUseridNullList =
                        thirdInfoList.stream().filter(r -> StringUtils.isEmpty(r.getThirdObjectId())).collect(Collectors.toList());
                List<McTaskMsgReceiveEntity> useridNullList =
                        thirdNotNullList.stream().filter(r -> thirdUseridNullList.stream().map(e ->
                                e.getId()).collect(Collectors.toList()).contains(r.getReceiveUserId())).collect(Collectors.toList());
                // third有userid
                List<SynThirdInfoEntity> thirdUseridNotNullList =
                        thirdInfoList.stream().filter(r -> StringUtils.isNotEmpty(r.getThirdObjectId())).collect(Collectors.toList());
                List<McTaskMsgReceiveEntity> useridNotNullList =
                        thirdNotNullList.stream().filter(r -> thirdUseridNotNullList.stream().map(e ->
                                e.getId()).collect(Collectors.toList()).contains(r.getReceiveUserId())).collect(Collectors.toList());

                Map<String, WxCpMessageSendResult> resultMap = new HashMap<>();
                // 多例企业配置
                if (!toUserIdAll) {
                    /**
                     * 指定接收消息的成员，成员ID列表（多个接收者用‘|’分隔，最多支持1000个）
                     */
                    Map<String, List<String>> corpUserMap = new HashMap<>();
                    if (CollectionUtil.isNotEmpty(thirdUseridNotNullList)) {
                        for (SynThirdInfoEntity stiEntity : thirdUseridNotNullList) {
                            if (StringUtils.isNotEmpty(stiEntity.getThirdObjectId())) {
                                final String corpid = stiEntity.getCropId();
                                final String userid = stiEntity.getThirdObjectId();
                                if (corpUserMap.containsKey(corpid)) {
                                    corpUserMap.get(corpid).add(userid);
                                } else {
                                    List<String> useridList = new ArrayList<>();
                                    useridList.add(userid);
                                    corpUserMap.put(corpid, useridList);
                                }
                            }
                        }
                    }

                    if (CollectionUtil.isNotEmpty(corpUserMap)) {
                        /**
                         * 调用企业API 发送消息
                         */
                        for (Map.Entry<String, List<String>> entry : corpUserMap.entrySet()) {
                            final WeComModel weComModel = wxCpConfiguration.getWeComModel(entry.getKey());
                            final String corpId = weComModel.getQyhCorpId();
                            final Integer agentId = Integer.valueOf(weComModel.getQyhAgentId());
                            final String touser = StringUtils.join(entry.getValue(), PART_VERTICAL);

                            WxCpMessageSendResult result;
                            try {
                                result = this.postMessageSend(taskMsgContent, corpId, agentId, touser);
                            } catch (WxErrorException e) {
                                e.printStackTrace();
                                result = new WxCpMessageSendResult();
                                result.setErrCode(SEND_EXCEPTION.getCode());
                                result.setInvalidUser(touser);
                                result.setErrMsg(e.getMessage());
                            }
                            System.out.println("~·~发送消息：" + corpId + result);
                            resultMap.put(corpId, result);// 不合法的userid
                        }
                    }
                } else {
                    /**
                     * 特殊情况：指定为"@ALL"，则向企业应用的全部成员发送
                     */
                    final Map<String, WeComModel> weComModels = wxCpConfiguration.getWeComModels();
                    for (WeComModel weComModel : weComModels.values()) {
                        final String corpId = weComModel.getQyhCorpId();
                        final Integer agentId = Integer.valueOf(weComModel.getQyhAgentId());
                        /**
                         * 调用企业API 发送消息
                         */
                        WxCpMessageSendResult result = null;
                        try {
                            result = this.postMessageSend(taskMsgContent, corpId, agentId, "@all");
                        } catch (WxErrorException e) {
                            e.printStackTrace();
                            result = new WxCpMessageSendResult();
                            result.setErrCode(SEND_EXCEPTION.getCode());
                            List<String> userIds =
                                    receiveList.stream().map(r -> r.getReceiveUserId()).collect(Collectors.toList());
                            String touser = StringUtils.join(userIds, PART_VERTICAL);
                            result.setInvalidUser(touser);
                            result.setErrMsg(e.getMessage());
                        }
                        System.out.println("~·~发送消息@all：" + corpId + result);
                        resultMap.put(corpId, result);// 不合法的userid
                    }
                }

                /**
                 * 处理发送结果
                 */
                if (CollectionUtil.isNotEmpty(resultMap)) {
                    this.wxCpSendMessageResult(isAgain, taskMsgContent, useridNotNullList, thirdUseridNotNullList,
                            resultMap);
                }

                if (CollectionUtil.isNotEmpty(useridNullList)) {
                    String errMsg = (isAgain ? "重发失败：" : "下发失败：") + "third无userid";
                    mcTaskMsgReceiveService.updateByList(isAgain, useridNullList, 2, errMsg);
                }
            }
            if (CollectionUtil.isNotEmpty(thirdNullList)) {
                String errMsg = (isAgain ? "重发失败：" : "下发失败：") + "无third绑定数据";
                mcTaskMsgReceiveService.updateByList(isAgain, thirdNullList, 2, errMsg);
            }
        }
        if (CollectionUtil.isNotEmpty(receiveBlacklist)) {
            // 黑名单
            mcTaskMsgReceiveService.updateBlacklist(receiveBlacklist);
        }
    }

    /**
     * 执行 企业微信应用消息post下发
     *
     * @param taskMsgContent 消息内容
     * @param corpId         企业id
     * @param agentId        应用id
     * @param touser         指定接收消息的成员，成员ID列表（多个接收者用‘|’分隔，最多支持1000个）。
     *                       特殊情况：指定为"@all"，则向该企业应用的全部成员发送
     */
    public WxCpMessageSendResult postMessageSend(McTaskMsgContentEntity taskMsgContent, String corpId, Integer agentId,
                                                 String touser) throws WxErrorException {
        if (ObjectUtil.isEmpty(taskMsgContent) || StringUtils.isEmpty(corpId) || StringUtils.isEmpty(touser)) {
            log.error("~·~·~·~执行 企业微信应用消息post下发，content 或 corpId企业id 或 touser为空~·~·~·~");
        }

        final WxCpService corpService = wxCpConfiguration.getCpService(corpId, agentId);
        final String contentType =
                StringUtils.isNotEmpty(taskMsgContent.getContentType()) ? taskMsgContent.getContentType() :
                        TEXT_MSG.getCode();
        final String title = taskMsgContent.getTitle();
        final String content = taskMsgContent.getContent();
        final String linkUrl = taskMsgContent.getLinkUrl();

        if (TEXT_MSG.getCode().equals(contentType)) {
            // 文本消息
            WxCpMessage message = WxCpMessage
                    .TEXT()
                    .toUser(touser)
                    .content(content)
                    .build();
            return corpService.getMessageService().send(message);
        }

        if (LINK_MSG.getCode().equals(contentType)) {
            // 链接消息
            WxCpMessage message = WxCpMessage
                    .TEXTCARD()
                    .toUser(touser)
                    .description(content)
                    .url(linkUrl)
                    .title(title)
                    .build();
            return corpService.getMessageService().send(message);
        }

        if (IMG_TEXT_MSG.getCode().equals(contentType)) {
            // 图文消息
            NewArticle article1 = new NewArticle();
            article1.setUrl(linkUrl);
            article1.setPicUrl(taskMsgContent.getPicUrl());
            article1.setDescription(content);
            article1.setTitle(title);
            WxCpMessage message = WxCpMessage.NEWS().toUser(touser).addArticle(article1).build();

            return corpService.getMessageService().send(message);
        }
        return null;
    }


    /**
     * 执行 企业微信应用消息post下发 结果处理
     *
     * @param isAgain        是否重发
     * @param taskMsgContent 消息内容表信息
     * @param receiveList    企业微信当前处理接收人集合
     * @param resultMap      企业微信消息发送结果集合
     */
    public void wxCpSendMessageResult(boolean isAgain, McTaskMsgContentEntity taskMsgContent,
                                      List<McTaskMsgReceiveEntity> receiveList,
                                      List<SynThirdInfoEntity> thirdInfoList,
                                      Map<String, WxCpMessageSendResult> resultMap) {
        /**
         * 将‘待下发’临时处理为‘下发成功’
         */
        if (CollectionUtil.isNotEmpty(receiveList)) {
            String errMsg = isAgain ? "重发成功：" : "下发成功：";
            mcTaskMsgReceiveService.updateByList(isAgain, receiveList, 1, errMsg);
        }

        /**
         * 处理企业微信下发接口返回的 不合法的userid
         */
        if (CollectionUtil.isNotEmpty(resultMap)) {
            for (Map.Entry<String, WxCpMessageSendResult> entry : resultMap.entrySet()) {
                WxCpMessageSendResult result = entry.getValue();
                if (ObjectUtil.isNotEmpty(result) && CollectionUtil.isNotEmpty(result.getInvalidUserList())) {
                    final List<String> invalidUserList = result.getInvalidUserList();// 不合法的userid
                    List<SynThirdInfoEntity> invalidThirdList =
                            thirdInfoList.stream().filter(r -> invalidUserList.contains(r.getThirdObjectId())).collect(Collectors.toList());
                    List<McTaskMsgReceiveEntity> list =
                            receiveList.stream().filter(r -> invalidThirdList.stream().map(e ->
                                    e.getId()).collect(Collectors.toList()).contains(r.getReceiveUserId())).collect(Collectors.toList());
                    if (CollectionUtil.isNotEmpty(list)) {
                        String errMsg = isAgain ? "重发失败：" : "下发失败：";
                        if (SEND_EXCEPTION.getCode() == result.getErrCode().intValue()) {
                            errMsg += errMsg + result.getErrMsg();
                        } else {
                            errMsg += errMsg + "不合法的userid";
                        }
                        mcTaskMsgReceiveService.updateByList(isAgain, list, 2, errMsg);
                    }
                }
            }
        }

    }


    /**
     * 互联企业微信消息下发
     * 发送应用消息
     *
     * @param isAgain        是否重发
     * @param taskMsgContent 消息
     * @param receiveListS   接收人
     */
    public void sendLinkedCorpMessage(boolean isAgain, McTaskMsgContentEntity taskMsgContent,
                                      List<McTaskMsgReceiveEntity> receiveListS) {
        if (ObjectUtil.isEmpty(taskMsgContent) || CollectionUtil.isEmpty(receiveListS)) {
            log.error("~·~·~·~调用了 互联企业微信消息下发，但message content为空 或 接收人列表为空~·~·~·~");
        }
        List<McTaskMsgReceiveEntity> receiveList = JsonUtil.getJsonToList(receiveListS, McTaskMsgReceiveEntity.class);

        /**
         * 判断是否全员发送
         */
        Boolean toUserIdAll = false; // true为全部成员，false为非全部成员
        // 黑名单成员
        List<McTaskMsgReceiveEntity> receiveBlacklist = BlacklistUtil.receiveListFilter(receiveList);

        if (CollectionUtil.isNotEmpty(receiveList)) {
            // 无third绑定数据
            List<McTaskMsgReceiveEntity> thirdNullList =
                    receiveList.stream().filter(r -> StringUtils.isEmpty(r.getReceiveUserId())).collect(Collectors.toList());
            // 有third绑定数据
            List<McTaskMsgReceiveEntity> thirdNotNullList =
                    receiveList.stream().filter(r -> StringUtils.isNotEmpty(r.getReceiveUserId())).collect(Collectors.toList());
            List<String> thirdInfoIds =
                    thirdNotNullList.stream().map(r -> r.getReceiveUserId()).collect(Collectors.toList());
            if (CollectionUtil.isNotEmpty(thirdInfoIds)) {
                List<SynThirdInfoEntity> thirdInfoList = synThirdInfoService.listByIds(thirdInfoIds);
                // third无userid
                List<SynThirdInfoEntity> thirdUseridNullList =
                        thirdInfoList.stream().filter(r -> StringUtils.isEmpty(r.getThirdObjectId())).collect(Collectors.toList());
                List<McTaskMsgReceiveEntity> useridNullList =
                        thirdNotNullList.stream().filter(r -> thirdUseridNullList.stream().map(e ->
                                e.getId()).collect(Collectors.toList()).contains(r.getReceiveUserId())).collect(Collectors.toList());
                // third有userid
                List<SynThirdInfoEntity> thirdUseridNotNullList =
                        thirdInfoList.stream().filter(r -> StringUtils.isNotEmpty(r.getThirdObjectId())).collect(Collectors.toList());
                List<McTaskMsgReceiveEntity> useridNotNullList =
                        thirdNotNullList.stream().filter(r -> thirdUseridNotNullList.stream().map(e ->
                                e.getId()).collect(Collectors.toList()).contains(r.getReceiveUserId())).collect(Collectors.toList());
                // 互联企业配置
                final Map<String, WeComModel> weComModels = wxCpConfiguration.getWeComModels();
                final Optional<WeComModel> firstValue = weComModels.values().stream().findFirst();
                if (firstValue.isPresent() && firstValue.get().getIsLinkedCorp()) {
                    final String qyhCorpId = firstValue.get().getQyhCorpId();
                    final Integer qyhAgentId = Integer.valueOf(firstValue.get().getQyhAgentId());
                    WxCpLinkedCorpMessageSendResult result;
                    if (!toUserIdAll) {
                        /**
                         * 调用企业API 发送消息
                         */
                        String[] toUsers =
                                thirdUseridNotNullList.stream().map(t -> t.getThirdObjectId()).collect(Collectors.toList()).toArray(new String[]{});

                        try {
                            result = this.postLinkedCorpMessageSend(taskMsgContent, qyhCorpId, qyhAgentId, toUserIdAll,
                                    toUsers);
                        } catch (WxErrorException e) {
                            e.printStackTrace();
                            result = new WxCpLinkedCorpMessageSendResult();
                            result.setErrcode(Long.valueOf(SEND_EXCEPTION.getCode()));
                            result.setInvalidUser(toUsers);
                            result.setErrmsg(e.getMessage());
                        }
                    } else {
                        /**
                         * 特殊情况：指定为"@ALL"，则向企业应用的全部成员发送
                         * 调用企业API 发送消息
                         */
                        try {
                            result = this.postLinkedCorpMessageSend(taskMsgContent, qyhCorpId, qyhAgentId, toUserIdAll,
                                    null);
                        } catch (WxErrorException e) {
                            e.printStackTrace();
                            result = new WxCpLinkedCorpMessageSendResult();
                            result.setErrcode(Long.valueOf(SEND_EXCEPTION.getCode()));
                            String[] toUsers =
                                    receiveList.stream().map(r -> r.getReceiveUserId()).collect(Collectors.toList()).toArray(new String[]{});
                            result.setInvalidUser(toUsers);
                            result.setErrmsg(e.getMessage());
                        }
                    }
                    /**
                     * 处理发送结果
                     */
                    if (ObjectUtil.isNotEmpty(result)) {
                        this.wxCpLinkedCorpSendMessageResult(isAgain, taskMsgContent, useridNotNullList,
                                thirdUseridNotNullList, result);
                    }
                }

                if (CollectionUtil.isNotEmpty(useridNullList)) {
                    String errMsg = (isAgain ? "重发失败：" : "下发失败：") + "third无userid";
                    mcTaskMsgReceiveService.updateByList(isAgain, useridNullList, 2, errMsg);
                }
            }
            if (CollectionUtil.isNotEmpty(thirdNullList)) {
                String errMsg = (isAgain ? "重发失败：" : "下发失败：") + "无third绑定数据";
                mcTaskMsgReceiveService.updateByList(isAgain, thirdNullList, 2, errMsg);
            }
        }
        if (CollectionUtil.isNotEmpty(receiveBlacklist)) {
            // 黑名单
            mcTaskMsgReceiveService.updateBlacklist(receiveBlacklist);
        }
    }

    /**
     * 执行 互联企业微信应用消息post下发
     *
     * @param taskMsgContent 消息内容
     * @param corpId         企业id
     * @param agentId        应用id
     * @param isToAll        是否发送给应用可见范围内的所有人（包括互联企业的成员）
     * @param toUsers        成员ID列表（消息接收者，最多支持1000个）。
     *                       每个元素的格式为： corpid/userid，其中，corpid为该互联成员所属的企业，userid为该互联成员所属企业中的帐号。
     *                       如果是本企业的成员，则直接传userid即可
     */
    public WxCpLinkedCorpMessageSendResult postLinkedCorpMessageSend(McTaskMsgContentEntity taskMsgContent,
                                                                     String corpId,
                                                                     Integer agentId, Boolean isToAll,
                                                                     String[] toUsers) throws WxErrorException {
        if (ObjectUtil.isEmpty(taskMsgContent) || StringUtils.isEmpty(corpId)) {
            log.error("~·~·~·~执行 互联企业微信应用消息post下发，但message info 或 corpId企业id 为空~·~·~·~");
        }

        final WxCpService corpService = wxCpConfiguration.getCpService(corpId, agentId);
        final String contentType = taskMsgContent.getContentType();
        final String title = taskMsgContent.getTitle();
        final String content = taskMsgContent.getContent();
        final String linkUrl = taskMsgContent.getLinkUrl();

        if (TEXT_MSG.getCode().equals(contentType)) {
            // 文本消息
            WxCpLinkedCorpMessage message = WxCpLinkedCorpMessage.builder()
                    .msgType(WxConsts.KefuMsgType.TEXT)
                    .toUsers(toUsers)
                    .isToAll(isToAll)
                    .isSafe(false)
                    .content(content)
                    .build();

            return corpService.getMessageService().sendLinkedCorpMessage(message);

        }

        if (LINK_MSG.getCode().equals(contentType)) {
            // 链接消息
            WxCpLinkedCorpMessage message = WxCpLinkedCorpMessage.builder()
                    .msgType(WxConsts.KefuMsgType.TEXTCARD)
                    .toUsers(toUsers)
                    .isToAll(isToAll)
                    .title(title)
                    .description(content)
                    .url(linkUrl)
                    //.btnTxt("更多")
                    .build();

            return corpService.getMessageService().sendLinkedCorpMessage(message);
        }

        if (IMG_TEXT_MSG.getCode().equals(contentType)) {
            // 图文消息
            WxCpLinkedCorpMessage message = WxCpLinkedCorpMessage.builder()
                    .msgType(WxConsts.KefuMsgType.NEWS)
                    .toUsers(toUsers)
                    .isToAll(isToAll)
                    .articles(Lists.newArrayList(NewArticle.builder()
                            .title(title)
                            .description(content)
                            .url(linkUrl)
                            .picUrl(taskMsgContent.getPicUrl())
                            //.btnText("更多")
                            .build()))
                    .build();

            return corpService.getMessageService().sendLinkedCorpMessage(message);
        }
        return null;
    }


    /**
     * 执行 互联企业微信应用消息post下发 结果处理
     *
     * @param isAgain        是否重发
     * @param taskMsgContent 消息内容表信息
     * @param receiveList    互联企业微信当前处理接收人集合
     * @param thirdInfoList  互联企业微信当前处理接收人集合
     * @param result         互联企业微信消息发送结果
     */
    public void wxCpLinkedCorpSendMessageResult(boolean isAgain, McTaskMsgContentEntity taskMsgContent,
                                                List<McTaskMsgReceiveEntity> receiveList,
                                                List<SynThirdInfoEntity> thirdInfoList,
                                                WxCpLinkedCorpMessageSendResult result) {
        /**
         * 将‘待下发’临时处理为‘下发成功’
         */
        if (CollectionUtil.isNotEmpty(receiveList)) {
            String errMsg = isAgain ? "重发成功：" : "下发成功：";
            mcTaskMsgReceiveService.updateByList(isAgain, receiveList, 1, errMsg);
        }

        /**
         * 处理企业微信下发接口返回的 不合法的userid
         */
        if (ObjectUtil.isNotEmpty(result)) {
            String[] invalidUser = result.getInvalidUser();// 不合法的userid

            if (ObjectUtil.isNotEmpty(invalidUser) && invalidUser.length != 0) {
                List<String> invalidUserList = Arrays.asList(invalidUser);
                List<SynThirdInfoEntity> invalidThirdList =
                        thirdInfoList.stream().filter(r -> invalidUserList.contains(r.getThirdObjectId())).collect(Collectors.toList());

                List<McTaskMsgReceiveEntity> list =
                        receiveList.stream().filter(r -> invalidThirdList.stream().map(e ->
                                e.getId()).collect(Collectors.toList()).contains(r.getReceiveUserId())).collect(Collectors.toList());

                if (CollectionUtil.isNotEmpty(list)) {
                    String errMsg = isAgain ? "重发失败：" : "下发失败：";
                    if (SEND_EXCEPTION.getCode() == result.getErrcode().intValue()) {
                        errMsg += errMsg + result.getErrmsg();
                    } else {
                        errMsg += errMsg + "不合法的userid";
                    }
                    mcTaskMsgReceiveService.updateByList(isAgain, list, 2, errMsg);
                }
            }
        }
    }

}
