package com.bringspring.system.msgCenter.service;

import cn.hutool.core.collection.CollectionUtil;
import com.alibaba.fastjson.JSONObject;
import com.bringspring.common.base.UserInfo;
import com.bringspring.common.model.OnlineUserModel;
import com.bringspring.common.model.OnlineUserProvider;
import com.bringspring.common.util.UserProvider;
import com.bringspring.system.external.config.mutil.WxCpConfiguration;
import com.bringspring.system.msgCenter.entity.McTaskMsgContentEntity;
import com.bringspring.system.msgCenter.entity.McTaskMsgReceiveEntity;
import com.bringspring.system.msgCenter.enums.ChannelTypeEnum;
import com.bringspring.system.msgCenter.model.mcMsgSendTemplate.McMsgSendTemplateModel;
import com.bringspring.system.msgCenter.util.BlacklistUtil;
import lombok.extern.slf4j.Slf4j;
import me.chanjar.weixin.common.error.WxErrorException;
import org.apache.commons.collections4.ListUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Date;
import java.util.List;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.stream.Collectors;

import static com.bringspring.system.msgCenter.enums.ChannelTypeEnum.getByCode;

/**
 * 消息发送配置表
 * 版本： V1.0.0
 * 版权： 荣科科技股份有限公司
 * 作者： RKKJ开发平台组
 * 日期： 2024-06-12
 */
@Component
@Slf4j
public class SendMessageService {

    @Autowired
    private UserProvider userProvider;
    @Autowired
    private WxCpConfiguration wxCpConfiguration;
    @Autowired
    private McTaskMsgReceiveService mcTaskMsgReceiveService;
    @Autowired
    private MsgToWxCpService msgToWxCpService;
    @Autowired
    private MsgToWxMiniappService msgToWxMiniappService;
    @Autowired
    private MsgToEmailService msgToEmailService;
    @Autowired
    private MsgToSmsService msgToSmsService;
    @Autowired
    private MsgToDingService msgToDingService;
    @Autowired
    private MsgToWebhookService msgToWebhookService;
    @Autowired
    private MsgToWxMpService msgToWxMpService;

    /**
     * 渠道消息发送
     *
     * @param isAgain        是否重发
     * @param sendTemplate
     * @param taskMsgContent 消息内容表信息
     * @param receiveList    接收人记录
     * @throws WxErrorException
     */
    public void sendMessageByChannel(boolean isAgain,
                                     McMsgSendTemplateModel sendTemplate,
                                     McTaskMsgContentEntity taskMsgContent,
                                     List<McTaskMsgReceiveEntity> receiveList) {
        String channelType = sendTemplate.getMessageType(); // 消息渠道
        if (CollectionUtil.isNotEmpty(receiveList)) {
            // 按1000个每组进行分割
            List<List<McTaskMsgReceiveEntity>> subReceives = ListUtils.partition(receiveList, 1000);
            // 渠道类型
            ChannelTypeEnum typeEnum = getByCode(channelType);
            switch (typeEnum) {
                case SYS_MSG:
                    // 站内信
                    // 黑名单成员。
                    List<McTaskMsgReceiveEntity> receiveBlacklist = BlacklistUtil.receiveListFilter(receiveList);
                    if (CollectionUtil.isNotEmpty(receiveBlacklist)) {
                        mcTaskMsgReceiveService.updateBlacklist(receiveBlacklist);
                    }

                    if (CollectionUtil.isNotEmpty(receiveList)) {
                        UserInfo userInfo = userProvider.get();
                        // 直接修改发送状态为‘下发成功’。
                        receiveList.stream().forEach(s -> {
                            s.setEnabledMark(1); // 渠道消息下发状态，0待下发、1下/重发成功、2下/重发失败
                            s.setSendTime(new Date());
                            s.setSendUserId(userInfo.getUserId());
                            s.setDescription("");
                        });
                        mcTaskMsgReceiveService.updateBatchById(receiveList);

                        // ====================== jsbos基础平台 站内信 WebSocket消息推送 START===============================
                        // 消息推送 - PC端，这段实现依赖jsbos基础平台架构，若消息中心实现迁移其他基础平台，这段需要考量实现。
                        // 遍历接收者列表，发送websocket消息
                        List<String> toUserIds = receiveList.stream().map(
                                i -> i.getSysUserId()
                        ).collect(Collectors.toList());
                        for (int i = 0; i < toUserIds.size(); i++) {
                            CopyOnWriteArraySet<OnlineUserModel> onlineUserList =
                                    OnlineUserProvider.getOnlineUserList();
                            for (OnlineUserModel item : onlineUserList) {
                                if (toUserIds.get(i).equals(item.getUserId()) && userInfo.getTenantId().equals(item.getTenantId())) {
                                    JSONObject map = new JSONObject();
                                    map.put("method", "messagePush");
                                    map.put("unreadNoticeCount", 1);
                                    map.put("messageType", 2);
                                    map.put("userId", userInfo.getTenantId());
                                    map.put("toUserId", toUserIds);
                                    map.put("title", taskMsgContent.getTitle());
                                    synchronized (map) {
                                        item.getWebSocket().getAsyncRemote().sendText(map.toJSONString());
                                    }
                                }
                            }
                        }
                        // ====================== jsbos基础平台 站内信 WebSocket消息推送 END===============================
                    }
                    break;
                case MAIL_MSG:
                    // 邮件
                    msgToEmailService.sendEmail(isAgain, sendTemplate, taskMsgContent, receiveList);
                    break;
                case SMS_MSG:
                    // 发送短信
                    // 对分批list的具体业务处理 process(list)
                    for (List subReceive : subReceives) {
                        msgToSmsService.sendSms(isAgain, sendTemplate, taskMsgContent, subReceive);
                    }
                    break;
                case DING_MSG:
                    // 钉钉
                    for (List subReceive : subReceives) {
                        // 按100个每组进行分割，接收者的userid列表，最大用户列表长度100。
                        List<List<McTaskMsgReceiveEntity>> dingReceives = ListUtils.partition(subReceive, 100);
                        for (List dingReceive : dingReceives) {
                            msgToDingService.sendMessage(isAgain, taskMsgContent, dingReceive);
                        }
                    }
                    break;
                case QY_MSG:
                    // 是否互联企业配置
                    Boolean isLinkedCorp = wxCpConfiguration.getIsLinkedCorp();

                    // 对分批list的具体业务处理 process(list)
                    for (List subReceive : subReceives) {
                        // 企业微信 企业微信用户关联base_sync_third_info获取
                        if (isLinkedCorp) {
                            /**
                             * 调用工具类，对接企业微信接口
                             */
                            msgToWxCpService.sendLinkedCorpMessage(isAgain, taskMsgContent, subReceive);
                        } else {
                            msgToWxCpService.sendMessage(isAgain, taskMsgContent, subReceive);
                        }
                    }
                    break;
                case WX_MINIAPP:
                    // 微信小程序
                    msgToWxMiniappService.sendMessage(isAgain, sendTemplate, taskMsgContent, receiveList);
                    break;
                case WX_MP:
                    // 微信公众号
                    msgToWxMpService.sendMessage(isAgain, sendTemplate, taskMsgContent, receiveList);
                    break;
                case WEBHOOK:
                    // 钉钉或企业微信的webhook
                    msgToWebhookService.sendWebhook(isAgain, taskMsgContent, receiveList);
                    break;
                default:
                    break;
            }
        }
    }

}