package com.bringspring.system.base.controller;

import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.vo.ListVO;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.exception.DataException;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.system.base.entity.ModuleDataAuthorizeSchemeEntity;
import com.bringspring.system.base.model.moduledataauthorizescheme.DataAuthorizeSchemeCrForm;
import com.bringspring.system.base.model.moduledataauthorizescheme.DataAuthorizeSchemeInfoVO;
import com.bringspring.system.base.model.moduledataauthorizescheme.DataAuthorizeSchemeListVO;
import com.bringspring.system.base.model.moduledataauthorizescheme.DataAuthorizeSchemeUpForm;
import com.bringspring.system.base.service.ModuleDataAuthorizeSchemeService;
import com.bringspring.system.base.util.JsonUtilEx;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;

/**
 * 数据权限方案
 *
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2017年9月27日 上午9:18
 */
@Api(tags = "数据权限方案", value = "ModuleDataAuthorizeScheme")
@RestController
@RequestMapping("/api/system/ModuleDataAuthorizeScheme")
public class ModuleDataAuthorizeSchemeController {

    @Autowired
    private ModuleDataAuthorizeSchemeService schemeService;

    /**
     * 列表
     *
     * @param moduleId 功能主键
     * @return ignore
     */
    @ApiOperation("方案列表")
    @GetMapping("/{moduleId}/List")
    public ActionResult<ListVO<DataAuthorizeSchemeListVO>> list(@PathVariable("moduleId") String moduleId) {
        List<ModuleDataAuthorizeSchemeEntity> data = schemeService.getList(moduleId);
        List<DataAuthorizeSchemeListVO> list = JsonUtil.getJsonToList(data, DataAuthorizeSchemeListVO.class);
        ListVO<DataAuthorizeSchemeListVO> vo = new ListVO<>();
        vo.setList(list);
        return ActionResult.success(vo);
    }

    /**
     * 信息
     *
     * @param id 主键值
     * @return ignore
     * @throws DataException ignore
     */
    @ApiOperation("获取方案信息")
    @GetMapping("/{id}")
    public ActionResult<DataAuthorizeSchemeInfoVO> info(@PathVariable("id") String id) throws DataException {
        ModuleDataAuthorizeSchemeEntity entity = schemeService.getInfo(id);
        DataAuthorizeSchemeInfoVO vo = JsonUtilEx.getJsonToBeanEx(entity, DataAuthorizeSchemeInfoVO.class);
        return ActionResult.success(vo);
    }

    /**
     * 新建
     *
     * @param dataAuthorizeSchemeCrForm 实体对象
     * @return ignore
     */
    @ApiOperation("新建方案")
    @PostMapping
    public ActionResult create(@RequestBody @Valid DataAuthorizeSchemeCrForm dataAuthorizeSchemeCrForm) {
        ModuleDataAuthorizeSchemeEntity entity = JsonUtil.getJsonToBean(dataAuthorizeSchemeCrForm,
                ModuleDataAuthorizeSchemeEntity.class);
        // 判断encode是否重复
        if (schemeService.isExistByEnCode(entity.getId(), entity.getEnCode(), entity.getModuleId())) {
            return ActionResult.fail("该方案已新增，不可重复添加");
        }
        // 判断fullName是否重复
        if (schemeService.isExistByFullName(entity.getId(), entity.getFullName(), entity.getModuleId())) {
            return ActionResult.fail("已存在相同名称");
        }
        schemeService.create(entity);
        return ActionResult.success(MsgCode.SU001.get());
    }

    /**
     * 更新
     *
     * @param id                        主键值
     * @param dataAuthorizeSchemeUpForm 实体对象
     * @return ignore
     */
    @ApiOperation("更新方案")
    @PutMapping("/{id}")
    public ActionResult update(@PathVariable("id") String id,
                               @RequestBody @Valid DataAuthorizeSchemeUpForm dataAuthorizeSchemeUpForm) {
        ModuleDataAuthorizeSchemeEntity entity = JsonUtil.getJsonToBean(dataAuthorizeSchemeUpForm,
                ModuleDataAuthorizeSchemeEntity.class);
        // 全部数据标识 不能修改
        if ("1".equals(String.valueOf(entity.getAllData()))) {
            return ActionResult.fail("修改失败，该方案不允许编辑");
        }
        // 判断fullName是否重复
        if (schemeService.isExistByFullName(id, entity.getFullName(), entity.getModuleId())) {
            return ActionResult.fail("已存在相同名称");
        }
        // 判断encode是否重复
        if (schemeService.isExistByEnCode(id, entity.getEnCode(), entity.getModuleId())) {
            return ActionResult.fail("已存在相同编码");
        }
        boolean flag = schemeService.update(id, entity);
        if (!flag) {
            return ActionResult.success(MsgCode.FA002.get());
        }
        return ActionResult.success(MsgCode.SU004.get());
    }

    /**
     * 删除
     *
     * @param id 主键值
     * @return ignore
     */
    @ApiOperation("删除方案")
    @DeleteMapping("/{id}")
    public ActionResult delete(@PathVariable("id") String id) {
        ModuleDataAuthorizeSchemeEntity entity = schemeService.getInfo(id);
        if (entity != null) {
            schemeService.delete(entity);
            return ActionResult.success(MsgCode.SU003.get());
        }
        return ActionResult.fail(MsgCode.FA003.get());
    }

}
