package com.bringspring.system.msgcenter.util;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.date.TimeInterval;
import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONException;
import com.bringspring.common.util.DateUtil;
import com.bringspring.common.util.StringUtils;
import com.bringspring.system.base.entity.SysConfigEntity;
import com.bringspring.system.base.exception.BaseException;
import com.bringspring.system.base.service.SysConfigService;
import com.bringspring.system.msgcenter.entity.McTaskMsgReceiveEntity;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.concurrent.Executor;
import java.util.stream.Collectors;

import static com.bringspring.system.msgcenter.constant.Constants.CacheKeys;

/**
 * 消息中心-黑名单工具类
 */
@Component
@Slf4j
public class BlacklistUtil implements ApplicationRunner {

    @Autowired
    private SysConfigService sysConfigService;
    /**
     * 取出线程池
     */
    @Autowired
    private Executor threadPoolExecutor;

    private static List<String> blacklist = new ArrayList<>();

    @Override
    public void run(ApplicationArguments args) {
        Set<String> blacklistConfig = new HashSet<>();

        try {
            /**
             * 读取配置黑名单用户
             */
            SysConfigEntity mcBlacklist = sysConfigService.getConfigByKeyName(CacheKeys.BLACKLIST);
            if (ObjectUtil.isNotEmpty(mcBlacklist)) {
                String keyValue = mcBlacklist.getKeyValue();
                if (StringUtils.isNotEmpty(keyValue)) {
                    JSONArray objects = JSONArray.parseArray(keyValue);
                    objects.stream().forEach(s ->
                            blacklistConfig.add(s.toString().trim())
                    );
                }
            }

        } catch (BaseException | JSONException e) {
            e.printStackTrace();
        }

        this.init(blacklistConfig, true);
        log.info("消息中心-初始化黑名单用户完毕, 共" + blacklistConfig.size() + "个用户");
    }

    /**
     * 初始黑名单成员列表
     *
     * @param isAsync         是否异步初始化
     * @param blacklistConfig 黑名单成员列表
     */
    public void init(final Collection<String> blacklistConfig, boolean isAsync) {
        if (isAsync) {
            threadPoolExecutor.execute(() -> {
                init(blacklistConfig);
            });
        } else {
            init(blacklistConfig);
        }
    }

    /**
     * 初始化黑名单成员列表
     *
     * @param blacklistConfig 黑名单成员列表
     */
    public void init(Collection<String> blacklistConfig) {
        blacklist.clear();
        blacklist.addAll(blacklistConfig);
    }

    /**
     * 处理过滤接收人列表
     *
     * @param receiveList 接收人列表
     * @return 过滤处理后的黑名单接收人列表
     */
    public static List<McTaskMsgReceiveEntity> receiveListFilter(List<McTaskMsgReceiveEntity> receiveList) {
        if (CollectionUtil.isEmpty(receiveList)) {
            return receiveList;
        }
        TimeInterval timer = DateUtil.timer();
        List<McTaskMsgReceiveEntity> receiveBlacklist =
                receiveList.stream().filter(r -> blacklist.contains(r.getSysUserId())).collect(Collectors.toList());
        receiveList.removeIf(r -> receiveBlacklist.contains(r));
        log.info("过滤黑名单, 耗时: " + timer.intervalMs() + "ms");
        return receiveBlacklist;
    }

    /**
     * 处理过滤接收人列表
     *
     * @param receiveList 接收人列表
     * @return 过滤处理后的黑名单接收人列表
     */
    public static Map<Boolean, List<McTaskMsgReceiveEntity>> receiveListPartitioned(List<McTaskMsgReceiveEntity> receiveList) {
        if (CollectionUtil.isEmpty(receiveList)) {
            return null;
        }
        TimeInterval timer = DateUtil.timer();
        Map<Boolean, List<McTaskMsgReceiveEntity>> partitioned = receiveList.stream()
                .collect(Collectors.partitioningBy(r -> blacklist.contains(r.getSysUserId())));

        log.info("过滤黑名单, 耗时: " + timer.intervalMs() + "ms");
        return partitioned;
    }
}
