package com.bringspring.system.permission.controller;

import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.Pagination;
import com.bringspring.common.base.vo.ListVO;
import com.bringspring.common.base.vo.PaginationVO;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.common.util.enums.DictionaryDataEnum;
import com.bringspring.common.util.treeutil.SumTree;
import com.bringspring.common.util.treeutil.newtreeutil.TreeDotUtils;
import com.bringspring.system.base.entity.DictionaryDataEntity;
import com.bringspring.system.base.service.DictionaryDataService;
import com.bringspring.system.permission.entity.GroupEntity;
import com.bringspring.system.permission.model.usergroup.*;
import com.bringspring.system.permission.service.GroupService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.ArrayList;
import java.util.List;

/**
 * 分组管理控制器
 *
 * @author ：RKKJ开发平台组
 * @version: V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date ：2022/3/10 17:57
 */
@RestController
@Api(tags = "分组管理", value = "UserGroupController")
@RequestMapping("/api/permission/Group")
public class GroupController {

    @Autowired
    private GroupService userGroupService;
    @Autowired
    private DictionaryDataService dictionaryDataService;

    /**
     * 获取分组管理列表
     *
     * @param pagination
     * @return
     */
    @ApiOperation("获取分组管理列表")
    @GetMapping
    public ActionResult list(Pagination pagination) {
        List<GroupEntity> list = userGroupService.getList(pagination);
        List<GroupPaginationVO> jsonToList = JsonUtil.getJsonToList(list, GroupPaginationVO.class);
        // 通过数据字典获取类型
        List<DictionaryDataEntity> dictionaryDataEntities =
                dictionaryDataService.getDicList(DictionaryDataEnum.PERMISSION_GROUP.getDictionaryTypeId());
        for (GroupPaginationVO userGroupPaginationVO : jsonToList) {
            DictionaryDataEntity dictionaryDataEntity =
                    dictionaryDataEntities.stream().filter(t -> t.getId().equals(userGroupPaginationVO.getType())).findFirst().orElse(null);
            userGroupPaginationVO.setType(dictionaryDataEntity != null ? dictionaryDataEntity.getFullName() :
                    userGroupPaginationVO.getId());
        }
        PaginationVO paginationVO = JsonUtil.getJsonToBean(pagination, PaginationVO.class);
        return ActionResult.page(jsonToList, paginationVO);
    }

    /**
     * 获取分组管理下拉框
     *
     * @return
     */
    @ApiOperation("获取分组管理下拉框")
    @GetMapping("/Selector")
    public ActionResult<List<GroupSelectorVO>> selector() {
        List<GroupTreeModel> tree = new ArrayList<>();
        List<GroupEntity> data = userGroupService.list();
        List<DictionaryDataEntity> dataEntityList =
                dictionaryDataService.getList(DictionaryDataEnum.PERMISSION_GROUP.getDictionaryTypeId());
        // 获取数据接口外层菜单
        for (DictionaryDataEntity dictionaryDataEntity : dataEntityList) {
            GroupTreeModel firstModel = JsonUtil.getJsonToBean(dictionaryDataEntity, GroupTreeModel.class);
            firstModel.setId(dictionaryDataEntity.getId());
            firstModel.setType("0");
            long num = data.stream().filter(t -> t.getType().equals(dictionaryDataEntity.getId())).count();
            firstModel.setNum(num);
            if (num > 0) {
                tree.add(firstModel);
            }
        }
        for (GroupEntity entity : data) {
            GroupTreeModel treeModel = JsonUtil.getJsonToBean(entity, GroupTreeModel.class);
            treeModel.setType(entity.getType());
            treeModel.setParentId(entity.getType());
            treeModel.setId(entity.getId());
            DictionaryDataEntity dataEntity = dictionaryDataService.getInfo(entity.getType());
            if (dataEntity != null) {
                tree.add(treeModel);
            }
        }
        List<SumTree<GroupTreeModel>> sumTrees = TreeDotUtils.convertListToTreeDot(tree);
        List<GroupSelectorVO> list = JsonUtil.getJsonToList(sumTrees, GroupSelectorVO.class);
        ListVO<GroupSelectorVO> vo = new ListVO<>();
        vo.setList(list);
        return ActionResult.success(list);
    }

    /**
     * 信息
     *
     * @param id
     * @return
     */
    @ApiOperation("信息")
    @GetMapping("/{id}")
    public ActionResult<GroupInfoVO> info(@PathVariable("id") String id) {
        GroupEntity entity = userGroupService.getInfo(id);
        GroupInfoVO vo = JsonUtil.getJsonToBean(entity, GroupInfoVO.class);
        return ActionResult.success(vo);
    }

    /**
     * 创建
     *
     * @param userGroupCrForm
     * @return
     */
    @ApiOperation("创建")
    @PostMapping
    public ActionResult create(@RequestBody @Valid GroupCrForm userGroupCrForm) {
        GroupEntity entity = JsonUtil.getJsonToBean(userGroupCrForm, GroupEntity.class);
        // 判断名称是否重复
        if (userGroupService.isExistByFullName(entity.getFullName(), entity.getId())) {
            return ActionResult.fail(MsgCode.EXIST001.get());
        }
        // 判断编码是否重复
        if (userGroupService.isExistByEnCode(entity.getEnCode(), entity.getId())) {
            return ActionResult.fail(MsgCode.EXIST001.get());
        }
        userGroupService.crete(entity);
        return ActionResult.success("创建成功");
    }

    /**
     * 更新
     *
     * @param id
     * @param userGroupUpForm
     * @return
     */
    @ApiOperation("更新")
    @PutMapping("/{id}")
    public ActionResult update(@PathVariable("id") String id, @RequestBody @Valid GroupUpForm userGroupUpForm) {
        GroupEntity entity = JsonUtil.getJsonToBean(userGroupUpForm, GroupEntity.class);
        // 判断名称是否重复
        if (userGroupService.isExistByFullName(entity.getFullName(), id)) {
            return ActionResult.fail(MsgCode.EXIST001.get());
        }
        // 判断编码是否重复
        if (userGroupService.isExistByEnCode(entity.getEnCode(), id)) {
            return ActionResult.fail(MsgCode.EXIST001.get());
        }
        Boolean flag = userGroupService.update(id, entity);
        if (!flag) {
            return ActionResult.fail(MsgCode.FA013.get());
        }
        return ActionResult.success(MsgCode.SU004.get());
    }

    /**
     * 删除
     *
     * @param id
     * @return
     */
    @ApiOperation("删除")
    @DeleteMapping("/{id}")
    public ActionResult delete(@PathVariable("id") String id) {
        GroupEntity entity = userGroupService.getInfo(id);
        if (entity != null) {
            userGroupService.delete(entity);
            return ActionResult.success(MsgCode.SU003.get());
        }
        return ActionResult.fail(MsgCode.FA003.get());
    }


    /**
     * 根据id列表获取数据
     *
     * @param ids
     * @return
     */
    @ApiOperation("根据id列表获取数据")
    @PostMapping("/getListByIds")
    public ActionResult getListByIds(@RequestBody @Valid List<String> ids) {
        List<GroupEntity> listByIds = userGroupService.getListByIds(ids);
        return ActionResult.success(listByIds);
    }

}
