package com.bringspring.system.external.service.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.lang.Validator;
import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSONObject;
import com.bringspring.common.base.UserInfo;
import com.bringspring.common.util.RandomUtil;
import com.bringspring.common.util.StringUtils;
import com.bringspring.system.external.bean.WeComModel;
import com.bringspring.system.external.config.mutil.WxCpConfiguration;
import com.bringspring.system.external.service.SynPushToThirdQyService;
import com.bringspring.system.message.entity.SynThirdInfoEntity;
import com.bringspring.system.message.service.SynThirdInfoService;
import com.bringspring.system.message.util.SynThirdConsts;
import com.bringspring.system.permission.entity.OrganizeEntity;
import com.bringspring.system.permission.entity.PositionEntity;
import com.bringspring.system.permission.entity.UserEntity;
import com.bringspring.system.permission.entity.UserRelationEntity;
import com.bringspring.system.permission.service.OrganizeService;
import com.bringspring.system.permission.service.PositionService;
import com.bringspring.system.permission.service.UserRelationService;
import me.chanjar.weixin.common.error.WxErrorException;
import me.chanjar.weixin.cp.api.WxCpService;
import me.chanjar.weixin.cp.bean.Gender;
import me.chanjar.weixin.cp.bean.WxCpDepart;
import me.chanjar.weixin.cp.bean.WxCpUser;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.transaction.annotation.Transactional;

import java.util.List;
import java.util.Optional;
import java.util.stream.Collectors;

import static com.bringspring.system.external.constant.WxCpSysConfigConsts.REGEX_CHAR;
import static com.bringspring.system.external.constant.WxCpSysConfigConsts.TARGET_CHAR;
import static com.bringspring.system.message.util.SynThirdConsts.*;
import static com.bringspring.system.permission.constant.PermissionConst.ORGANIZE;
import static org.springframework.transaction.annotation.Isolation.READ_COMMITTED;

/**
 * 本系统的公司、部门、用户与企业微信的同步
 *
 * @版本： V1.0.0
 * @版权： 荣科科技股份有限公司
 * @作者： RKKJ开发平台组
 * @日期： 2021/4/27 11:12
 */
@Component
public class SynPushToThirdQyServiceImpl implements SynPushToThirdQyService {
    @Autowired
    private SynThirdInfoService synThirdInfoService;

    @Autowired
    WxCpConfiguration wxCpConfiguration;

    @Autowired
    private OrganizeService organizeService;

    @Autowired
    private PositionService positionService;

    @Autowired
    private UserRelationService userRelationService;

    //------------------------------------本系统同步公司、部门到企业微信-------------------------------------

    /**
     * 创建部门：往企业微信创建部门
     * <p>
     * 新增或第三方同步表信息systemObjectId没有关联到，走此方法。
     * 第一步：获取组织所属corpid
     * 第二步：获取系统配置，判断是否执行同步，否则只执行最后一步更新第三方同步表。
     * <p>  单条记录执行时,受开关限制(系统配置>触发事件),isBatch为true跳过开关限制
     * 第三步：企业微信的部门。是执行同步，则设置需要同步到企业微信的对象属性值；
     * <p>  通讯录secret: 新增、更新、删除部门信息。
     * <p>  应用secret: 获取企业微信部门列表。
     * <p>  获取企业微信上的所有部门列表信息，判断父级部门的合法性。
     * <p>  部门中文名称与英文名称不能相同（若中文名称与英文名称相同，直接更新第三方同步表）
     * 第四步：部门管理接口 - 创建部门.
     * 第五步：往同步写入本系统与第三方的对应信息，同步成功。
     *
     * @param currentUser 当前操作用户(异步线程无法获取，所以传参)
     * @param isBatch     是否批量(批量不受开关限制)，仅用作标识 不受配置开关限制。
     * @param deptEntity
     * @param corpId      企业微信ID
     * @return
     * @throws WxErrorException
     */
    @Override
    public JSONObject createDepartmentSysToQy(UserInfo currentUser, boolean isBatch, OrganizeEntity deptEntity,
                                              String corpId) {
        JSONObject retMsg = new JSONObject(); // 返回结果
        String deptFlag = "创建：";
        boolean isDeptDiff = true; // 上级部门检查是否异常
        String thirdObjId = ""; // 第三方id
        Integer synState = SYN_STATE_NO; // 同步状态，默认0未同步
        String description = ""; // 同步描述

        /**
         * 第一步：获取组织所属corpid
         */
        if (StringUtils.isEmpty(corpId)) {
            retMsg = getOrganizeCorpid(deptEntity.getOrganizeIdTree(), deptFlag);
            if (retMsg.getBoolean("code")) {
                corpId = retMsg.getString("cropId");
            }
        }

        if (StringUtils.isNotEmpty(corpId)) {
            /**
             * 第二步：获取到系统配置
             */
            final WeComModel weComModel = wxCpConfiguration.getWeComModel(corpId);
            if (weComModel == null) {
                synState = SYN_STATE_NO;
                description = String.format("[%s]未找到对应corpId=[%s]的配置", deptFlag, corpId);
                retMsg.put("code", false);
                retMsg.put("msg", description);
            } else {
                // 单条记录执行时,受开关限制(系统配置>触发事件),(批量不受开关限制)
                int qyhIsSyn = isBatch ? 1 : weComModel.getQyhIsSynOrg();
                boolean isLinkedCorp = weComModel.getIsLinkedCorp();

                // 支持同步
                if (qyhIsSyn == 1 && !isLinkedCorp) {
                    String agentId = weComModel.getQyhAgentId();
                    /**
                     * 通讯录secret
                     */
                    final WxCpService wxCorpService = wxCpConfiguration.getCorpService(corpId);
                    /**
                     * 应用secret
                     */
                    final WxCpService wxCpService = wxCpConfiguration.getCpService(corpId, Integer.valueOf(agentId));
                    if (wxCorpService == null || wxCpService == null) {
                        synState = SYN_STATE_NO;
                        description = String.format("[%s]未找到对应corpId=[%s]、agentId=[%d]的配置", deptFlag, corpId, agentId);
                        retMsg.put("code", false);
                        retMsg.put("msg", description);
                    } else {
                        try {
                            // 获取企业微信部门列表，企业微信创建接口调用前 校验ID是否存在
                            List<WxCpDepart> departList = wxCpService.getDepartmentService().list(null);
                            if (departList == null || departList.size() == 0) {
                                synState = SYN_STATE_FAIL;
                                description = String.format("[%s]获取企业微信部门列表接口数据为空", deptFlag);
                                retMsg.put("code", false);
                                retMsg.put("msg", description);
                            } else {
                                /**
                                 *  第三步：企业微信的部门
                                 *  name:必填项,同一个层级的部门名称不能重复
                                 *  name_en:必填项,同一个层级的部门名称不能重复
                                 *  name与name_en的值不能相同，否则会报错
                                 */
                                WxCpDepart depart = new WxCpDepart();
                                depart.setName(deptEntity.getFullName());
                                depart.setEnName(deptEntity.getEnCode());
                                // 父级部门有效性校验，从本地数据库的同步表获取对应的企业微信ID，为空报异常，不为空再验证所获取接口部门列表是否当前ID 未处理
                                if (OBJECT_TYPE_COMPANY.equals(deptEntity.getCategory()) && "-1".equals(deptEntity.getParentId())) {
                                    // 顶级节点时，企业微信的父节点设置为0
//                                    depart.setParentId(TOP_PID); 设置0会报错
                                } else {
                                    // 获取部门ParentId的第三方表信息
                                    SynThirdInfoEntity parentSynThirdInfo =
                                            synThirdInfoService.getInfoBySysObjId(THIRD_TYPE_QY, DATA_TYPE_ORG,
                                                    deptEntity.getParentId());
                                    // 根据部门的同步表信息判断同步情况
                                    retMsg = checkDepartmentSysToQy(parentSynThirdInfo, departList);
                                    isDeptDiff = retMsg.getBoolean("code");
                                    if (isDeptDiff) { // 父级部门已同步且有效
                                        // 企业微信的父节点设置
                                        depart.setParentId(Long.valueOf(parentSynThirdInfo.getThirdObjectId()));
                                    }
                                }
                                depart.setOrder(111100000 - deptEntity.getSortCode());

                                // 上级部门检查是否异常
                                if (isDeptDiff) {
                                    /**
                                     *  创建时：部门中文名称与英文名称不能相同（若中文名称与英文名称相同，直接更新第三方同步表）
                                     */
                                    retMsg = checkCnEnName(depart.getName(), depart.getEnName(),
                                            OBJECT_OP_ADD, null, currentUser, corpId, THIRD_TYPE_QY,
                                            DATA_TYPE_ORG, deptEntity.getId(), thirdObjId, deptFlag);
                                    if (!retMsg.getBoolean("code")) {
                                        return retMsg;
                                    }

                                    /**
                                     * 第四步：部门管理接口 - 创建部门.
                                     * 往企业微信写入公司或部门
                                     *
                                     */
                                    Long departId = wxCorpService.getDepartmentService().create(depart);
                                    // 同步成功
                                    thirdObjId = String.valueOf(departId);
                                    synState = SYN_STATE_OK;
                                    description = deptFlag + "同步成功";
                                    retMsg.put("code", true);
                                    retMsg.put("error", description);
                                } else {
                                    // 同步失败,上级部门检查有异常
                                    synState = SYN_STATE_FAIL;
                                    description = deptFlag + "上级部门无对应的企业微信ID";
                                    retMsg.put("code", false);
                                    retMsg.put("error", description);
                                }
                            }
                        } catch (WxErrorException e) {
                            // 同步失败
                            synState = SYN_STATE_FAIL;
                            description = deptFlag + e.getError().getErrorMsg();
                            retMsg.put("code", false);
                            retMsg.put("error", description);
                        }
                    }
                } else {
                    // 未设置单条同步,归并到未同步状态
                    // 未同步
                    synState = SYN_STATE_NO;
                    description = deptFlag + "系统未设置单条同步";

                    retMsg.put("code", true);
                    retMsg.put("error", description);
                    retMsg.put("retDeptId", "0");
                }
            }

        } else {
            description = retMsg.getString("error"); // 同步描述
        }

        /**
         *  第五步：往同步写入本系统与第三方的对应信息
         *  更新同步表
         */
        saveSynThirdInfoEntity(OBJECT_OP_ADD, null, currentUser, corpId, THIRD_TYPE_QY,
                DATA_TYPE_ORG, deptEntity.getId(), thirdObjId, synState, description);

        return retMsg;
    }


    /**
     * 更新部门：往企业微信更新部门
     * <p>
     * 第一步：获取组织所属corpid
     * 第二步：获取指定第三方工具、指定数据类型、指定corpId、对象ID的同步信息
     * 第三步：获取系统配置
     * <p>  判断是否执行同步，否则只执行最后一步更新第三方同步表。
     * <p>  单条记录执行时,受开关限制(系统配置>触发事件),isBatch为true跳过开关限制
     * <p>  通讯录secret: 新增、更新、删除部门信息。
     * <p>  应用secret: 获取企业微信部门列表。
     * 第四步：企业微信的部门。
     * <p>  是执行同步，
     * <p>  获取企业微信上的所有部门列表信息。
     * <p>  则设置需要同步到企业微信的对象属性值；
     * <p>  判断父级部门的合法性。
     * <p>  判断当前部门对应的第三方的合法性
     * <p>     flag:1 已同步但第三方上没对应的ID，需要删除原来的同步信息，再创建同步到企业微信、写入同步表
     * <p>     flag:2 已同步但第三方ID为空，需要创建同步到企业微信、修改同步表
     * <p>     flag:3 未同步，需要创建同步到企业微信、写入同步表
     * <p>  部门中文名称与英文名称不能相同（若中文名称与英文名称相同，直接更新第三方同步表）
     * <p>  部门管理接口 - 往企业微信写入部门.
     * <p>  往同步写入本系统与第三方的对应信息，同步成功。
     *
     * @param currentUser 当前操作用户(异步线程无法获取，所以传参)
     * @param isBatch     是否批量(批量不受开关限制)，仅用作标识 不受配置开关限制。
     * @param deptEntity
     * @param corpId      企业微信ID
     * @return
     * @throws WxErrorException
     */
    @Override
    public JSONObject updateDepartmentSysToQy(UserInfo currentUser, boolean isBatch, OrganizeEntity deptEntity,
                                              String corpId) {
        JSONObject retMsg = new JSONObject(); // 返回结果
        String deptFlag = "更新：";
        boolean isDeptDiff = true; // 上级部门检查是否异常
        String opType = OBJECT_OP_UPD;// 操作类型
        Integer synState = SYN_STATE_NO; // 同步状态，默认0未同步
        String description = ""; // 同步描述

        /**
         * 第一步：获取指定第三方工具、指定数据类型、指定corpId、对象ID的同步信息
         */
        SynThirdInfoEntity synThirdInfoEntity =
                synThirdInfoService.getInfoBySysObjId(THIRD_TYPE_QY, DATA_TYPE_ORG, deptEntity.getId());
        String thirdObjId = ObjectUtil.isNotEmpty(synThirdInfoEntity) ? synThirdInfoEntity.getThirdObjectId() : "";
        SynThirdInfoEntity synThirdInfoPara = new SynThirdInfoEntity();

        /**
         * 第二步：获取组织所属corpid
         */
        if (StringUtils.isEmpty(corpId)) {
            retMsg = getOrganizeCorpid(deptEntity.getOrganizeIdTree(), deptFlag);
            if (retMsg.getBoolean("code")) {
                corpId = retMsg.getString("cropId");
            }
        }

        if (StringUtils.isNotEmpty(corpId)) {

            /**
             * 第三步：获取到系统配置
             */
            final WeComModel weComModel = wxCpConfiguration.getWeComModel(corpId);
            if (weComModel == null) {
                synState = SYN_STATE_NO;
                description = String.format("[%s]未找到对应corpId=[%s]的配置", deptFlag, corpId);
                retMsg.put("code", false);
                retMsg.put("msg", description);
            } else {
                // 单条记录执行时,受开关限制(系统配置>触发事件),(批量不受开关限制)
                int qyhIsSyn = isBatch ? 1 : weComModel.getQyhIsSynOrg();
                boolean isLinkedCorp = weComModel.getIsLinkedCorp();

                // 支持同步
                if (qyhIsSyn == 1 && !isLinkedCorp) {
                    String agentId = weComModel.getQyhAgentId();
                    /**
                     * 通讯录secret
                     */
                    final WxCpService wxCorpService = wxCpConfiguration.getCorpService(corpId);
                    /**
                     * 应用secret
                     */
                    final WxCpService wxCpService = wxCpConfiguration.getCpService(corpId, Integer.valueOf(agentId));
                    if (wxCorpService == null || wxCpService == null) {
                        synState = SYN_STATE_NO;
                        description = String.format("[%s]未找到对应corpId=[%s]、agentId=[%d]的配置", deptFlag, corpId, agentId);
                        retMsg.put("code", false);
                        retMsg.put("msg", description);
                    } else {
                        try {
                            // 获取企业微信部门列表，企业微信创建接口调用前 校验ID是否存在
                            List<WxCpDepart> departList = wxCpService.getDepartmentService().list(null);
                            if (departList == null || departList.size() == 0) {
                                synState = SYN_STATE_FAIL;
                                description = String.format("[%s]获取企业微信部门列表接口数据为空", deptFlag);
                                retMsg.put("code", false);
                                retMsg.put("msg", description);
                            } else {

                                /**
                                 *  第四步：企业微信的部门
                                 *  name:必填项,同一个层级的部门名称不能重复
                                 *  name_en:必填项,同一个层级的部门名称不能重复
                                 *  name与name_en的值不能相同，否则会报错
                                 */
                                WxCpDepart depart = new WxCpDepart();
                                depart.setName(deptEntity.getFullName());
                                depart.setEnName(deptEntity.getEnCode());
                                // 父级部门有效性校验，从本地数据库的同步表获取对应的企业微信ID，为空报异常，不为空再验证所获取接口部门列表是否当前ID 未处理
//                                if (OBJECT_TYPE_COMPANY.equals(deptEntity.getCategory()) && "-1".equals(deptEntity.getParentId())) {
                                if ("-1".equals(deptEntity.getParentId())) {
                                    //顶级节点时，企业微信的父节点设置为0
//                                    depart.setParentId(TOP_PID);  设置0会报错
                                } else {
                                    // 获取部门ParentId的第三方表信息
                                    SynThirdInfoEntity parentSynThirdInfo =
                                            synThirdInfoService.getInfoBySysObjId(THIRD_TYPE_QY, DATA_TYPE_ORG,
                                                    deptEntity.getParentId());
                                    // 根据部门的同步表信息判断同步情况
                                    retMsg = checkDepartmentSysToQy(parentSynThirdInfo, departList);
                                    isDeptDiff = retMsg.getBoolean("code");
                                    if (isDeptDiff) { // 父级部门已同步且有效
                                        // 企业微信的父节点设置
                                        depart.setParentId(Long.valueOf(parentSynThirdInfo.getThirdObjectId()));
                                    }
                                }
                                depart.setOrder(1000000000 - deptEntity.getSortCode());

                                // 上级部门检查是否异常
                                if (isDeptDiff) {
                                    /**
                                     * 判断当前部门对应的第三方的合法性
                                     */
                                    retMsg = checkDepartmentSysToQy(synThirdInfoEntity, departList);
                                    if (!retMsg.getBoolean("code")) {
                                        final String flag = retMsg.getString("flag");
                                        if ("1".equals(flag) || "3".equals(flag)) {
                                            // flag:1 已同步但第三方上没对应的ID，需要删除原来的同步信息，再创建同步到企业微信、写入同步表
                                            // flag:3 未同步，需要创建同步到企业微信、写入同步表
                                            if ("1".equals(flag)) {
                                                synThirdInfoService.delete(synThirdInfoEntity);
                                            }
                                            opType = SynThirdConsts.OBJECT_OP_ADD;
                                            synThirdInfoPara = null;
                                            thirdObjId = "";

                                            // 部门中文名称与英文名称不能相同
                                            retMsg = checkCnEnName(depart.getName(), depart.getEnName(),
                                                    opType, synThirdInfoPara, currentUser, corpId, THIRD_TYPE_QY,
                                                    DATA_TYPE_ORG, deptEntity.getId(), thirdObjId, deptFlag);
                                            if (!retMsg.getBoolean("code")) {
                                                return retMsg;
                                            }
                                            // 往企业微信写入公司或部门
                                            Long departId = wxCorpService.getDepartmentService().create(depart);
                                            // 同步成功
                                            thirdObjId = String.valueOf(departId);
                                            synState = SYN_STATE_OK;
                                            description = deptFlag + "同步成功";
                                            retMsg.put("code", true);
                                            retMsg.put("error", description);

                                        }

                                        if ("2".equals(flag)) {
                                            // flag:2 已同步但第三方ID为空，需要创建同步到企业微信、修改同步表
                                            opType = SynThirdConsts.OBJECT_OP_UPD;
                                            synThirdInfoPara = synThirdInfoEntity;
                                            thirdObjId = "";

                                            // 部门中文名称与英文名称不能相同
                                            retMsg = checkCnEnName(depart.getName(), depart.getEnName(),
                                                    OBJECT_OP_UPD, synThirdInfoPara, currentUser, corpId, THIRD_TYPE_QY,
                                                    DATA_TYPE_ORG, deptEntity.getId(), thirdObjId, deptFlag);
                                            if (!retMsg.getBoolean("code")) {
                                                return retMsg;
                                            }

                                            // 往企业微信写入公司或部门
                                            Long departId = wxCorpService.getDepartmentService().create(depart);
                                            // 同步成功
                                            thirdObjId = String.valueOf(departId);
                                            synState = SYN_STATE_OK;
                                            description = deptFlag + "同步成功";
                                            retMsg.put("code", true);
                                            retMsg.put("error", description);
                                        }

                                    } else {
                                        // 更新同步表
                                        opType = SynThirdConsts.OBJECT_OP_UPD;
                                        synThirdInfoPara = synThirdInfoEntity;
                                        thirdObjId = synThirdInfoEntity.getThirdObjectId();

                                        // 部门中文名称与英文名称不能相同
                                        retMsg = checkCnEnName(depart.getName(), depart.getEnName(),
                                                OBJECT_OP_UPD, synThirdInfoPara, currentUser, corpId, THIRD_TYPE_QY,
                                                DATA_TYPE_ORG, deptEntity.getId(), thirdObjId, deptFlag);
                                        if (!retMsg.getBoolean("code")) {
                                            return retMsg;
                                        }

                                        // 往企业微信写入公司或部门
                                        Long departId = Long.valueOf(synThirdInfoEntity.getThirdObjectId());
                                        depart.setId(departId);
                                        wxCorpService.getDepartmentService().update(depart);
                                        // 同步成功
                                        thirdObjId = String.valueOf(departId);
                                        synState = SYN_STATE_OK;
                                        description = deptFlag + "同步成功";
                                        retMsg.put("code", true);
                                        retMsg.put("error", description);
                                    }
                                } else {
                                    // 同步失败,上级部门检查有异常
                                    synState = SYN_STATE_FAIL;
                                    description = deptFlag + "上级部门无对应的企业微信ID";
                                    retMsg.put("code", false);
                                    retMsg.put("error", description);
                                }
                            }
                        } catch (WxErrorException e) {
                            // 同步失败
                            synState = SYN_STATE_FAIL;
                            description = deptFlag + e.getError().getErrorMsg();
                            retMsg.put("code", false);
                            retMsg.put("error", description);
                        }
                    }
                } else {
                    // 未设置单条同步,归并到未同步状态
                    // 未同步
                    synState = SYN_STATE_NO;
                    description = deptFlag + "系统未设置单条同步";

                    retMsg.put("code", true);
                    retMsg.put("error", description);
                    retMsg.put("retDeptId", "0");
                }
            }
        } else {
            description = retMsg.getString("error"); // 同步描述
        }

        // 往同步写入本系统与第三方的对应信息
        // 更新同步表
        saveSynThirdInfoEntity(opType, synThirdInfoEntity, currentUser, corpId, THIRD_TYPE_QY,
                DATA_TYPE_ORG, deptEntity.getId(), thirdObjId, synState, description);

        return retMsg;
    }


    /**
     * 删除部门：往企业微信删除部门
     * <p>
     * 第一步：获取组织所属corpid
     * 第二步：获取指定第三方工具、指定数据类型、指定corpId、对象ID的同步信息
     * 第三步：获取系统配置
     * <p>  判断是否执行同步，否则只执行最后一步更新第三方同步表。
     * <p>  单条记录执行时,受开关限制(系统配置>触发事件),isBatch为true跳过开关限制
     * <p>  通讯录secret: 新增、更新、删除部门信息。
     * <p>  应用secret: 获取企业微信部门列表。
     * 第四步：企业微信的部门。
     * <p>  是执行同步，
     * <p>  获取企业微信上的所有部门列表信息。
     * <p>  则设置需要同步到企业微信的对象属性值；
     * <p>  判断父级部门的合法性。
     * <p>  判断当前部门对应的第三方的合法性
     * <p>     flag:1 已同步但第三方上没对应的ID，需要删除原来的同步信息，再创建同步到企业微信、写入同步表
     * <p>     flag:2 已同步但第三方ID为空，需要创建同步到企业微信、修改同步表
     * <p>     flag:3 未同步，需要创建同步到企业微信、写入同步表
     * <p>  部门中文名称与英文名称不能相同（若中文名称与英文名称相同，直接更新第三方同步表）
     * <p>  部门管理接口 - 往企业微信写入部门.
     * <p>  往同步写入本系统与第三方的对应信息，同步成功。
     *
     * @param currentUser 当前操作用户(异步线程无法获取，所以传参)
     * @param isBatch     是否批量(批量不受开关限制)，仅用作标识 不受配置开关限制。
     * @param deptId
     * @param corpId      企业微信ID
     * @return
     * @throws WxErrorException
     */
    @Override
    public JSONObject deleteDepartmentSysToQy(UserInfo currentUser, boolean isBatch, String deptId, String corpId) {
        JSONObject retMsg = new JSONObject(); // 返回结果
        String deptFlag = "删除：";
        /**
         * 第一步：获取指定第三方工具、指定数据类型、对象ID的同步信息
         */
        SynThirdInfoEntity synThirdInfoEntity =
                synThirdInfoService.getInfoBySysObjId(THIRD_TYPE_QY, DATA_TYPE_ORG, deptId);
        if (ObjectUtil.isNotEmpty(synThirdInfoEntity)) {
            String thirdObjId = synThirdInfoEntity.getThirdObjectId(); // 第三方id
            Integer synState = SYN_STATE_NO; // 同步状态，默认0未同步
            String description = ""; // 同步描述
            if (StringUtils.isEmpty(corpId)) {
                corpId = synThirdInfoEntity.getCropId();
            }
            /**
             * 第二步：获取到系统配置
             */
            final WeComModel weComModel = wxCpConfiguration.getWeComModel(corpId);
            if (weComModel == null) {
                synState = SYN_STATE_NO;
                description = String.format("[%s]未找到对应corpId=[%s]的配置", deptFlag, corpId);
                retMsg.put("code", false);
                retMsg.put("msg", description);
            } else {
                // 单条记录执行时,受开关限制(系统配置>触发事件),(批量不受开关限制)
                int qyhIsSyn = isBatch ? 1 : weComModel.getQyhIsSynOrg();
                boolean isLinkedCorp = weComModel.getIsLinkedCorp();
                // 支持同步
                if (qyhIsSyn == 1 && !isLinkedCorp) {
                    String agentId = weComModel.getQyhAgentId();
                    /**
                     * 通讯录secret
                     */
                    final WxCpService wxCorpService = wxCpConfiguration.getCorpService(corpId);
                    /**
                     * 应用secret
                     */
                    final WxCpService wxCpService = wxCpConfiguration.getCpService(corpId, Integer.valueOf(agentId));
                    if (wxCorpService == null || wxCpService == null) {
                        synState = SYN_STATE_NO;
                        description = String.format("[%s]未找到对应corpId=[%s]、agentId=[%d]的配置", deptFlag, corpId, agentId);
                        retMsg.put("code", false);
                        retMsg.put("msg", description);
                    } else {
                        try {
                            if (StringUtils.isNotEmpty(thirdObjId)) {
                                // 部门Id必须有值,不能为空
                                Long departId = Long.valueOf(synThirdInfoEntity.getThirdObjectId());
                                wxCorpService.getDepartmentService().delete(departId);
                                // 同步成功,直接删除同步表记录
                                synThirdInfoService.delete(synThirdInfoEntity);
                            } else {
                                // 根据企业微信ID找不到相应的信息,直接删除同步表记录
                                synThirdInfoService.delete(synThirdInfoEntity);
                            }
                            retMsg.put("code", true);
                            retMsg.put("msg", deptFlag + "同步成功");
                            return retMsg;
                        } catch (WxErrorException e) {
                            // 同步失败
                            synState = SYN_STATE_FAIL;
                            description = deptFlag + e.getError().getErrorMsg();
                            retMsg.put("code", false);
                            retMsg.put("error", description);
                        }
                    }
                } else {
                    // 未设置单条同步,归并到未同步状态
                    // 未同步
                    synState = SYN_STATE_NO;
                    description = deptFlag + "系统未设置单条同步";

                    retMsg.put("code", true);
                    retMsg.put("error", description);
                    retMsg.put("retDeptId", "0");
                }
            }

            // 往同步写入本系统与第三方的对应信息
            // 更新同步表
            saveSynThirdInfoEntity(OBJECT_OP_UPD, synThirdInfoEntity, currentUser, corpId, THIRD_TYPE_QY,
                    DATA_TYPE_ORG, deptId, thirdObjId, synState, description);
        } else {
            retMsg.put("code", true);
            retMsg.put("msg", "第三方同步信息为空");
        }
        return retMsg;
    }

    @Override
    public boolean triggeringCondition(OrganizeEntity oldDept, OrganizeEntity newDept) {
        if (ObjectUtil.isNotEmpty(oldDept) && ObjectUtil.isNotEmpty(newDept)) {
            if (!StringUtils.equals(oldDept.getParentId(), newDept.getParentId())) {
                // ParentId变更时触发
                return true;
            } else if (!StringUtils.equals(oldDept.getFullName(), newDept.getFullName())) {
                // FullName变更时触发
                return true;
            } else if (oldDept.getSortCode().longValue() != newDept.getSortCode().longValue()) {
                // SortCode变更时触发
                return true;
            }
        }
        return false;
    }

    @Override
    public boolean triggeringConditionForUser(OrganizeEntity oldDept, OrganizeEntity newDept) {
        if (ObjectUtil.isNotEmpty(oldDept) && ObjectUtil.isNotEmpty(newDept)) {
            if (!StringUtils.equals(oldDept.getManagerId(), newDept.getManagerId())) {
                // ManagerId变更时触发
                return true;
            }
        }
        return false;
    }

    /**
     * 根据部门的同步表信息判断同步情况
     *
     * @param synThirdInfoEntity 第三方表同步信息
     * @return 判断结果
     */
    public JSONObject checkDepartmentSysToQy(SynThirdInfoEntity synThirdInfoEntity) {
        return checkDepartmentSysToQy(synThirdInfoEntity, null);
    }

    /**
     * 根据部门的同步表信息判断同步情况
     *
     * @param synThirdInfoEntity 第三方表同步信息
     * @param qyDeptList         企业微信部门列表
     * @return 判断结果
     */
    public JSONObject checkDepartmentSysToQy(SynThirdInfoEntity synThirdInfoEntity, List<WxCpDepart> qyDeptList) {
        JSONObject retMsg = new JSONObject();
        retMsg.put("code", true);
        retMsg.put("flag", "");
        retMsg.put("error", "");

        if (synThirdInfoEntity != null) {
            if (StringUtils.isNotEmpty(synThirdInfoEntity.getThirdObjectId())) {
                if (CollectionUtil.isNotEmpty(qyDeptList)) {
                    // 同步表存在企业微信ID,仍需要判断企业微信上有没此部门
                    if (qyDeptList.stream().filter(t -> t.getId().toString().equals(synThirdInfoEntity.getThirdObjectId())).count() == 0 ? true : false) {
                        retMsg.put("code", false);
                        retMsg.put("flag", "1");
                        retMsg.put("error", "企业微信不存在同步表对应的部门ID!");
                    }
                }
            } else {
                // 同步表的企业微信ID为空
                retMsg.put("code", false);
                retMsg.put("flag", "2");
                retMsg.put("error", "同步表中部门对应的企业微信ID为空!");
            }
        } else {
            // 上级部门未同步
            retMsg.put("code", false);
            retMsg.put("flag", "3");
            retMsg.put("error", "部门未同步到企业微信!");
        }

        return retMsg;
    }

    /**
     * 检查部门中文名称与英文名称是否相同
     * 若中文名称与英文名称相同，直接更新第三方同步表
     *
     * @param cnName             中文名称
     * @param EnName             英文名称
     * @param opType             对象操作类型(add:创建;upd:修改)
     * @param synThirdInfoEntity 第三方表信息
     * @param thirdType          第三方工具类型(1:企业微信;2:钉钉)
     * @param dataType           数据类型(1:组织(公司与部门);2:用户)
     * @param sysObjId           系统对象ID
     * @param thirdObjId         第三方对象ID
     * @param deptFlag
     * @return
     */
    public JSONObject checkCnEnName(String cnName, String EnName, String opType, SynThirdInfoEntity synThirdInfoEntity,
                                    UserInfo currentUser, String corpId, String thirdType,
                                    String dataType, String sysObjId, String thirdObjId, String deptFlag) {
        JSONObject retMsg = new JSONObject();
        retMsg.put("code", true);
        retMsg.put("error", "");
        if (cnName.equals(EnName)) {
            // 同步失败
            Integer synState = SYN_STATE_FAIL;
            String description = deptFlag + "部门中文名称与英文名称不能相同";

            // 更新同步表
            saveSynThirdInfoEntity(opType, synThirdInfoEntity, currentUser, corpId, thirdType, dataType, sysObjId,
                    thirdObjId,
                    synState, description);

            retMsg.put("code", false);
            retMsg.put("error", description);
        }
        return retMsg;
    }


    /**
     * 将组织、用户的信息写入同步表
     *
     * @param opType             "add":创建 “upd”:修改
     * @param synThirdInfoEntity 本地同步表信息
     * @param thirdType          第三方类型
     * @param dataType           数据类型
     * @param sysObjId           本地对象ID
     * @param thirdObjId         第三方对象ID
     * @param synState           同步状态(0:未同步;1:同步成功;2:同步失败)
     * @param description
     */
    public void saveSynThirdInfoEntity(String opType, SynThirdInfoEntity synThirdInfoEntity, UserInfo currentUser,
                                       String corpId, String thirdType, String dataType,
                                       String sysObjId, String thirdObjId, Integer synState, String description) {
        SynThirdInfoEntity entity = new SynThirdInfoEntity();
        if (OBJECT_OP_ADD.equals(opType)) {
            entity.setId(RandomUtil.uuId());
            entity.setCropId(corpId);
            entity.setThirdType(Integer.parseInt(thirdType));
            entity.setDataType(Integer.parseInt(dataType));
            entity.setSystemObjectId(sysObjId);
            entity.setThirdObjectId(thirdObjId);
            entity.setSynState(synState);
            // 备注当作同步失败信息来用
            entity.setDescription(description);
            entity.setCreatorUserId(currentUser.getUserId());
//            entity.setCreatorTime(DateUtil.getNowDate());
            entity.setLastModifyUserId(currentUser.getUserId());
            // 修改时间当作最后同步时间来用
//            entity.setLastModifyTime(DateUtil.getNowDate());
            synThirdInfoService.create(entity);
        } else {
            entity = synThirdInfoEntity;
            entity.setCropId(corpId);
            entity.setThirdType(Integer.parseInt(thirdType));
            entity.setDataType(Integer.parseInt(dataType));
            entity.setThirdObjectId(thirdObjId);
            entity.setSynState(synState);
            // 备注当作同步失败信息来用
            entity.setDescription(description);
            entity.setLastModifyUserId(currentUser.getUserId());
            // 修改时间当作最后同步时间来用
//            entity.setLastModifyTime(DateUtil.getNowDate());
            synThirdInfoService.update(entity.getId(), entity);
        }
    }

    /**
     * 获取组织所属corpid
     *
     * @param organizeIdTree 部门祖宗IDS
     * @param deptFlag
     * @return
     */
    public JSONObject getOrganizeCorpid(String organizeIdTree, String deptFlag) {
        JSONObject retMsg = new JSONObject();
        retMsg.put("code", false);
        if (StringUtils.isNotEmpty(organizeIdTree)) {
            String[] organizeIdArr = organizeIdTree.split(REGEX_CHAR);
            if (ObjectUtil.isNotEmpty(organizeIdArr) && organizeIdArr.length > 0) {
                // 获取顶级部门的第三方表信息
                SynThirdInfoEntity topEntity = synThirdInfoService.getInfoBySysObjId(SynThirdConsts.THIRD_TYPE_QY,
                        SynThirdConsts.DATA_TYPE_ORG, organizeIdArr[0]);
                if (ObjectUtil.isNotEmpty(topEntity)) {
                    if (StringUtils.isNotEmpty(topEntity.getCropId())) {
                        retMsg.put("code", true);
                        retMsg.put("error", deptFlag + "获取组织所属corpid成功");
                        retMsg.put("cropId", topEntity.getCropId());
                    } else {
                        retMsg.put("code", false);
                        retMsg.put("error", deptFlag + "获取组织所属corpid失败，系统组织顶级id第三方表信息Corpid为空");
                    }
                } else {
                    retMsg.put("code", false);
                    retMsg.put("error", deptFlag + "获取组织所属corpid失败，系统组织顶级id不存在第三方表信息");
                }
            } else {
                retMsg.put("code", false);
                retMsg.put("error", deptFlag + "获取组织所属corpid失败，系统组织顶级id获取失败");
            }
        } else {
            retMsg.put("code", false);
            retMsg.put("error", deptFlag + "获取组织所属corpid失败，系统组织父级组织tree为空");
        }

        return retMsg;
    }


    //------------------------------------本系统同步成员到企业微信-------------------------------------

    /**
     * isolation 的 Eum 类中定义了“五个”表示隔离级别的值，如下：
     *
     * Isolation.DEFAULT：使用各个数据库默认的隔离级别【默认】
     * Isolation.READ_UNCOMMITTED：读取未提交数据（会出现脏读, 不可重复读）（基本不使用）
     * Isolation.READ_COMMITTED：读取已提交数据（会出现不可重复读和幻读）
     * Isolation.REPEATABLE_READ：可重复读（会出现幻读）
     * Isolation.SERIALIZABLE：串行化
     */

    @Override
    @Transactional(isolation = READ_COMMITTED)
    public JSONObject createUserSysToQy(UserInfo currentUser, boolean isBatch, UserEntity userEntity, String corpId) {
        JSONObject retMsg = new JSONObject(); // 返回结果
        retMsg.put("code", false);
        String userFlag = "创建：";
        String thirdObjId = ""; // 第三方id
        Integer synState = SYN_STATE_NO; // 同步状态，默认0未同步
        String description = ""; // 同步描述

        /**
         * 第一步：获取成员所属部门
         */
        String organizeId = userEntity.getOrganizeId();
        OrganizeEntity deptEntity = organizeService.getInfo(organizeId);

        /**
         * 第二步：获取成员所属组织所属corpid
         */
        if (StringUtils.isEmpty(corpId)) {
            retMsg = getOrganizeCorpid(deptEntity.getOrganizeIdTree(), userFlag);
            if (retMsg.getBoolean("code")) {
                corpId = retMsg.getString("cropId");
            }
        }

        if (StringUtils.isNotEmpty(corpId)) {
            /**
             * 第三步：获取到系统配置
             */
            final WeComModel weComModel = wxCpConfiguration.getWeComModel(corpId);
            if (weComModel == null) {
                synState = SYN_STATE_NO;
                description = String.format("[%s]未找到对应corpId=[%s]的配置", userFlag, corpId);
                retMsg.put("code", false);
                retMsg.put("msg", description);
            } else {
                // 单条记录执行时,受开关限制isSynUser(系统配置>触发事件),(批量不受开关限制)
                int qyhIsSyn = isBatch ? 1 : weComModel.getQyhIsSynUser();
                boolean isLinkedCorp = weComModel.getIsLinkedCorp();

                // 支持同步
                if (qyhIsSyn == 1 && !isLinkedCorp) {
                    String agentId = weComModel.getQyhAgentId();
                    /**
                     * 通讯录secret
                     */
                    final WxCpService wxCorpService = wxCpConfiguration.getCorpService(corpId);
                    /**
                     * 应用secret
                     */
                    final WxCpService wxCpService = wxCpConfiguration.getCpService(corpId, Integer.valueOf(agentId));
                    if (wxCorpService == null || wxCpService == null) {
                        synState = SYN_STATE_NO;
                        description = String.format("[%s]未找到对应corpId=[%s]、agentId=[%d]的配置", userFlag, corpId, agentId);
                        retMsg.put("code", false);
                        retMsg.put("msg", description);
                    } else {
                        // 执行同步
                        try {
                            retMsg = wxCpUserCreateOrUpdate(wxCpService, userEntity, retMsg, null);
                            if (retMsg.getBoolean("code")) {
                                // 往同步写入本系统与第三方的对应信息
                                WxCpUser user = retMsg.getObject("wxCpUser", WxCpUser.class);
                                wxCorpService.getUserService().create(user);
                                // 同步成功
//                                thirdObjId = corpId + TARGET_CHAR + user.getUserId();
                                thirdObjId = user.getUserId();
                                synState = SYN_STATE_OK;
                                description = userFlag + "同步成功";
                                retMsg.put("code", true);
                                retMsg.put("error", description);
                            } else {
                                // 同步失败,原因：部门找不到对应的第三方ID、邮箱格式不合法
                                synState = SYN_STATE_FAIL;
                                description = userFlag + retMsg.getString("error");
                                retMsg.put("code", false);
                                retMsg.put("error", description);
                            }
                        } catch (WxErrorException e) {
                            // 同步失败
                            synState = SYN_STATE_FAIL;
                            description = userFlag + e.getError().getErrorMsg();
                            retMsg.put("code", false);
                            retMsg.put("error", description);
                        }
                    }
                } else {
                    // 未设置单条同步,归并到未同步状态
                    // 未同步
                    synState = SYN_STATE_NO;
                    description = userFlag + "系统未设置单条同步";

                    retMsg.put("code", false);
                    retMsg.put("error", description);
                }
            }

        } else {
            description = retMsg.getString("error"); // 同步描述
        }

        /**
         *  往同步写入本系统与第三方的对应信息
         *  更新同步表
         */
        saveSynThirdInfoEntity(OBJECT_OP_ADD, null, currentUser, corpId, THIRD_TYPE_QY,
                DATA_TYPE_USER, userEntity.getId(), thirdObjId, synState, description);
        return retMsg;
    }

    /**
     * 设置需要提交给企业微信接口的用户对象
     *
     * @param wxCpService 应用secret
     * @param userEntity  本地用户信息
     * @param retMsg      封装结果
     * @return
     * @throws WxErrorException
     */
    public JSONObject wxCpUserCreateOrUpdate(WxCpService wxCpService, UserEntity userEntity,
                                             JSONObject retMsg, WxCpUser wxCpUserInfo) throws WxErrorException {
//        List<UserEntity> userList = userService.getList();
        /**
         *  封装企业微信创建成员的参数信息
         */
        WxCpUser user = new WxCpUser();

        /**
         * 验证邮箱格式的合法性
         */
        if (StringUtils.isNotEmpty(userEntity.getEmail())) {
            if (!Validator.isEmail(userEntity.getEmail())) {
                retMsg.put("code", false);
                retMsg.put("error", "邮箱格式不合法！");
                retMsg.put("wxCpUser", null);
                return retMsg;
            }
        }
        /**
         * 验证部门有效性
         */
        // 获取企业微信部门列表，企业微信创建接口调用前 校验ID是否存在
        List<WxCpDepart> departList = wxCpService.getDepartmentService().list(null);
        if (CollectionUtil.isEmpty(departList)) {
            retMsg.put("code", false);
            retMsg.put("error", "企业微信部门列表接口数据isEmpty");
            retMsg.put("wxCpUser", null);
            return retMsg;
        } else {
            // 用户所属部门ids
            List<String> userAllOrgIds =
                    userRelationService.getListByUserIdAndObjType(userEntity.getId(), ORGANIZE).stream().map(UserRelationEntity::getObjectId).collect(Collectors.toList());
            // 主部门ID
            String organizeId = userEntity.getOrganizeId();
            if (CollectionUtil.isEmpty(userAllOrgIds)) {
                // 部门无效
                retMsg.put("code", false);
                retMsg.put("error", "成员所属部门isEmpty");
                retMsg.put("wxCpUser", null);
                return retMsg;
            } else {
                // 根据系统部门ID批量获取第三方同步表信息
                List<SynThirdInfoEntity> orgSynThirdList =
                        synThirdInfoService.getListByObjId(Integer.valueOf(THIRD_TYPE_QY), userAllOrgIds);
                if (CollectionUtil.isEmpty(orgSynThirdList)) {
                    // 部门无效
                    retMsg.put("code", false);
                    retMsg.put("error", "成员所属部门第三方同步表信息isEmpty");
                    retMsg.put("wxCpUser", null);
                    return retMsg;
                } else {
                    String mainDepartment = null;
                    boolean orgAllSyn = true;
                    String description = "";
                    for (SynThirdInfoEntity entity : orgSynThirdList) {
                        JSONObject validateRetMsg = checkDepartmentSysToQy(entity, departList);
                        if (!validateRetMsg.getBoolean("code")) { // 部门未同步
                            String errorMsg = validateRetMsg.getString("error");
                            orgAllSyn = false;
                            // 企业微信的父节点设置
                            description += String.format("[%s]，[%s]", entity.getSystemObjectId(), errorMsg);
                        } else {
                            if (entity.getSystemObjectId().equals(organizeId)) {
                                mainDepartment = entity.getThirdObjectId();
                            }
                        }
                    }

                    if (!orgAllSyn) {
                        // 部门存在无效
                        retMsg.put("code", false);
                        retMsg.put("error", description);
                        retMsg.put("wxCpUser", null);
                        return retMsg;
                    } else {
                        /**
                         * 部门有效
                         */
                        // 系统部门信息
                        List<OrganizeEntity> listAll = organizeService.getListAll(userAllOrgIds, null);
                        int size = orgSynThirdList.size();
                        // 成员所属部门id列表
                        Long[] departIds =new Long[size];
                        // 部门内的排序值，默认为0，成员次序以创建时间从小到大排列。个数必须和参数department的个数一致，数值越大排序越前面。
                        Integer[] orders = new Integer[size];
                        Long sortCode = userEntity.getSortCode();
                        // 个数必须和参数department的个数一致，表示在所在的部门内是否为部门负责人。1 表示为部门负责人，0 表示非部门负责人。
                        Integer[] isLeaderInDept = new Integer[size];
                        for (int i = 0; i < size; i++) {
                            SynThirdInfoEntity thirdInfoEntity = orgSynThirdList.get(i);
                            // 设置部门id
                            departIds[i] = Long.valueOf(thirdInfoEntity.getThirdObjectId());
                            Optional<OrganizeEntity> first =
                                    listAll.stream().filter(s -> s.getId().equals(thirdInfoEntity.getSystemObjectId())).collect(Collectors.toList()).stream().findFirst();
                            if (userEntity.getId().equals(first.get().getManagerId())) {
                                // 设置部门id->负责人
                                isLeaderInDept[i] = 1;
                            } else {
                                // 设置部门id->负责人
                                isLeaderInDept[i] = 0;
                            }
                            isLeaderInDept[i] = 0;
                            // 设置部门id->部门内排序 (数量必须和department一致，数值越大排序越前面)
                            orders[i] = ObjectUtil.isNull(sortCode) ? 0 : 1000000000 - sortCode.intValue();
                        }
                        // 成员所属部门id列表
                        user.setDepartIds(departIds);
                        // 主部门
                        user.setMainDepartment(mainDepartment);
                        // 成员所属部门内排序
                        user.setOrders(orders);
                        // 成员所属部门是否为负责人
                        user.setIsLeaderInDept(isLeaderInDept);
                    }
                }
            }
        }

        /**
         * 岗位
         */
        PositionEntity positionEntity = positionService.getInfo(userEntity.getPositionId());
        if (ObjectUtil.isNotEmpty(positionEntity)) {
            user.setPosition(positionEntity.getFullName());
        }

        // 成员UserID。对应管理端的账号，企业内必须唯一。
        user.setUserId(userEntity.getAccount());
        // 成员名称
        user.setName(userEntity.getRealName());
        // 成员别名
        user.setAlias(userEntity.getQuickQuery());
        // 手机号码。企业内必须唯一，mobile/email二者不能同时为空
        user.setMobile(userEntity.getMobilePhone());
        // 邮箱。且为有效的email格式。企业内必须唯一，mobile/email二者不能同时为空
        user.setEmail(userEntity.getEmail());
        // 性别。1表示男性，2表示女性
        user.setGender(Gender.fromCode(userEntity.getGender().toString()));
        // 座机
        user.setTelephone(userEntity.getTelePhone());
        // 启用/禁用成员。1表示启用成员，0表示禁用成员
        Integer enabledMark = userEntity.getEnabledMark();
        user.setEnable(enabledMark != null && enabledMark != 0 ? 1 : 0);
        // 自定义字段。自定义字段需要先在WEB管理端添加
        user.addExtAttr("工号", userEntity.getCode());
        user.addExtAttr("手机号码", userEntity.getMobilePhone());

        // 是否邀请该成员使用企业微信（将通过微信服务通知或短信或邮件下发邀请，每天自动下发一次，最多持续3个工作日），默认值为true。
//        user.setToInvite(true);
        // 	地址。长度最大128个字符
        user.setAddress(userEntity.getPostalAddress());
        // 成员头像的mediaid，通过素材管理接口上传图片获得的mediaid
//        user.setAvatarMediaId("");
        // 成员对外信息
//        WxCpUser.ExternalAttribute externalAttr = new WxCpUser.ExternalAttribute();
//        user.addExternalAttr(externalAttr);
        //对外职务，如果设置了该值，则以此作为对外展示的职务，否则以position来展示。长度12个汉字内
//        user.setExternalPosition();

        // 直属上级UserID，设置范围为企业内成员，可以设置最多1个上级
        List<SynThirdInfoEntity> synThirdInfoEntityList =
                synThirdInfoService.getInfoBySysObjIds(THIRD_TYPE_QY, DATA_TYPE_USER, userEntity.getManagerId(), false);
        if (CollectionUtil.isNotEmpty(synThirdInfoEntityList)) {
            user.setDirectLeader(new String[]{synThirdInfoEntityList.get(0).getThirdObjectId()});
        }


        // 修改时:未更新字段信息来源企业微信
        if (ObjectUtil.isNotEmpty(wxCpUserInfo)) {
            // 成员UserID。
            user.setUserId(wxCpUserInfo.getUserId());
            // 成员别名
            user.setAlias(wxCpUserInfo.getAlias());
            // 成员头像的mediaid
            user.setAvatarMediaId(wxCpUserInfo.getAvatarMediaId());
            // 成员对外信息
            user.setExternalAttrs(wxCpUserInfo.getExternalAttrs());
            //对外职务
            user.setExternalPosition(wxCpUserInfo.getExternalPosition());
        }

        retMsg.put("code", true);
        retMsg.put("error", "提交企业微信的用户信息封装完成");
        retMsg.put("wxCpUser", user);
        return retMsg;
    }

    @Override
    @Transactional(isolation = READ_COMMITTED)
    public JSONObject updateUserSysToQy(UserInfo currentUser, boolean isBatch, UserEntity userEntity, String corpId) {
        JSONObject retMsg = new JSONObject(); // 返回结果
        retMsg.put("code", false);
        String userFlag = "更新：";
        String opType = OBJECT_OP_UPD;
        Integer synState = SYN_STATE_NO; // 同步状态，默认0未同步
        String description = ""; // 同步描述

        /**
         * 第一步：获取同步表信息
         */
        SynThirdInfoEntity synThirdInfoEntity = null;
        List<SynThirdInfoEntity> infoBySysObjIds = synThirdInfoService.getInfoBySysObjIds(THIRD_TYPE_QY, DATA_TYPE_USER, userEntity.getId(), false);
        if (CollectionUtil.isNotEmpty(infoBySysObjIds)) {
            synThirdInfoEntity = infoBySysObjIds.get(0);
        }
        String thirdObjId = ObjectUtil.isNotEmpty(synThirdInfoEntity) ? synThirdInfoEntity.getThirdObjectId() : "";

        /**
         * 第二步：获取成员所属部门
         */
        String organizeId = userEntity.getOrganizeId();
        OrganizeEntity deptEntity = organizeService.getInfo(organizeId);

        /**
         * 第三步：获取成员所属组织所属corpid
         */
        if (StringUtils.isEmpty(corpId)) {
            retMsg = getOrganizeCorpid(deptEntity.getOrganizeIdTree(), userFlag);
            if (retMsg.getBoolean("code")) {
                corpId = retMsg.getString("cropId");
            }
        }

        if (StringUtils.isEmpty(corpId)) {
            synState = SYN_STATE_NO;
            description = retMsg.getString("error");
            retMsg.put("code", false);
            retMsg.put("error", description);
        } else {
            /**
             * 第四步：获取到系统配置
             */
            final WeComModel weComModel = wxCpConfiguration.getWeComModel(corpId);
            if (weComModel == null) {
                synState = SYN_STATE_NO;
                description = String.format("[%s]未找到对应corpId=[%s]的配置", userFlag, corpId);
                retMsg.put("code", false);
                retMsg.put("msg", description);
            } else {
                // 单条记录执行时,受开关限制isSynUser(系统配置>触发事件),(批量不受开关限制)
                int qyhIsSyn = isBatch ? 1 : weComModel.getQyhIsSynUser();
                boolean isLinkedCorp = weComModel.getIsLinkedCorp();

                // 支持同步
                if (qyhIsSyn == 1 && !isLinkedCorp) {
                    String agentId = weComModel.getQyhAgentId();
                    /**
                     * 通讯录secret
                     */
                    final WxCpService wxCorpService = wxCpConfiguration.getCorpService(corpId);
                    /**
                     * 应用secret
                     */
                    final WxCpService wxCpService = wxCpConfiguration.getCpService(corpId, Integer.valueOf(agentId));
                    if (wxCorpService == null || wxCpService == null) {
                        synState = SYN_STATE_NO;
                        description = String.format("[%s]未找到对应corpId=[%s]、agentId=[%d]的配置", userFlag, corpId, agentId);
                        retMsg.put("code", false);
                        retMsg.put("msg", description);
                    } else {
                        // 执行同步
                        try {
                            retMsg = wxCpUserCreateOrUpdate(wxCpService, userEntity, retMsg, null);
                            if (retMsg.getBoolean("code")) {
                                WxCpUser wxCpUser = retMsg.getObject("wxCpUser", WxCpUser.class);

                                Long departId = 1L; // 根部门ID
                                boolean fetchChild = true; // 递归获取子部门下面的成员
                                Integer status = 0; // 获取全部员工
                                List<WxCpUser> wxCpUserList =
                                        wxCpService.getUserService().listByDepartment(departId, fetchChild, status);
                                if (StringUtils.isEmpty(wxCpUserList)) {
                                    synState = SYN_STATE_FAIL;
                                    description = String.format("[%s]获取企业微信用户列表接口数据为空", userFlag);
                                    retMsg.put("code", false);
                                    retMsg.put("msg", description);
                                } else {
                                    /**
                                     * 判断当前用户对应的第三方的合法性
                                     */
                                    retMsg = checkUserSysToQy(synThirdInfoEntity, wxCpUserList);
                                    if (!retMsg.getBoolean("code")) {
                                        final String flag = retMsg.getString("flag");
                                        if ("1".equals(flag) || "3".equals(flag)) {
                                            // flag:1 已同步但第三方上没对应的ID，需要删除原来的同步信息，再创建同步到企业微信、写入同步表
                                            // flag:3 未同步，需要创建同步到企业微信、写入同步表
                                            if ("1".equals(flag)) {
                                                synThirdInfoService.delete(synThirdInfoEntity);
                                            }
                                            opType = OBJECT_OP_ADD;
                                            synThirdInfoEntity = null;
                                            thirdObjId = "";

                                            // 往企业微信写入成员
                                            wxCorpService.getUserService().create(wxCpUser);
                                            // 同步成功
//                                            thirdObjId = corpId + TARGET_CHAR + wxCpUser.getUserId();
                                            thirdObjId = wxCpUser.getUserId();
                                            synState = SYN_STATE_OK;
                                            description = userFlag + "同步成功";
                                            retMsg.put("code", true);
                                            retMsg.put("error", description);
                                        }

                                        if ("2".equals(flag)) {
                                            // flag:2 已同步但第三方ID为空，需要创建同步到企业微信、修改同步表
                                            opType = OBJECT_OP_UPD;
                                            thirdObjId = "";

                                            // 往企业微信写入成员
                                            wxCorpService.getUserService().create(wxCpUser);
                                            // 同步成功
//                                            thirdObjId = corpId + TARGET_CHAR + wxCpUser.getUserId();
                                            thirdObjId = wxCpUser.getUserId();
                                            synState = SYN_STATE_OK;
                                            description = userFlag + "同步成功";
                                            retMsg.put("code", true);
                                            retMsg.put("error", description);
                                        }
                                    } else {
                                        // 更新同步表
                                        opType = OBJECT_OP_UPD;
                                        thirdObjId = synThirdInfoEntity.getThirdObjectId();
//                                        String userId = thirdObjId.substring(thirdObjId.lastIndexOf(TARGET_CHAR) + 1);
                                        String userId = thirdObjId;

                                        // 获取当前成员信息
                                        WxCpUser wxCpUserInfo = wxCpService.getUserService().getById(userId);
                                        if (ObjectUtil.isEmpty(wxCpUserInfo)) {
                                            // 同步失败,获取企业微信当前用户信息失败
                                            synState = SynThirdConsts.SYN_STATE_FAIL;
                                            description = userFlag + "获取企业微信当前用户信息失败";
                                        } else {
                                            // 要同步到企业微信的对象重新赋值
                                            retMsg = wxCpUserCreateOrUpdate(wxCpService, userEntity, retMsg,
                                                    wxCpUserInfo);
                                            wxCpUser = retMsg.getObject("wxCpUser", WxCpUser.class);

                                            // 往企业微信更新成员信息
                                            wxCorpService.getUserService().update(wxCpUser);
                                            // 同步成功
//                                            thirdObjId = corpId + TARGET_CHAR + wxCpUser.getUserId();
                                            thirdObjId = wxCpUser.getUserId();
                                            synState = SYN_STATE_OK;
                                            description = userFlag + "同步成功";
                                            retMsg.put("code", true);
                                            retMsg.put("error", description);
                                        }
                                    }


                                }

                            } else {
                                // 同步失败,原因：部门找不到对应的第三方ID、邮箱格式不合法
                                synState = SYN_STATE_FAIL;
                                description = userFlag + retMsg.getString("error");
                                retMsg.put("code", false);
                                retMsg.put("error", description);
                            }
                        } catch (WxErrorException e) {
                            // 同步失败
                            synState = SYN_STATE_FAIL;
                            description = userFlag + e.getError().toString();
                            retMsg.put("code", false);
                            retMsg.put("error", description);
                        }
                    }
                } else {
                    // 未设置单条同步,归并到未同步状态
                    // 未同步
                    synState = SYN_STATE_NO;
                    description = userFlag + "系统未设置单条同步";

                    retMsg.put("code", false);
                    retMsg.put("error", description);
                }
            }
        }

        if (ObjectUtil.isEmpty(synThirdInfoEntity) && !OBJECT_OP_ADD.equals(opType)) {
            // syn 无数据则新增
            opType = OBJECT_OP_ADD;
        }

        /**
         *  往同步写入本系统与第三方的对应信息
         *  更新同步表
         */
        saveSynThirdInfoEntity(opType, synThirdInfoEntity, currentUser, corpId, THIRD_TYPE_QY, DATA_TYPE_USER,
                userEntity.getId(), thirdObjId, synState, description);
        return retMsg;
    }

    @Override
    @Transactional(isolation = READ_COMMITTED)
    public JSONObject deleteUserSysToQy(UserInfo currentUser, boolean isBatch, String sysUserId, String corpId) {
        JSONObject retMsg = new JSONObject(); // 返回结果
        String deptFlag = "删除：";
        /**
         * 第一步：获取指定第三方工具、指定数据类型、对象ID的同步信息
         */
        SynThirdInfoEntity synThirdInfoEntity =
                synThirdInfoService.getInfoBySysObjId(THIRD_TYPE_QY, DATA_TYPE_USER, sysUserId);
        if (ObjectUtil.isNotEmpty(synThirdInfoEntity)) {
            String thirdObjId = synThirdInfoEntity.getThirdObjectId(); // 第三方id
            Integer synState = SYN_STATE_NO; // 同步状态，默认0未同步
            String description = ""; // 同步描述
            if (StringUtils.isEmpty(corpId)) {
                corpId = synThirdInfoEntity.getCropId();
            }
            /**
             * 第二步：获取到系统配置
             */
            final WeComModel weComModel = wxCpConfiguration.getWeComModel(corpId);
            if (weComModel == null) {
                synState = SYN_STATE_NO;
                description = String.format("[%s]未找到对应corpId=[%s]的配置", deptFlag, corpId);
                retMsg.put("code", false);
                retMsg.put("msg", description);
            } else {
                // 单条记录执行时,受开关限制isSynUser(系统配置>触发事件),(批量不受开关限制)
                int qyhIsSyn = isBatch ? 1 : weComModel.getQyhIsSynUser();
                boolean isLinkedCorp = weComModel.getIsLinkedCorp();
                // 支持同步
                if (qyhIsSyn == 1 && !isLinkedCorp) {
                    String agentId = weComModel.getQyhAgentId();
                    /**
                     * 通讯录secret
                     */
                    final WxCpService wxCorpService = wxCpConfiguration.getCorpService(corpId);
                    /**
                     * 应用secret
                     */
                    final WxCpService wxCpService = wxCpConfiguration.getCpService(corpId, Integer.valueOf(agentId));
                    if (wxCorpService == null || wxCpService == null) {
                        synState = SYN_STATE_NO;
                        description = String.format("[%s]未找到对应corpId=[%s]、agentId=[%d]的配置", deptFlag, corpId, agentId);
                        retMsg.put("code", false);
                        retMsg.put("msg", description);
                    } else {
                        try {
                            if (StringUtils.isNotEmpty(thirdObjId)) {
                                // 用户UserId必须有值,不能为空
//                                String userId = thirdObjId.substring(thirdObjId.lastIndexOf(TARGET_CHAR) + 1);
                                String userId = thirdObjId;
                                if (StringUtils.isEmpty(userId)) {
                                    // 同步失败
                                    synState = SYN_STATE_FAIL;
                                    description = deptFlag + "用户UserId是isEmpty";
                                    retMsg.put("code", false);
                                    retMsg.put("error", description);
                                } else {
                                    // 获取当前成员信息
                                    wxCorpService.getUserService().delete(userId);
                                    // 同步成功,直接删除同步表记录
                                    synThirdInfoService.delete(synThirdInfoEntity);
                                    retMsg.put("code", true);
                                    retMsg.put("msg", deptFlag + "同步成功");
                                    return retMsg;
                                }
                            } else {
                                // 根据企业微信ID找不到相应的信息,直接删除同步表记录
                                synThirdInfoService.delete(synThirdInfoEntity);
                                retMsg.put("code", true);
                                retMsg.put("msg", deptFlag + "同步成功");
                                return retMsg;
                            }
                        } catch (WxErrorException e) {
                            // 同步失败
                            synState = SYN_STATE_FAIL;
                            description = deptFlag + e.getError().getErrorMsg();
                            retMsg.put("code", false);
                            retMsg.put("error", description);
                        }
                    }
                } else {
                    // 未设置单条同步,归并到未同步状态
                    // 未同步
                    synState = SYN_STATE_NO;
                    description = deptFlag + "系统未设置单条同步";

                    retMsg.put("code", true);
                    retMsg.put("error", description);
                    retMsg.put("retDeptId", "0");
                }
            }

            // 往同步写入本系统与第三方的对应信息
            // 更新同步表
            saveSynThirdInfoEntity(OBJECT_OP_UPD, synThirdInfoEntity, currentUser, corpId, THIRD_TYPE_QY,
                    DATA_TYPE_USER, sysUserId, thirdObjId, synState, description);
        } else {
            retMsg.put("code", true);
            retMsg.put("msg", "第三方同步信息为空");
        }
        return retMsg;
    }


    /**
     * 根据部门的同步表信息判断同步情况
     *
     * @param synThirdInfoEntity 第三方表同步信息
     * @param wxCpUserList       企业微信用户列表
     * @return
     */
    public JSONObject checkUserSysToQy(SynThirdInfoEntity synThirdInfoEntity, List<WxCpUser> wxCpUserList) {
        JSONObject retMsg = new JSONObject();
        retMsg.put("code", true);
        retMsg.put("flag", "");
        retMsg.put("error", "");

        if (synThirdInfoEntity != null) {
            String thirdObjectId = synThirdInfoEntity.getThirdObjectId();
            if (StringUtils.isNotEmpty(thirdObjectId)) {
                String userId = thirdObjectId.substring(thirdObjectId.lastIndexOf(TARGET_CHAR) + 1);
                // 同步表存在企业微信ID,仍需要判断企业微信上有没此用户
                if (wxCpUserList.stream().filter(t -> t.getUserId().equals(userId)).count() == 0 ? true : false) {
                    retMsg.put("code", false);
                    retMsg.put("flag", "1");
                    retMsg.put("error", "企业微信不存在同步表对应的用户ID!");
                }
            } else {
                // 同步表的企业微信ID为空
                retMsg.put("code", false);
                retMsg.put("flag", "2");
                retMsg.put("error", "同步表中用户对应的第三方ID为空!");
            }
        } else {
            // 上级用户未同步
            retMsg.put("code", false);
            retMsg.put("flag", "3");
            retMsg.put("error", "用户未同步!");
        }

        return retMsg;
    }

    @Override
    public boolean triggeringCondition(UserEntity oldUser, UserEntity newUser) {
        if (ObjectUtil.isNotEmpty(oldUser) && ObjectUtil.isNotEmpty(newUser)) {
            if (!StringUtils.equals(oldUser.getRealName(), newUser.getRealName())) {
                // RealName变更时触发
                return true;
            } else if (!StringUtils.equals(oldUser.getNickName(), newUser.getNickName())) {
                // NickName变更时触发
                return true;
            } else if (!StringUtils.equals(oldUser.getMobilePhone(), newUser.getMobilePhone())) {
                // MobilePhone变更时触发
                return true;
            } else if (!StringUtils.equals(oldUser.getTelePhone(), newUser.getTelePhone())) {
                // TelePhone变更时触发
                return true;
            } else if (!StringUtils.equals(oldUser.getOrganizeId(), newUser.getOrganizeId())) {
                // OrganizeId变更时触发
                return true;
            } else if (!StringUtils.equals(oldUser.getPositionId(), newUser.getPositionId())) {
                // PositionId变更时触发
                return true;
            } else if (!(oldUser.getGender().intValue() == newUser.getGender().intValue())) {
                // Gender变更时触发
                return true;
            } else if (!StringUtils.equals(oldUser.getEmail(), newUser.getEmail())) {
                // Email变更时触发
                return true;
            } else if (!StringUtils.equals(oldUser.getCode(), newUser.getCode())) {
                // Code变更时触发
                return true;
            } else if (ObjectUtil.isNotEmpty(oldUser.getSortCode())&&ObjectUtil.isNotEmpty(newUser.getSortCode())
                    &&oldUser.getSortCode().longValue() != newUser.getSortCode().longValue()) {
                // SortCode变更时触发
                return true;
            } else if (!StringUtils.equals(oldUser.getManagerId(), newUser.getManagerId())) {
                // ManagerId变更时触发
                return true;
            } else if (oldUser.getEnabledMark().intValue() != newUser.getEnabledMark().intValue()) {
                // EnabledMark变更时触发
                return true;
            }
        }
        return false;
    }

}
