package com.bringspring.system.msgcenter.controller;

import cn.hutool.core.util.ObjectUtil;
import com.bringspring.common.base.ActionResult;
import com.bringspring.common.exception.DataException;
import com.bringspring.common.util.StringUtils;
import com.bringspring.system.msgcenter.model.BusinessKeys;
import com.bringspring.system.msgcenter.model.TaskMsg;
import com.bringspring.system.msgcenter.service.McMsgSendService;
import com.bringspring.system.msgcenter.util.SendTaskMsgUtils;
import com.bringspring.system.permission.util.BaseDataUtil;
import io.swagger.annotations.Api;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

import javax.validation.Valid;
import java.util.concurrent.Executor;

import static com.bringspring.system.msgcenter.constant.DictType.TASK_CODE;

@Slf4j
@RestController
@Api(tags = "业务消息发送接口服务", value = "messageService")
@RequestMapping("/api/outer/messageService")
public class MsgCenterService {
    /**
     * 取出线程池
     */
    @Autowired
    private Executor threadPoolExecutor;
    @Autowired
    private SendTaskMsgUtils sendTaskMsgUtils;

    @Autowired
    private McMsgSendService mcMsgSendService;
    @Autowired
    private BaseDataUtil baseDataUtil;

    /**
     * 业务消息发送
     *
     * @param taskMsg
     * @return
     */
    @PostMapping("/taskMsg/send")
    public ActionResult create(@RequestBody @Valid TaskMsg taskMsg) throws DataException {
        final String operation = "业务消息发送";
        if (ObjectUtil.isEmpty(taskMsg)) {
            return buildFailedResult(operation, "请求参数不能为空");
        }

        // 分步验证参数
        String error = validateTaskCode(taskMsg.getTaskCode(), operation);
        if (error != null) return ActionResult.fail(error);

        error = validateSendCode(taskMsg.getSendCode(), operation);
        if (error != null) return ActionResult.fail(error);

        error = validateBusinessKey(taskMsg.getBusinessKey(), operation);
        if (error != null) return ActionResult.fail(error);

        error = validateReceivers(taskMsg.getToUserIds(), taskMsg.getToDeptIds(), operation);
        if (error != null) return ActionResult.fail(error);

        // 异步处理
        submitAsyncSendTask(taskMsg);
        return ActionResult.success("发送成功");
    }

    /**
     * 提交异步发送任务（明确 Runnable 泛型）
     */
    private void submitAsyncSendTask(TaskMsg taskMsg) {
        threadPoolExecutor.execute(() -> {
            try {
                sendTaskMsgUtils.sendTaskMsg(taskMsg);
            } catch (Exception e) {
                log.error("消息发送失败: {}", e.getMessage(), e);
            }
        });
    }

    /**
     * 统一构建失败响应
     */
    private ActionResult buildFailedResult(String operation, String reason) {
        String errorMsg = String.format("%s，失败：%s", operation, reason);
        log.error(errorMsg);
        return ActionResult.fail(errorMsg);
    }

    //----------------------------- 验证方法 -----------------------------
    private String validateTaskCode(String taskCode, String operation) {
        if (StringUtils.isEmpty(taskCode)) {
            return buildFailedMsg(operation, "taskCode不能为空");
        }
        if (StringUtils.isEmpty(baseDataUtil.getDictName(taskCode, TASK_CODE.getCode()))) {
            return buildFailedMsg(operation, "taskCode未授权");
        }
        return null;
    }

    private String validateSendCode(String sendCode, String operation) {
        if (StringUtils.isEmpty(sendCode)) {
            return buildFailedMsg(operation, "sendCode不能为空");
        }
        if (ObjectUtil.isEmpty(mcMsgSendService.getValidInfo(sendCode))) {
            return buildFailedMsg(operation, "sendCode不存在或禁用");
        }
        return null;
    }

    private String validateBusinessKey(BusinessKeys businessKey, String operation) {
        if (ObjectUtil.isEmpty(businessKey)) {
            return buildFailedMsg(operation, "businessKey不能为空");
        }
        return null;
    }

    private String validateReceivers(String userIds, String deptIds, String operation) {
        if (StringUtils.isEmpty(userIds) && StringUtils.isEmpty(deptIds)) {
            return buildFailedMsg(operation, "无效的接收人");
        }
        return null;
    }

    /**
     * 统一构建错误消息
     */
    private String buildFailedMsg(String operation, String reason) {
        return String.format("%s，失败：%s", operation, reason);
    }
}
