package com.bringspring.system.permission.service;

import com.baomidou.mybatisplus.extension.service.IService;
import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.Pagination;
import com.bringspring.system.permission.entity.OrganizeEntity;
import com.bringspring.system.permission.exception.PermissionException;
import com.bringspring.system.permission.model.organize.OrganizeImportModel;
import com.bringspring.system.permission.model.organize.OrganizeImportVO;
import com.bringspring.system.permission.model.organize.OrganizeListVO;
import com.bringspring.system.permission.model.organize.OrganizePagination;

import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 组织机构
 *
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2017年9月26日 上午9:18
 */
public interface OrganizeService extends IService<OrganizeEntity> {

    /**
     * 组织列表
     *
     * @param idAll   组织id集合
     * @param keyWord 关键字过滤名称或编码
     * @return
     */
    List<OrganizeEntity> getListAll(List<String> idAll, String keyWord);

    /**
     * 查询所有公司列表
     *
     * @param
     * @return
     */
    List<OrganizeEntity> getAllCompanyList(String keyWord);

    /**
     * 智行平台根据权限查询公司列表
     *
     * @param
     * @return
     */
    List<OrganizeEntity> getAllCompanyDataPermission(OrganizePagination organizePagination);

    /**
     * 根据权限查询组织数据
     *
     * @param organizePagination 菜单ID menuId
     *                           查询类型 公司(company)\部门（department）
     *                           PermissionConst.COMPANY
     * @return
     */
    List<OrganizeEntity> getOrganizeByDataPermission(OrganizePagination organizePagination);

    /**
     * 根据数据权限获取公司列表
     *
     * @param modelId 关键字过滤名称或编码
     * @return
     */
    List<OrganizeEntity> getCompanyList(String modelId);

    /**
     * 根据所属部门id查询[直属公司]
     *
     * @param id 所属组织/部门id
     * @return
     */
    OrganizeEntity getDirectlyUnderCompanyById(String id);

    /**
     * 查询直属父级组织下属部门列表
     * category为department
     * @param id 所属组织/部门id
     * @return
     */
    List<OrganizeEntity> getCompanyById(String id);

    /**
     * 查询下属部门列表（不包含公司）
     * category为department
     *
     * @param id 所属组织/部门id
     * @return
     */
    List<OrganizeEntity> getParentIdList(String id);

    /**
     * 查询下属部门列表
     *
     * @return
     */
    List<OrganizeEntity> getParentIdListAsc(String id);

    /**
     * 获取部门主管为managerId的部门
     *
     * @return
     */
    List<OrganizeEntity> getListAscByManagerId(String managerId);


    /**
     * 查询所有列表
     * 根据排序码和创建时间排序
     *
     * @return
     */
    List<OrganizeEntity> getList();


    /**
     * 根据关键字查询组织列表【包含所有父级组织】
     *
     * @return
     */
    List<OrganizeEntity> getList(String keyWord);

    /**
     * 列表(有效的组织)
     *
     * @param enable 根据状态过滤
     * @return
     */
    List<OrganizeEntity> getListById(Boolean enable);

    /**
     * 列表
     *
     * @param fullName 组织名称
     * @return
     */
    OrganizeEntity getIdListByFullName(String fullName);


    /**
     * 获取部门名列表(在线开发转换数据使用)
     *
     * @return
     */
    List<OrganizeEntity> getOrgEntityList(List<String> idList, Boolean enable);

    /**
     * 获取部门名列表(在线开发转换数据使用)
     *
     * @return
     */
    List<OrganizeEntity> getOrgEntityList(Set<String> idList);

    /**
     * 获取没有编码的组织机构
     *
     * @return
     */
    List<OrganizeEntity> getOrgListCodeNull();

    /**
     * 获取redis存储的部门信息
     *
     * @return
     */
    List<OrganizeEntity> getOrgRedisList();

    /**
     * 信息
     *
     * @param id 主键值
     * @return
     */
    OrganizeEntity getInfo(String id);

    /**
     * 通过名称查询id
     *
     * @param fullName 名称
     * @return
     */
    OrganizeEntity getByFullName(String fullName);

    /**
     * 验证名称
     *
     * @param entity
     * @param isCheck  组织名称是否不分级判断
     * @param isFilter 是否需要过滤id
     * @return
     */
    boolean isExistByFullName(OrganizeEntity entity, boolean isCheck, boolean isFilter);

    /**
     * 获取父级id
     *
     * @param organizeId           组织id
     * @param organizeParentIdList 父级id集合
     */
    void getOrganizeIdTree(String organizeId, List<String> organizeParentIdList);

    /**
     * 获取公司id
     *
     * @param organizeId           组织id
     * @param organizeParentIdList 父级id集合
     */
    void getCompanyId(String organizeId, List<String> organizeParentIdList);

    /**
     * 获取父级id
     *
     * @param entity 组织
     */
    String getOrganizeIdTreeALL(OrganizeEntity entity);

    /**
     * 获取第一个组织id
     *
     * @param entity 组织
     */
    String getOrganizeIds(OrganizeEntity entity);

    /**
     * 获取父级id
     *
     * @param organizeParentIdList 父级id集合
     */
    void getOrganizeId(String organizeId, List<OrganizeEntity> organizeParentIdList);

    /**
     * 验证编码
     *
     * @param enCode
     * @param id
     * @return
     */
    boolean isExistByEnCode(String enCode, String id);

    /**
     * 部门或公司状态状态改为停用，下面有员工，不能修改
     *
     * @param id
     * @return
     */
    boolean isEnabledMark(String id);

    /**
     * 创建
     *
     * @param entity 实体对象
     */
    void create(OrganizeEntity entity) throws PermissionException;

    /**
     * 创建部门
     *
     * @param entity 实体对象
     */
    void createDepartment(OrganizeEntity entity) throws PermissionException;


    /**
     * 批量创建或新增
     *
     * @param entity 实体对象
     */
    void batchSaveOrUpdateBatch(List<OrganizeEntity> entity);

    /**
     * 更新组织
     *
     * @param id     主键值
     * @param entity 实体对象
     */
    boolean update(String id, OrganizeEntity entity);

    /**
     * 封装IService的updateById方法
     * 注解处理缓存@CacheEvict(value = "OrganizeCache", allEntries = true)
     *
     * @param entity
     * @return
     */
    Boolean update(OrganizeEntity entity);

    /**
     * 更新组织
     *
     * @param id     主键值
     * @param entity 实体对象
     */
    boolean updateDepartment(String id, OrganizeEntity entity);

    /**
     * 通过父级id修改父级组织树
     *
     * @param entity
     * @param category
     */
    void update(OrganizeEntity entity, String category);

    /**
     * 判断是否允许删除
     *
     * @param orgId 主键值
     * @return
     */
    String allowDelete(String orgId);

    /**
     * 删除
     *
     * @param orgId 实体对象
     */
    ActionResult<String> delete(String orgId);

    /**
     * 删除
     *
     * @param orgId      实体对象
     * @param actionType 操作类型
     * @return
     */
    ActionResult<String> delete(String orgId, String actionType);

    /**
     * 逻辑删除删除
     *
     * @param orgId 实体对象
     * @return
     */
    boolean deleteDepartmentByLogical(String orgId) throws PermissionException;

    /**
     * 上移
     *
     * @param id 主键值
     */
    boolean first(String id);

    /**
     * 下移
     *
     * @param id 主键值
     */
    boolean next(String id);

    /**
     * 获取名称
     *
     * @return
     */
    List<OrganizeEntity> getOrganizeName(List<String> id);

    /**
     * 获取名称
     *
     * @return
     */
    List<OrganizeEntity> getOrganizeNameSort(List<String> id);

    /**
     * 根据id查询子部门
     *
     * @param organizeParentId 父id
     * @return List<String> 接收子结构
     */
    List<String> getOrganize(String organizeParentId);

    /**
     * @param organizeParentId 父id
     * @return List<String> 接收子结构
     */
    List<String> getOrganizeByOraParentId(String organizeParentId);

    /**
     * 获取所有当前用户的组织及子组织
     *
     * @param organizeId
     * @return
     */
    List<String> getUnderOrganizations(String organizeId);

    /**
     * 通过名称获取组织列表
     *
     * @param fullName
     * @return
     */
    List<OrganizeEntity> getListByFullName(String fullName);

    /**
     * 通过id判断是否有子集
     *
     * @param id 主键
     * @return
     */
    List<OrganizeEntity> getListByParentId(String id);

    /**
     * 获取用户所有所在组织
     *
     * @return 组织对象集合
     */
    List<OrganizeEntity> getAllOrgByUserId(String userId);

    /**
     * 通过组织id树获取名称
     *
     * @param orgIdTree 组织id树
     * @param regex     分隔符
     * @return 组织对象集合
     */
    String getFullNameByOrgIdTree(String orgIdTree, String regex);

    /**
     * 模糊查询部门祖宗ID
     * 根据排序码和创建时间倒序
     *
     * @return
     */
    List<OrganizeEntity> getListLikeOrgIdTree(String orgId);

    /**
     * 根据直属公司查询部门
     * 根据排序码和创建时间倒序
     *
     * @return
     */
    List<OrganizeEntity> getListByCompanyIds(List<String> companyIds);

    /**
     * 根据用户权限获取组织信息
     *
     * @return 菜单id
     */
    List<OrganizeEntity> getOrganizeListDataPermission(String menuId);

    /**
     * 智行平台根据数据权限获取部门列表
     *
     * @return 公司id
     */
    List<OrganizeEntity> getListDepartmentDataPermission(String companyId, String menuId);

    /**
     * 智行平台根据数据权限获取全部部门列表（包括下级列表） 部门管理菜单
     *
     * @return 全部公司id
     */
    List<OrganizeEntity> getListAllJurisdiction(List<String> childId, Pagination pagination);


    /**
     * 根据id查询直属公司
     *
     * @param id 机构id
     * @return
     */
    OrganizeEntity getCompanyListById(String id);

    /**
     * 校验编码code的唯一
     *
     * @return 结果
     */
    public boolean checkCodeUnique(String code, String id);

    /**
     * 获取组织树形数据
     *
     * @param parentId
     * @return
     */
    List<OrganizeListVO> getListTree(String parentId);

    OrganizeEntity getByEnCode(String enCode);

    /**
     * 导入预览
     */
    Map<String, Object> importPreview(List<OrganizeImportModel> importList);

    /**
     * 导入数据
     */
    OrganizeImportVO importData(List<OrganizeImportModel> dataList, String menuId) throws Exception;
}
