package com.bringspring.system.permission.service.impl;

import com.baomidou.dynamic.datasource.annotation.DSTransactional;
import com.bringspring.common.base.UserInfo;
import com.bringspring.common.util.RandomUtil;
import com.bringspring.common.util.StringUtils;
import com.bringspring.common.util.UserProvider;
import com.bringspring.system.base.exception.ExcelValidationException;
import com.bringspring.system.permission.entity.OrganizeEntity;
import com.bringspring.system.permission.model.organize.OrganizeImportModel;
import com.bringspring.system.permission.model.organize.OrganizeImportVO;
import com.bringspring.system.permission.service.OrganizeImportService;
import com.bringspring.system.permission.service.OrganizeService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

import static com.bringspring.system.permission.constant.PermissionConst.COMPANY;
import static com.bringspring.system.permission.constant.PermissionConst.DEPARTMENT;


/**
 * 组织导入服务实现
 */
@Slf4j
@Service
public class OrganizeImportServiceImpl implements OrganizeImportService {

    @Autowired
    private UserProvider userProvider;

    @Autowired
    private OrganizeService organizeService;

    @Override
    public Map<String, Object> importPreview(List<OrganizeImportModel> organizeList) {
        List<Map<String, Object>> dataRow = new ArrayList<>();
        List<Map<String, Object>> columns = new ArrayList<>();

        if (organizeList != null) {
            for (OrganizeImportModel model : organizeList) {
                Map<String, Object> dataRowMap = new HashMap<>();
                dataRowMap.put("parentId", model.getParentId());
                dataRowMap.put("organizeId", model.getOrganizeId());
                dataRowMap.put("category", model.getCategory());
                dataRowMap.put("fullName", model.getFullName());
                dataRowMap.put("enCode", model.getEnCode());
                dataRowMap.put("sortCode", model.getSortCode());
                dataRowMap.put("description", model.getDescription());
                dataRow.add(dataRowMap);
            }

            // 设置列信息
            String[] columnNames = {"上级组织", "直属组织", "组织类型", "组织名称", "组织编码", "排序", "说明"};
            for (int i = 0; i < columnNames.length; i++) {
                Map<String, Object> columnsMap = new HashMap<>();
                columnsMap.put("AllowDBNull", true);
                columnsMap.put("Caption", columnNames[i]);
                columnsMap.put("ColumnName", columnNames[i]);
                columnsMap.put("DataType", "System.String, mscorlib, Version=4.0.0.0, Culture=neutral, PublicKeyToken=b77a5c561934e089");
                columns.add(columnsMap);
            }
        }

        Map<String, Object> map = new HashMap<>();
        map.put("dataRow", dataRow);
        map.put("columns", columns);
        return map;
    }

    @Override
    @DSTransactional
    @CacheEvict(value = "OrganizeCache", allEntries = true)
    public OrganizeImportVO importData(List<OrganizeImportModel> dataList, String menuId) throws Exception {
        UserInfo userInfo = userProvider.get();

        // 预处理：解析上级组织信息
        dataList.forEach(OrganizeImportModel::parseParentInfo);

        // 获取所有现有组织数据
        List<OrganizeEntity> allOrganizes = organizeService.list();
        Map<String, OrganizeEntity> existingOrganizeByCode = new HashMap<>();
        allOrganizes.forEach(org -> existingOrganizeByCode.put(org.getEnCode(), org));

        Map<String, String> organizeCodeToIdMap = allOrganizes.stream()
                .collect(Collectors.toMap(OrganizeEntity::getEnCode, OrganizeEntity::getId));

        // 按层级组织数据
        Map<Integer, List<OrganizeImportModel>> organizedData = organizeByLevel(dataList, existingOrganizeByCode, organizeCodeToIdMap);

        int successCount = 0;
        int failCount = 0;
        List<OrganizeImportModel> errorList = new ArrayList<>();

        // 按层级处理组织数据
        for (int level = 0; level < organizedData.size(); level++) {
            List<OrganizeImportModel> levelData = organizedData.get(level);
            if (levelData == null) continue;

            for (OrganizeImportModel model : levelData) {
                try {
                    // 校验数据
                    validateOrganizeData(model, existingOrganizeByCode, organizeCodeToIdMap);

                    // 处理组织（新增或更新）
                    OrganizeEntity entity = processOrganizeEntity(model, userInfo, existingOrganizeByCode, organizeCodeToIdMap);

                    // 更新现有组织映射
                    existingOrganizeByCode.put(entity.getEnCode(), entity);
                    // 更新编码到ID的映射
                    organizeCodeToIdMap.put(entity.getEnCode(), entity.getId());

                    successCount++;
                } catch (ExcelValidationException e) {
                    handleImportError(model, e, errorList);
                    failCount++;
                }
            }
        }

        // 批量更新 organizeIdTree（确保层级关系正确）
        if (successCount > 0) {
            try {
                batchUpdateOrganizeIdTree(existingOrganizeByCode);
            } catch (Exception e) {
                log.error("批量更新organizeIdTree失败", e);
            }
        }

        return buildImportResult(successCount, failCount, errorList);
    }

    /**
     * 按层级组织数据
     */
    private Map<Integer, List<OrganizeImportModel>> organizeByLevel(List<OrganizeImportModel> dataList,
                                                                    Map<String, OrganizeEntity> existingOrganizeByCode,
                                                                    Map<String, String> organizeCodeToIdMap) {
        Map<Integer, List<OrganizeImportModel>> result = new HashMap<>();

        // 第一层：没有上级组织的
        List<OrganizeImportModel> rootLevel = dataList.stream()
                .filter(org -> StringUtils.isEmpty(org.getParentId()))
                .collect(Collectors.toList());
        result.put(0, rootLevel);

        // 构建后续层级
        int currentLevel = 0;
        Set<String> processedCodes = new HashSet<>();

        // 添加已存在的根组织编码
        rootLevel.forEach(org -> {
            if (existingOrganizeByCode.containsKey(org.getEnCode())) {
                processedCodes.add(org.getEnCode());
            }
        });

        List<OrganizeImportModel> remaining = new ArrayList<>(dataList);
        remaining.removeAll(rootLevel);

        while (!remaining.isEmpty()) {
            currentLevel++;
            List<OrganizeImportModel> currentLevelData = new ArrayList<>();
            Iterator<OrganizeImportModel> iterator = remaining.iterator();

            while (iterator.hasNext()) {
                OrganizeImportModel model = iterator.next();
                if (isParentProcessed(model, processedCodes, existingOrganizeByCode, organizeCodeToIdMap)) {
                    currentLevelData.add(model);
                    iterator.remove();
                    processedCodes.add(model.getEnCode());
                }
            }

            if (!currentLevelData.isEmpty()) {
                result.put(currentLevel, currentLevelData);
            } else {
                // 无法找到上级组织的记录，直接添加到当前层级并标记错误
                for (OrganizeImportModel model : remaining) {
                    model.setErrorMessage("上级组织不存在: " + model.getParentId());
                    currentLevelData.add(model);
                }
                result.put(currentLevel, currentLevelData);
                break;
            }

            // 防止无限循环
            if (currentLevel > 50) {
                // 处理剩余无法找到上级组织的记录
                for (OrganizeImportModel model : remaining) {
                    model.setErrorMessage("上级组织不存在或存在循环依赖: " + model.getParentId());
                    currentLevelData.add(model);
                }
                break;
            }
        }

        return result;
    }

    /**
     * 检查上级组织是否已处理
     */
    private boolean isParentProcessed(OrganizeImportModel model, Set<String> processedCodes,
                                      Map<String, OrganizeEntity> existingOrganizeByCode,
                                      Map<String, String> organizeCodeToIdMap) {
        if (StringUtils.isEmpty(model.getParentId())) {
            return true; // 根组织
        }

        // 优先使用编码查找
        if (StringUtils.isNotEmpty(model.getParentEnCode())) {
            // 检查是否在已处理的编码中
            if (processedCodes.contains(model.getParentEnCode())) {
                return true;
            }
            // 检查是否在现有组织中
            if (organizeCodeToIdMap.containsKey(model.getParentEnCode())) {
                return true;
            }
        }

        // 备用：使用名称查找（注意名称可能重复，这里只做简单匹配）
        if (StringUtils.isNotEmpty(model.getParentName())) {
            // 这里可以扩展名称匹配逻辑，但建议业务上使用编码
        }

        return false;
    }

    /**
     * 处理组织实体（新增或更新）
     */
    private OrganizeEntity processOrganizeEntity(OrganizeImportModel model, UserInfo userInfo,
                                                 Map<String, OrganizeEntity> existingOrganizeByCode,
                                                 Map<String, String> organizeCodeToIdMap) {
        OrganizeEntity entity;
        boolean isUpdate = false;

        // 检查是否已存在（通过编码）
        if (existingOrganizeByCode.containsKey(model.getEnCode())) {
            // 更新现有组织
            entity = existingOrganizeByCode.get(model.getEnCode());
            isUpdate = true;
        } else {
            // 新增组织
            entity = new OrganizeEntity();
            entity.setId(RandomUtil.uuId());
            entity.setCreatorUserId(userInfo.getUserId());
            entity.setCreatorTime(new Date());
        }

        // 设置/更新组织信息
        entity.setFullName(model.getFullName());
        entity.setShortName(model.getFullName()); // 简称为全称
        entity.setEnCode(model.getEnCode());
        entity.setCategory(getCategoryMap().get(model.getCategory()));
        entity.setSortCode(model.getSortCode() != null ? model.getSortCode() : 0);
        entity.setDescription(model.getDescription());
        entity.setEnabledMark(1); // 默认启用
        entity.setDeleteMark(0); // 默认未删除

        // 设置上级组织ID
        String newParentId = determineParentId(model, existingOrganizeByCode, organizeCodeToIdMap);
        entity.setParentId(newParentId);

        // 直属组织
        String organizeId = determineParentIdWithDirectOrganize(model, existingOrganizeByCode, organizeCodeToIdMap);
        entity.setOrganizeId(organizeId);

        if (isUpdate) {
            entity.setLastModifyUserId(userInfo.getUserId());
            entity.setLastModifyTime(new Date());
            organizeService.updateById(entity);
        } else {
            organizeService.save(entity);
        }

        return entity;
    }

    /**
     * 确定上级组织ID（支持直属组织）
     */
    private String determineParentIdWithDirectOrganize(OrganizeImportModel model,
                                                       Map<String, OrganizeEntity> existingOrganizeByCode,
                                                       Map<String, String> organizeCodeToIdMap) {

        if (StringUtils.isEmpty(model.getOrganizeId())) {
            return "1"; // 根组织
        }

        // 优先使用编码查找
        if (StringUtils.isNotEmpty(model.getDirectOrganizeEnCode())) {
            // 在现有组织中查找
            if (existingOrganizeByCode.containsKey(model.getDirectOrganizeEnCode())) {
                return existingOrganizeByCode.get(model.getDirectOrganizeEnCode()).getId();
            }
            // 在编码到ID的映射中查找
            if (organizeCodeToIdMap.containsKey(model.getDirectOrganizeEnCode())) {
                return organizeCodeToIdMap.get(model.getDirectOrganizeEnCode());
            }
        }

        // 备用：使用名称查找（注意名称可能重复）
        if (StringUtils.isNotEmpty(model.getDirectOrganizeName())) {
            // 这里可以实现名称匹配逻辑，但建议业务上使用编码
            // 由于名称可能重复，这里需要更复杂的逻辑
        }

        throw new ExcelValidationException("无法找到直属组织: " + model.getOrganizeId());
    }

    /**
     * 确定上级组织ID
     */
    private String determineParentId(OrganizeImportModel model,
                                     Map<String, OrganizeEntity> existingOrganizeByCode,
                                     Map<String, String> organizeCodeToIdMap) {
        if (StringUtils.isEmpty(model.getParentId())) {
            return "-1"; // 根组织
        }

        // 优先使用编码查找
        if (StringUtils.isNotEmpty(model.getParentEnCode())) {
            // 在现有组织中查找
            if (existingOrganizeByCode.containsKey(model.getParentEnCode())) {
                return existingOrganizeByCode.get(model.getParentEnCode()).getId();
            }
            // 在编码到ID的映射中查找
            if (organizeCodeToIdMap.containsKey(model.getParentEnCode())) {
                return organizeCodeToIdMap.get(model.getParentEnCode());
            }
        }

        // 备用：使用名称查找（注意名称可能重复）
        if (StringUtils.isNotEmpty(model.getParentName())) {
            // 这里可以实现名称匹配逻辑，但建议业务上使用编码
            // 由于名称可能重复，这里需要更复杂的逻辑
        }

        throw new ExcelValidationException("无法找到上级组织: " + model.getParentId());
    }

    /**
     * 校验组织数据
     */
    private void validateOrganizeData(OrganizeImportModel model,
                                      Map<String, OrganizeEntity> existingOrganizeByCode,
                                      Map<String, String> organizeCodeToIdMap)
            throws ExcelValidationException {
        Map<String, String> errorMessages = new HashMap<>();

        // 组织名称校验
        String fullName = StringUtils.trim(model.getFullName());
        if (StringUtils.isEmpty(fullName)) {
            errorMessages.put("组织名称", "不能为空");
        } else {
            // 检查名称是否重复（排除自身）
//            boolean nameExists = organizeNameToIdMap.entrySet().stream()
//                    .anyMatch(entry -> entry.getKey().equals(fullName) &&
//                            !isSameOrganize(model, entry.getValue(), existingOrganizeByCode));
//            if (nameExists) {
//                errorMessages.put("组织名称", "组织名称已存在: " + fullName);
//            }
            // 检查同一父级下是否存在相同名称的组织
            boolean nameExistsInSameParent = isNameExistsInSameParentByCode(
                    fullName, model.getParentEnCode(), model.getEnCode(), existingOrganizeByCode);
            if (nameExistsInSameParent) {
                errorMessages.put("组织名称", "同一上级组织下已存在同名组织: " + fullName);
            }
        }

        // 组织编码校验
        String enCode = StringUtils.trim(model.getEnCode());
        if (StringUtils.isEmpty(enCode)) {
            errorMessages.put("组织编码", "不能为空");
        } else {
            // 检查编码是否重复（排除自身）
            boolean codeExists = existingOrganizeByCode.containsKey(enCode) &&
                    !isSameOrganize(model, existingOrganizeByCode.get(enCode).getId(), existingOrganizeByCode);
            if (codeExists) {
                errorMessages.put("组织编码", "组织编码已存在: " + enCode);
            }
        }

        // 组织类型校验
        String category = model.getCategory();
        if (StringUtils.isEmpty(category)) {
            errorMessages.put("组织类型", "不能为空");
        } else if (!getCategoryMap().containsKey(category)) {
            errorMessages.put("组织类型", "组织类型不存在: " + category);
        }

        //直属组织校验
        if (StringUtils.isNotEmpty(model.getOrganizeId())) {
            // 检查直属组织是否存在
            if (StringUtils.isNotEmpty(model.getDirectOrganizeEnCode())) {
                if (!existingOrganizeByCode.containsKey(model.getDirectOrganizeEnCode())) {
                    errorMessages.put("直属组织", "直属组织不存在: " + model.getOrganizeId());
                }
            } else {
                // 如果没有编码，尝试用名称查找
                boolean directOrganizeExists = existingOrganizeByCode.values().stream()
                        .anyMatch(org -> org.getFullName().equals(model.getDirectOrganizeName()));
                if (!directOrganizeExists) {
                    errorMessages.put("直属组织", "直属组织不存在: " + model.getOrganizeId());
                }
            }
        }

        if (!errorMessages.isEmpty()) {
            throw new ExcelValidationException(errorMessages);
        }
    }

    /**
     * 基于父级编码检查同一父节点下是否存在相同名称的组织
     */
    private boolean isNameExistsInSameParentByCode(String fullName, String parentEnCode,
                                                   String currentEnCode,
                                                   Map<String, OrganizeEntity> existingOrganizeByCode) {
        // 获取所有现有组织
        List<OrganizeEntity> allOrganizes = new ArrayList<>(existingOrganizeByCode.values());

        for (OrganizeEntity existing : allOrganizes) {
            // 排除自身（如果是更新操作）
            if (existing.getEnCode().equals(currentEnCode)) {
                continue;
            }

            // 检查名称是否相同
            if (!existing.getFullName().equals(fullName)) {
                continue;
            }

            // 获取现有组织的父级编码
            String existingParentEnCode = getParentEnCode(existing, existingOrganizeByCode);

            // 比较父级编码
            if (isSameParent(parentEnCode, existingParentEnCode)) {
                return true;
            }
        }

        return false;
    }

    /**
     * 获取组织的父级编码
     */
    private String getParentEnCode(OrganizeEntity organize, Map<String, OrganizeEntity> existingOrganizeByCode) {
        if (organize.getParentId() == null || "-1".equals(organize.getParentId())) {
            return null; // 根组织没有父级
        }

        // 查找父级组织
        for (OrganizeEntity parent : existingOrganizeByCode.values()) {
            if (parent.getId().equals(organize.getParentId())) {
                return parent.getEnCode();
            }
        }

        return null;
    }

    /**
     * 判断是否为同一父级
     */
    private boolean isSameParent(String parentEnCode1, String parentEnCode2) {
        // 两个都是根组织
        if (parentEnCode1 == null && parentEnCode2 == null) {
            return true;
        }

        // 其中一个为根组织
        if (parentEnCode1 == null || parentEnCode2 == null) {
            return false;
        }

        // 比较父级编码
        return parentEnCode1.equals(parentEnCode2);
    }

    /**
     * 检查是否为同一组织（用于更新时的重复校验）
     */
    private boolean isSameOrganize(OrganizeImportModel model, String organizeId,
                                   Map<String, OrganizeEntity> existingOrganizeByCode) {
        // 如果是新增，肯定不是同一组织
        if (!existingOrganizeByCode.containsKey(model.getEnCode())) {
            return false;
        }

        // 如果是更新，检查ID是否匹配
        OrganizeEntity existing = existingOrganizeByCode.get(model.getEnCode());
        return existing.getId().equals(organizeId);
    }

    /**
     * 构建导入结果
     */
    private OrganizeImportVO buildImportResult(int successCount, int failCount,
                                               List<OrganizeImportModel> errorList) {
        OrganizeImportVO vo = new OrganizeImportVO();
        vo.setSnum(successCount);
        vo.setFnum(failCount);
        vo.setResultType(failCount > 0 ? 1 : 0);
        vo.setFailResult(errorList);
        return vo;
    }

    /**
     * 获取组织分类映射
     */
    private Map<String, String> getCategoryMap() {
        Map<String, String> categoryMap = new HashMap<>();
        categoryMap.put("公司", COMPANY);
        categoryMap.put("部门", DEPARTMENT);
        // 可以根据需要添加更多类型
        categoryMap.put("集团", COMPANY);
        categoryMap.put("子公司", COMPANY);
        categoryMap.put("机构", DEPARTMENT);
        categoryMap.put("机关", DEPARTMENT);
        categoryMap.put("单位", DEPARTMENT);
        return categoryMap;
    }

    /**
     * 处理导入错误
     */
    private void handleImportError(OrganizeImportModel model, ExcelValidationException e,
                                   List<OrganizeImportModel> errorList) {
        StringBuilder errorMsg = new StringBuilder();
        if (e.getErrorMessages() != null && !e.getErrorMessages().isEmpty()) {
            int index = 0;
            int size = e.getErrorMessages().size();
            for (Map.Entry<String, String> error : e.getErrorMessages().entrySet()) {
                errorMsg.append(error.getKey()).append(": ").append(error.getValue());
                if (index != size - 1) {
                    errorMsg.append("; ");
                }
                index++;
            }
        } else {
            errorMsg.append(e.getMessage());
        }
        model.setErrorMessage(errorMsg.toString());
        errorList.add(model);
    }

    /**
     * 批量更新组织ID树路径
     */
    private void batchUpdateOrganizeIdTree(Map<String, OrganizeEntity> existingOrganizeByCode) {
        if (existingOrganizeByCode.isEmpty()) {
            return;
        }

        // 构建ID到实体的映射
        Map<String, OrganizeEntity> idToEntityMap = existingOrganizeByCode.values().stream()
                .collect(Collectors.toMap(OrganizeEntity::getId, Function.identity()));

        // 先处理根组织
        List<OrganizeEntity> rootOrganizes = existingOrganizeByCode.values().stream()
                .filter(org -> org.getParentId() == null || "-1".equals(org.getParentId()))
                .collect(Collectors.toList());

        for (OrganizeEntity root : rootOrganizes) {
            root.setOrganizeIdTree(root.getId());
        }

        // 按层级处理其他组织
        boolean hasChanges;
        int maxIterations = 50; // 防止无限循环
        int iteration = 0;

        do {
            hasChanges = false;
            iteration++;

            for (OrganizeEntity entity : existingOrganizeByCode.values()) {
                // 跳过已经处理过的组织
//                if (entity.getOrganizeIdTree() != null) {
//                    continue;
//                }

                String parentId = entity.getParentId();
                // 处理根组织
                if (parentId == null || "-1".equals(parentId)) {
                    entity.setOrganizeIdTree(entity.getId());
                    hasChanges = true;
                    continue;
                }

                // 查找父组织
                OrganizeEntity parent = idToEntityMap.get(parentId);
                if (parent != null && parent.getOrganizeIdTree() != null) {
                    // 父组织已处理，可以设置当前组织的 organizeIdTree
                    entity.setOrganizeIdTree(parent.getOrganizeIdTree() + "," + entity.getId());
                    hasChanges = true;
                }
            }
        } while (hasChanges && iteration < maxIterations);

        // 处理剩余未设置 organizeIdTree 的组织（可能是孤儿节点或循环引用）
        for (OrganizeEntity entity : existingOrganizeByCode.values()) {
            if (entity.getOrganizeIdTree() == null) {
                // 对于无法确定父级路径的组织，只设置自己的ID
                entity.setOrganizeIdTree(entity.getId());
            }
        }

        // 批量更新到数据库
        List<OrganizeEntity> toUpdate = existingOrganizeByCode.values().stream()
                .filter(org -> org.getOrganizeIdTree() != null)
                .collect(Collectors.toList());

        if (!toUpdate.isEmpty()) {
            organizeService.updateBatchById(toUpdate);
        }
    }
}
