package com.bringspring.system.permission.service.strategy;

import com.baomidou.mybatisplus.core.toolkit.ObjectUtils;
import com.bringspring.common.util.JsonUtil;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.context.ApplicationContext;
import org.springframework.stereotype.Component;

import java.util.*;
import java.util.stream.Collectors;

/**
 * 策略工厂类
 * <p>
 * 数据权限-条件策略工厂类，根据字段类型返回对应的策略类
 */
@Component
public class ConditionStrategyFactory implements InitializingBean {

    @Autowired
    private ApplicationContext applicationContext;

    private Map<Integer, ConditionStrategy> conditionStrategyMap = new HashMap<>();
    private List<FieldContentDateStrategy> fieldContentDateStrategyList = new ArrayList<>();
    private List<FieldContentNumberStrategy> fieldContentNumberStrategyList = new ArrayList<>();
    private List<FieldContentStringStrategy> fieldContentStringStrategyList = new ArrayList<>();
    private Collection<FieldContentTextStrategy> fieldContentTextStrategyList = new ArrayList<>();

    @Override
    public void afterPropertiesSet() throws Exception {
        // 从 IoC 容器中获取所有 FieldContentDateStrategy 类型的 Bean，并注册到 List 中
        fieldContentDateStrategyList.addAll(applicationContext.getBeansOfType(FieldContentDateStrategy.class).values());
        fieldContentDateStrategyList = fieldContentDateStrategyList.stream().sorted(Comparator.comparingInt(FieldContentDateStrategy::getOrder)).collect(Collectors.toList());
        // 从 IoC 容器中获取所有 FieldContentNumberStrategy 类型的 Bean，并注册到 List 中
        fieldContentNumberStrategyList.addAll(applicationContext.getBeansOfType(FieldContentNumberStrategy.class).values());
        fieldContentNumberStrategyList = fieldContentNumberStrategyList.stream().sorted(Comparator.comparingInt(FieldContentNumberStrategy::getOrder)).collect(Collectors.toList());
        // 从 IoC 容器中获取所有 FieldContentStringStrategy 类型的 Bean，并注册到 List 中
        fieldContentStringStrategyList.addAll(applicationContext.getBeansOfType(FieldContentStringStrategy.class).values());
        fieldContentStringStrategyList = fieldContentStringStrategyList.stream().sorted(Comparator.comparingInt(FieldContentStringStrategy::getOrder)).collect(Collectors.toList());
        // 从 IoC 容器中获取所有 FieldContentTextStrategy 类型的 Bean，并注册到 List 中
        fieldContentTextStrategyList.addAll(applicationContext.getBeansOfType(FieldContentTextStrategy.class).values());
        fieldContentTextStrategyList = fieldContentTextStrategyList.stream().sorted(Comparator.comparingInt(FieldContentTextStrategy::getOrder)).collect(Collectors.toList());

        // 从 IoC 容器中获取所有 ConditionStrategy 类型的 Bean，并注册到 Map 中
        Map<String, ConditionStrategy> beansOfType = applicationContext.getBeansOfType(ConditionStrategy.class);
        // 将实现类中的key方法的返回值作为键值，策略类作为值
        for (ConditionStrategy strategy : beansOfType.values()) {
            conditionStrategyMap.put(strategy.key(), strategy); // 注册策略
        }
    }

    public ConditionStrategy getConditionStrategy(Integer key) {
        return conditionStrategyMap.get(key);
    }

    public Object getFieldContentDate(String fieldContent, String symbol, String format) {
        FieldContentDateStrategy strategy = null;
        for (FieldContentDateStrategy s : fieldContentDateStrategyList) {
            List<Map<String, Object>> conditionOptions = s.getConditionOptions();
            for (Map<String, Object> option : conditionOptions) {
                if (option.containsKey("value") && String.valueOf(option.get("value")).equals(fieldContent)) {
                    strategy = s;
                    break;
                }
            }
            if (ObjectUtils.isNotEmpty(strategy)) {
                break;
            }
        }
        if (ObjectUtils.isNotEmpty(strategy)) {
            return strategy.convertFieldContent(fieldContent, symbol, format);
        }
        return null;
    }

    public Object getFieldContentNumber(String fieldContent, String symbol) {
        FieldContentNumberStrategy strategy = null;
        for (FieldContentNumberStrategy s : fieldContentNumberStrategyList) {
            List<Map<String, Object>> conditionOptions = s.getConditionOptions();
            for (Map<String, Object> option : conditionOptions) {
                if (option.containsKey("value") && String.valueOf(option.get("value")).equals(fieldContent)) {
                    strategy = s;
                    break;
                }
            }
            if (ObjectUtils.isNotEmpty(strategy)) {
                break;
            }
        }
        if (ObjectUtils.isNotEmpty(strategy)) {
            return strategy.convertFieldContent(fieldContent, symbol);
        }
        return null;
    }

    public Object getFieldContentText(String fieldContent, String symbol) {
        FieldContentTextStrategy strategy = null;
        for (FieldContentTextStrategy s : fieldContentTextStrategyList) {
            List<Map<String, Object>> conditionOptions = s.getConditionOptions();
            for (Map<String, Object> option : conditionOptions) {
                if (option.containsKey("value") && String.valueOf(option.get("value")).equals(fieldContent)) {
                    strategy = s;
                    break;
                }
            }
            if (ObjectUtils.isNotEmpty(strategy)) {
                break;
            }
        }
        if (ObjectUtils.isNotEmpty(strategy)) {
            return strategy.convertFieldContent(fieldContent, symbol);
        }
        return null;
    }

    public Object getFieldContentString(String fieldContent, String symbol) {
        FieldContentStringStrategy strategy = null;
        for (FieldContentStringStrategy s : fieldContentStringStrategyList) {
            List<Map<String, Object>> conditionOptions = s.getConditionOptions();
            for (Map<String, Object> option : conditionOptions) {
                if (option.containsKey("value") && String.valueOf(option.get("value")).equals(fieldContent)) {
                    strategy = s;
                    break;
                }
            }
            if (ObjectUtils.isNotEmpty(strategy)) {
                break;
            }
        }
        if (ObjectUtils.isNotEmpty(strategy)) {
            return strategy.convertFieldContent(fieldContent, symbol);
        }
        return null;
    }

    @Cacheable(value = "DataAuthorizeForm", keyGenerator = "cacheKeyGenerator", unless = "#result.size()==0")
    public Map<String, List<Map<String, Object>>> getDataAuthorizeForm() {
        Map<String, List<Map<String, Object>>> dataAuthorizeForm = new HashMap<>();

        List<Map<String, Object>> numberConditionOptions = new ArrayList<>();
        List<Map<String, Object>> dateConditionOptions = new ArrayList<>();
        List<Map<String, Object>> textConditionOptions = new ArrayList<>();
        List<Map<String, Object>> stringConditionOptions = new ArrayList<>();

        List<ConditionModel> numberList = JsonUtil.getJsonToList(fieldContentNumberStrategyList, ConditionModel.class);
        numberList.stream().forEach(strategy -> {
            List<Map<String, Object>> conditionOptions = strategy.getConditionOptions();
            conditionOptions.stream().forEach(option -> {
                option.put("symbolOptions", strategy.getSymbolOptions());
                numberConditionOptions.add(option);
            });
        });

        List<ConditionModel> dateList = JsonUtil.getJsonToList(fieldContentDateStrategyList, ConditionModel.class);
        dateList.stream().forEach(strategy -> {
            List<Map<String, Object>> conditionOptions = strategy.getConditionOptions();
            conditionOptions.stream().forEach(option -> {
                option.put("symbolOptions", strategy.getSymbolOptions());
                dateConditionOptions.add(option);
            });
        });

        List<ConditionModel> textList = JsonUtil.getJsonToList(fieldContentTextStrategyList, ConditionModel.class);
        textList.stream().forEach(strategy -> {
            List<Map<String, Object>> conditionOptions = strategy.getConditionOptions();
            conditionOptions.stream().forEach(option -> {
                option.put("symbolOptions", strategy.getSymbolOptions());
                textConditionOptions.add(option);
            });
        });

        stringConditionOptions.addAll(textConditionOptions);
        List<ConditionModel> stringList = JsonUtil.getJsonToList(fieldContentStringStrategyList, ConditionModel.class);
        stringList.stream().forEach(strategy -> {
            List<Map<String, Object>> conditionOptions = strategy.getConditionOptions();
            conditionOptions.stream().forEach(option -> {
                option.put("symbolOptions", strategy.getSymbolOptions());
                stringConditionOptions.add(option);
            });
        });

        dataAuthorizeForm.put("numberOptions", numberConditionOptions);
        dataAuthorizeForm.put("dataTimeOptions", dateConditionOptions);
        dataAuthorizeForm.put("textOptions", textConditionOptions);
        dataAuthorizeForm.put("organizeOptions", stringConditionOptions);
        return dataAuthorizeForm;
    }


}
