package com.bringspring.system.base.controller;

import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.vo.PageListVO;
import com.bringspring.common.base.vo.PaginationVO;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.common.util.StringUtils;
import com.bringspring.system.base.entity.LogEntity;
import com.bringspring.system.base.model.BurialPoint.VisitVO;
import com.bringspring.system.base.model.logs.*;
import com.bringspring.system.base.service.LogService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.collections4.map.HashedMap;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 系统日志
 *
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2017年9月27日 上午9:18
 */
@Api(tags = "系统日志", value = "Log")
@RestController
@RequestMapping("/api/system/Log")
public class LogController {

    @Autowired
    private LogService logService;


    /**
     * 获取系统日志信息
     *
     * @param category 主键值分类 1：登录日志，2.访问日志，3.操作日志，4.异常日志，5.请求日志
     * @return
     */
    @ApiOperation("获取系统日志列表")
    @GetMapping("/{category}")
    public ActionResult getInfoList(@PathVariable("category") String category, PaginationLogModel paginationTime) {
        if (StringUtils.isEmpty(category) || !StringUtils.isNumeric(category)) {
            return ActionResult.fail("获取失败");
        }
        List<LogEntity> list = logService.getList(Integer.parseInt(category), paginationTime);
        PaginationVO paginationVO = JsonUtil.getJsonToBean(paginationTime, PaginationVO.class);
        int i = Integer.parseInt(category);
        switch (i) {
            case 1:
                List<LoginLogVO> loginLogVOList = JsonUtil.getJsonToList(list, LoginLogVO.class);
                return ActionResult.page(loginLogVOList, paginationVO);
            case 3:
                List<HandleLogVO> handleLogVOList = JsonUtil.getJsonToList(list, HandleLogVO.class);
                for (int j = 0; j < handleLogVOList.size(); j++) {
                    handleLogVOList.get(j).setJson(list.get(j).getJsons());
                }
                return ActionResult.page(handleLogVOList, paginationVO);
            case 4:
                List<ErrorLogVO> errorLogVOList = JsonUtil.getJsonToList(list, ErrorLogVO.class);
                for (int j = 0; j < errorLogVOList.size(); j++) {
                    errorLogVOList.get(j).setJson(list.get(j).getJsons());
                }
                return ActionResult.page(errorLogVOList, paginationVO);
            case 5:
                List<RequestLogVO> requestLogVOList = JsonUtil.getJsonToList(list, RequestLogVO.class);
                return ActionResult.page(requestLogVOList, paginationVO);
            default:
                return ActionResult.fail("获取失败");
        }
    }

    /**
     * 批量删除系统日志
     *
     * @return
     */
    @ApiOperation("批量删除系统日志")
    @DeleteMapping
    public ActionResult delete(@RequestBody LogDelForm logDelForm) {
        boolean flag = logService.delete(logDelForm.getIds());
        if (flag == false) {
            return ActionResult.fail(MsgCode.FA003.get());
        }
        return ActionResult.success(MsgCode.SU003.get());
    }

    /**
     * 一键清空操作日志
     *
     * @return
     */
    @ApiOperation("一键清空操作日志")
    @DeleteMapping("/{type}")
    public ActionResult deleteHandelLog(@PathVariable("type") String type) {
        logService.deleteHandleLog(type);
        return ActionResult.success(MsgCode.SU005.get());
    }

    /**
     * 获取菜单名
     *
     * @return
     */
    @ApiOperation("获取菜单名")
    @GetMapping("/ModuleName")
    public ActionResult<List<Map<String, String>>> moduleName() {
        List<Map<String, String>> list = new ArrayList<>(16);
        Set<String> set = logService.queryList();
        for (String moduleName : set) {
            Map<String, String> map = new HashedMap<>(1);
            map.put("moduleName", moduleName);
            list.add(map);
        }
        return ActionResult.success(list);
    }


    @ApiOperation("根据请求日志统计近30分钟内，每分钟上，访问请求所属用户的用户数量")
    @GetMapping("/countUserGroupByCreatorTime/{category}")
    public ActionResult countUserGroupByCreatorTime(@PathVariable("category") String category, PaginationLogModel logModel) {
        ActionResult result = new ActionResult();
        List<Map<String, Object>> data = logService.selectUserGroupByCreatorTime(category, logModel);
        result.setData(data);
        result.setCode(200);
        result.setMsg(MsgCode.SU000.get());
        return result;
    }

    @ApiOperation("根据请求日志统计近30分钟内，每分钟上，访问请求所属模块的模块数量")
    @GetMapping("/countModuleGroupByCreatorTime/{category}")
    public ActionResult countModuleGroupByCreatorTime(@PathVariable("category") String category, PaginationLogModel logModel) {
        ActionResult result = new ActionResult();
        List<Map<String, Object>> data = logService.selectModuleGroupByCreatorTime(category, logModel);
        result.setData(data);
        result.setCode(200);
        result.setMsg(MsgCode.SU000.get());
        return result;
    }

    @ApiOperation("根据请求日志统计近30分钟内，每分钟上，访问请求数量")
    @GetMapping("/countRequestGroupByCreatorTime/{category}")
    public ActionResult countRequestGroupByCreatorTime(@PathVariable("category") String category, PaginationLogModel logModel) {
        ActionResult result = new ActionResult();
        List<Map<String, Object>> data = logService.selectGroupByCreatorTime(category, logModel);
        result.setData(data);
        result.setCode(200);
        result.setMsg(MsgCode.SU000.get());
        return result;
    }

    @ApiOperation("根据请求日志统计近30分钟内，每分钟上，混合数据")
    @GetMapping("/countMixedGroupByCreatorTime/{category}")
    public ActionResult countMixedGroupByCreatorTime(@PathVariable("category") String category, PaginationLogModel logModel) {
        ActionResult result = new ActionResult();
        List<Map<String, Object>> data = logService.selectMixedGroupByCreatorTime(category, logModel);
        result.setData(data);
        result.setCode(200);
        result.setMsg(MsgCode.SU000.get());
        return result;
    }

    @ApiOperation("根据请求日志统计近30分钟内，功能模块的请求量")
    @GetMapping("/countGroupByModule/{category}")
    public ActionResult countGroupByModule(@PathVariable("category") String category, PaginationLogModel logModel) {
        ActionResult result = new ActionResult();
        List<Map<String, Object>> data = logService.selectGroupByModule(category, logModel);
        result.setData(data);
        result.setCode(200);
        result.setMsg(MsgCode.SU000.get());
        return result;
    }
    /**
     * 用户访问功能次数（每分钟算一次）
     *
     * @return 结果
     */
    @ApiOperation("用户访问功能次数")
    @PostMapping("/userVisit")
    public ActionResult userVisit(@RequestBody VisitVO visitVO) {
        ActionResult result = new ActionResult();
        List<VisitVO> data = logService.userVisit(visitVO);
        result.setData(data);
        result.setCode(200);
        result.setMsg(MsgCode.SU000.get());
        PageListVO vo = new PageListVO();
        vo.setList(data);
        PaginationVO page = JsonUtil.getJsonToBean(visitVO, PaginationVO.class);
        vo.setPagination(page);
        return ActionResult.success(vo);
    }
    /**
     * 功能访问次数（每分钟算一次）
     *
     * @return 结果
     */
    @ApiOperation("功能访问次数")
    @PostMapping("/functionVisit")
    public ActionResult functionVisit(@RequestBody VisitVO visitVO) {
        ActionResult result = new ActionResult();
        List<VisitVO> data = logService.functionVisit(visitVO);
        result.setData(data);
        result.setCode(200);
        result.setMsg(MsgCode.SU000.get());
        PageListVO vo = new PageListVO();
        vo.setList(data);
        PaginationVO page = JsonUtil.getJsonToBean(visitVO, PaginationVO.class);
        vo.setPagination(page);
        return ActionResult.success(vo);
    }
    /**
     * 用户访问总次数（每分钟算一次）
     *
     * @return 结果
     */
    @ApiOperation("用户访问总次数")
    @PostMapping("/functionUserVisit")
    public ActionResult functionUserVisit(@RequestBody VisitVO visitVO) {
        ActionResult result = new ActionResult();
        List<VisitVO> data = logService.functionUserVisit(visitVO);
        result.setData(data);
        result.setCode(200);
        result.setMsg(MsgCode.SU000.get());
        return result;
    }
    /**
     * 活跃用户数
     * 进行过功能操作的用户数量
     * @return 结果
     */
    @ApiOperation("活跃用户数")
    @PostMapping("/functionUserCountVisit")
    public ActionResult functionUserCountVisit(@RequestBody VisitVO visitVO) {
        ActionResult result = new ActionResult();
        List<VisitVO> data = logService.functionUserCountVisit(visitVO);
        result.setData(data);
        result.setCode(200);
        result.setMsg(MsgCode.SU000.get());
        return result;
    }
    /**
     * 功能列表
     * 时间段内使用过的功能列表,用户筛选选择
     * @return 结果
     */
    @ApiOperation("使用过的功能列表")
    @PostMapping("/functionList")
    public ActionResult functionList(@RequestBody VisitVO visitVO) {
        ActionResult result = new ActionResult();
        List<VisitVO> data = logService.functionList(visitVO);
        result.setData(data);
        result.setCode(200);
        result.setMsg(MsgCode.SU000.get());
        return result;
    }

    /**
     * 登录用户数据统计
     * @return 结果
     */
    @ApiOperation("登录用户数据统计")
    @PostMapping("/loginUserStatistics")
    public ActionResult loginUserStatistics(@RequestBody LoginUserStatisticsQuery loginUserStatisticsQuery) {
        ActionResult result = new ActionResult();
        LoginUserStatistics loginUserStatistics = logService.loginUserStatistics(loginUserStatisticsQuery);
        result.setData(loginUserStatistics);
        result.setCode(200);
        result.setMsg(MsgCode.SU000.get());
        return result;
    }
}
