package com.bringspring.system.permission.service.impl;


import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import com.baomidou.dynamic.datasource.annotation.DSTransactional;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import com.bringspring.common.base.UserInfo;
import com.bringspring.common.database.model.DataSourceModel;
import com.bringspring.common.database.model.dto.PreparedStatementDTO;
import com.bringspring.common.database.sql.SqlBase;
import com.bringspring.common.database.util.ConnUtil;
import com.bringspring.common.database.util.DbTypeUtil;
import com.bringspring.common.database.util.JdbcUtil;
import com.bringspring.common.util.*;
import com.bringspring.common.util.treeutil.ListToTreeUtil;
import com.bringspring.common.util.treeutil.SumTree;
import com.bringspring.common.util.treeutil.newtreeutil.TreeDotUtils;
import com.bringspring.system.base.entity.*;
import com.bringspring.system.base.model.button.ButtonModel;
import com.bringspring.system.base.model.column.ColumnModel;
import com.bringspring.system.base.model.form.ModuleFormModel;
import com.bringspring.system.base.model.module.ModuleModel;
import com.bringspring.system.base.model.resource.ResourceModel;
import com.bringspring.system.base.service.*;
import com.bringspring.system.permission.constant.AuthorizeConst;
import com.bringspring.system.permission.entity.AuthorizeEntity;
import com.bringspring.system.permission.entity.UserEntity;
import com.bringspring.system.permission.entity.UserRelationEntity;
import com.bringspring.system.permission.mapper.AuthorizeMapper;
import com.bringspring.system.permission.model.authorize.*;
import com.bringspring.system.permission.service.*;
import com.bringspring.system.permission.service.strategy.ConditionStrategy;
import com.bringspring.system.permission.service.strategy.ConditionStrategyFactory;
import lombok.Cleanup;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.ss.formula.functions.T;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.sql.Connection;
import java.util.*;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import static com.bringspring.common.constant.CacheConsts.LOGIN_CACHE;
import static com.bringspring.common.util.enums.SearchMethodEnum.And;

/**
 * 操作权限
 *
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2017年9月26日 上午9:18
 */
@Slf4j
@Service
@DSTransactional
public class AuthorizeServiceImpl extends ServiceImpl<AuthorizeMapper, AuthorizeEntity> implements AuthorizeService {

    @Autowired
    private UserRelationService userRelationService;
    @Autowired
    private DataSourceModel dataSourceUtils;
    @Autowired
    private UserProvider userProvider;
    @Autowired
    private CacheKeyUtil cacheKeyUtil;
    @Autowired
    private CacheUtil cacheUtil;
    @Autowired
    private PositionService positionService;
    @Autowired
    private UserService userService;
    @Autowired
    private RoleService roleService;
    @Autowired
    private OrganizeService organizeService;
    @Autowired
    private OrganizeAdministratorService organizeAdminIsTratorService;
    @Autowired
    private OrganizeRelationService organizeRelationService;
    @Autowired
    private ModuleService moduleService;
    @Autowired
    private ModuleButtonService buttonService;
    @Autowired
    private ModuleColumnService columnService;
    @Autowired
    private ModuleFormService formService;
    @Autowired
    private ModuleDataAuthorizeSchemeService schemeService;
    @Autowired
    private ConditionStrategyFactory conditionStrategyFactory;

    @Override
    public AuthorizeVO getAuthorize(boolean isCache) {
        // 是否从缓冲里面获取权限
        if (isCache) {
            return getCacheAuthor(userProvider.get());
        } else {
            return getAuthorize(userProvider.get());
        }
    }

    private AuthorizeVO getCacheAuthor(UserInfo userInfo) {
        // 是否从缓冲里面获取权限
        String cacheKey = cacheKeyUtil.getUserAuthorize() + userInfo.getUserId();
        if (!cacheUtil.exists(cacheKey)) {
            AuthorizeVO authorizeModel = getAuthorize(userInfo);
            if (authorizeModel.getModuleList().size() != 0) {
                cacheUtil.insert(LOGIN_CACHE, cacheKey, authorizeModel, 60);
            }
            return authorizeModel;
        } else {
            return JsonUtil.getJsonToBean(cacheUtil.query(LOGIN_CACHE, cacheKey).toString(), AuthorizeVO.class);
        }
    }


    @Override
    public AuthorizeVO getAuthorize(UserInfo userInfo) {
        List<ModuleModel> moduleList = new ArrayList<>();
        List<ButtonModel> buttonList = new ArrayList<>();
        List<ColumnModel> columnList = new ArrayList<>();
        List<ResourceModel> resourceList = new ArrayList<>();
        List<ModuleFormModel> formsList = new ArrayList<>();
        Boolean isAdmin = userInfo.getIsAdministrator();
        if (!isAdmin) {
            List<String> roleIds = new ArrayList<>();
            roleService.getRoleIdsByCurrentUser(userInfo.getDepartmentId()).forEach(role -> {
                //判断角色状态是否为有效，显示当前组织内角色的并集
                if (role != null) {
                    roleIds.add("'" + role + "'");
                }
            });
            if (roleIds.size() != 0) {
                String roleIdsStr = String.join(",", roleIds);
                moduleList = this.baseMapper.findModule(roleIdsStr);
                buttonList = this.baseMapper.findButton(roleIdsStr);
                columnList = this.baseMapper.findColumn(roleIdsStr);
                resourceList = this.baseMapper.findResource(roleIdsStr);
                formsList = this.baseMapper.findForms(roleIdsStr);
            }

        } else {
            moduleList = this.baseMapper.findModuleAdmin(1);
            buttonList = this.baseMapper.findButtonAdmin(1);
            columnList = this.baseMapper.findColumnAdmin(1);
            resourceList = this.baseMapper.findResourceAdmin(1);
            formsList = this.baseMapper.findFormsAdmin(1);
        }
        return new AuthorizeVO(moduleList, buttonList, columnList, resourceList, formsList);
    }

    @Override
    public AuthorizeVO getAuthorizeByRoleId(String roleId) {
        List<ModuleModel> moduleList = new ArrayList<>();
        List<ButtonModel> buttonList = new ArrayList<>();
        List<ColumnModel> columnList = new ArrayList<>();
        List<ResourceModel> resourceList = new ArrayList<>();
        List<ModuleFormModel> formsList = new ArrayList<>();

        moduleList = this.baseMapper.findModule(roleId);
        buttonList = this.baseMapper.findButton(roleId);
        columnList = this.baseMapper.findColumn(roleId);
        resourceList = this.baseMapper.findResource(roleId);
        formsList = this.baseMapper.findForms(roleId);


        return new AuthorizeVO(moduleList, buttonList, columnList, resourceList, formsList);
    }

    @Override
    public void save(String objectId, AuthorizeDataUpForm authorizeList) {
        SaveBatchForm form = JsonUtil.getJsonToBean(authorizeList, SaveBatchForm.class);
        form.setRoleIds(new String[]{objectId});
        this.saveBatch(form, false);
    }

    @Override
    public void saveBatch(SaveBatchForm saveBatchForm, boolean isBatch) {
        try {
            UserInfo userInfo = userProvider.get();
            String dbName = userInfo.getTenantDbConnectionString() == null ?
                    dataSourceUtils.getDbName() : userInfo.getTenantDbConnectionString();

            List<AuthorizeEntity> objectList = new ArrayList<>();
            List<AuthorizeEntity> authorizeList = new ArrayList<>();
            // 设置权限归属对象
            setEntity(saveBatchForm.getUserIds(), AuthorizeConst.USER, objectList, true);
            setEntity(saveBatchForm.getPositionIds(), AuthorizeConst.POSITION, objectList, true);
            setEntity(saveBatchForm.getRoleIds(), AuthorizeConst.ROLE, objectList, true);
            // 设置权限模块
            setEntity(saveBatchForm.getButton(), AuthorizeConst.BUTTON, authorizeList, false);
            setEntity(saveBatchForm.getModule(), AuthorizeConst.MODULE, authorizeList, false);
            setEntity(saveBatchForm.getColumn(), AuthorizeConst.COLUMN, authorizeList, false);
            setEntity(saveBatchForm.getResource(), AuthorizeConst.RESOURCE, authorizeList, false);
            setEntity(saveBatchForm.getForm(), AuthorizeConst.FROM, authorizeList, false);

            @Cleanup Connection conn = ConnUtil.getConn(dataSourceUtils, dbName);

            // 删除角色相关信息
            List<String> objectIdAll =
                    objectList.stream().map(AuthorizeEntity::getObjectId).collect(Collectors.toList());
            userService.delCurRoleUser(objectIdAll);
            if (!isBatch) {
                JdbcUtil.creUpDe(new PreparedStatementDTO(conn,
                        XSSEscape.escapeEmpty(SqlBase.getAuthorDelSql(objectIdAll))));
            }

            // 插入数据
            String sql = DbTypeUtil.checkOracle(dataSourceUtils) || DbTypeUtil.checkPostgre(dataSourceUtils) ?
                    SqlBase.INSERT_AUTHORIZE2 : SqlBase.INSERT_AUTHORIZE;
            List<PreparedStatementDTO> prepareDtoList = new LinkedList<>();
            for (int i = 0; i < objectList.size(); i++) {
                for (AuthorizeEntity entityItem : authorizeList) {
                    LinkedList<Object> paramList = new LinkedList<>();
                    paramList.add(RandomUtil.uuId());
                    paramList.add(entityItem.getItemType());
                    paramList.add(entityItem.getItemId());
                    paramList.add(objectList.get(i).getObjectType());
                    paramList.add(objectList.get(i).getObjectId());
                    paramList.add(i);
                    paramList.add(DateUtil.getNow());
                    paramList.add(userInfo.getUserId());
                    prepareDtoList.add(new PreparedStatementDTO(null, null, paramList));
                }
            }
            JdbcUtil.creUpDeBatchOneSql(conn, sql, prepareDtoList);
        } catch (Exception e) {
            e.printStackTrace();
            log.error("权限报错:" + e.getMessage());
        }
    }

    /**
     * 权限
     */
    private void setEntity(String[] ids, String type, List<AuthorizeEntity> entityList, Boolean objectFlag) {
        if (ids != null) {
            for (String id : ids) {
                AuthorizeEntity entity = new AuthorizeEntity();
                if (objectFlag) {
                    entity.setObjectType(type);
                    entity.setObjectId(id);
                } else {
                    entity.setItemType(type);
                    entity.setItemId(id);
                }
                entityList.add(entity);
            }
        }
    }

    @Override
    public List<AuthorizeEntity> getListByUserId(boolean isAdmin, String userId) {
        if (!isAdmin) {
            QueryWrapper<UserRelationEntity> queryWrapper = new QueryWrapper<>();
            queryWrapper.lambda().in(UserRelationEntity::getUserId, userId);
            List<UserRelationEntity> list = userRelationService.list(queryWrapper);
            List<String> userRelationList = list.stream().map(u -> u.getObjectId()).collect(Collectors.toList());
            userRelationList.add(userId);
            QueryWrapper<AuthorizeEntity> wrapper = new QueryWrapper<>();
            wrapper.lambda().in(AuthorizeEntity::getObjectId, userRelationList);
            return this.list(wrapper);
        } else {
            return new ArrayList<>();
        }
    }

    @Override
    public List<AuthorizeEntity> getListByObjectId(String objectId) {
        QueryWrapper<AuthorizeEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(AuthorizeEntity::getObjectId, objectId);
        return this.list(queryWrapper);
    }

    @Override
    public Boolean existByObjId(String objectId) {
        QueryWrapper<AuthorizeEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(AuthorizeEntity::getObjectId, objectId);
        queryWrapper.lambda().eq(AuthorizeEntity::getItemType, "module");
        return this.count(queryWrapper) > 0;
    }

    @Override
    public List<AuthorizeEntity> getListByObjectId(String objectId, String itemType) {
        QueryWrapper<AuthorizeEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(AuthorizeEntity::getObjectId, objectId);
        queryWrapper.lambda().eq(AuthorizeEntity::getItemType, itemType);
        return this.list(queryWrapper);
    }

    @Override
    public List<AuthorizeEntity> getListByObjectAndItem(String itemId, String objectType) {
        QueryWrapper<AuthorizeEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(AuthorizeEntity::getObjectType, objectType).eq(AuthorizeEntity::getItemId, itemId);
        return this.list(queryWrapper);
    }

    @Override
    public List<AuthorizeEntity> getListByItemId(String itemId) {
        QueryWrapper<AuthorizeEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(AuthorizeEntity::getItemId, itemId);
        return this.list(queryWrapper);
    }

    @Override
    public Map<String, List<Map<String, Object>>> getDataAuthorizeForm() {
        return conditionStrategyFactory.getDataAuthorizeForm();
    }

    @Override
    public String getDescByCondition(String condition) {
        String desc = null;
        Map<String, List<Map<String, Object>>> dataAuthorizeForm = conditionStrategyFactory.getDataAuthorizeForm();
        if (CollectionUtil.isNotEmpty(dataAuthorizeForm)) {
            List<Map<String, Object>> textConditionOptions = dataAuthorizeForm.get("textOptions");
            if (CollectionUtil.isNotEmpty(textConditionOptions)) {
                for (Map<String, Object> textCondition: textConditionOptions) {
                    if (textCondition.get("value").equals(condition)) {
                        desc = String.valueOf(textCondition.get("label"));
                        break;
                    }
                }
            }
            if (StringUtils.isEmpty(desc)) {
                List<Map<String, Object>> stringConditionOptions = dataAuthorizeForm.get("organizeOptions");
                if (CollectionUtil.isNotEmpty(stringConditionOptions)) {
                    for (Map<String, Object> stringCondition: stringConditionOptions) {
                        if (stringCondition.get("value").equals(condition)) {
                            desc = String.valueOf(stringCondition.get("label"));
                            break;
                        }
                    }
                }
            }
            if (StringUtils.isEmpty(desc)) {
                List<Map<String, Object>> numberConditionOptions = dataAuthorizeForm.get("numberOptions");
                if (CollectionUtil.isNotEmpty(numberConditionOptions)) {
                    for (Map<String, Object> numberCondition: numberConditionOptions) {
                        if (numberCondition.get("value").equals(condition)) {
                            desc = String.valueOf(numberCondition.get("label"));
                            break;
                        }
                    }
                }
            }
            if (StringUtils.isEmpty(desc)) {
                List<Map<String, Object>> dateConditionOptions = dataAuthorizeForm.get("dataTimeOptions");
                if (CollectionUtil.isNotEmpty(dateConditionOptions)) {
                    for (Map<String, Object> dateCondition: dateConditionOptions) {
                        if (dateCondition.get("value").equals(condition)) {
                            desc = String.valueOf(dateCondition.get("label"));
                            break;
                        }
                    }
                }
            }
        }
        return desc;
    }

    @Override
    public void saveAuth(String itemId, SaveAuthForm saveAuthForm) {
        String creatorUserId = userProvider.get().getUserId();
        List<AuthorizeEntity> list = new ArrayList<>();
        for (int i = 0; i < saveAuthForm.getObjectId().length; i++) {
            AuthorizeEntity authorizeEntity = new AuthorizeEntity();
            authorizeEntity.setId(RandomUtil.uuId());
            authorizeEntity.setItemType("portal");
            authorizeEntity.setItemId(itemId);
            authorizeEntity.setObjectType(saveAuthForm.getObjectType());
            authorizeEntity.setObjectId(saveAuthForm.getObjectId()[i]);
            authorizeEntity.setSortCode(Long.valueOf(i));
            authorizeEntity.setCreatorTime(new Date());
            authorizeEntity.setCreatorUserId(creatorUserId);
            list.add(authorizeEntity);
        }
        QueryWrapper<AuthorizeEntity> queryWrapper = new QueryWrapper<>();
        queryWrapper.lambda().eq(AuthorizeEntity::getItemId, itemId);
        this.remove(queryWrapper);
        if (list.size() > 0) {
            this.saveBatch(list);

            // 删除角色相关信息
            List<String> objectIdAll = list.stream().map(AuthorizeEntity::getObjectId).collect(Collectors.toList());
            userService.delCurRoleUser(objectIdAll);
        }
    }

    /**
     * 获取条件过滤
     *
     * @param conditionModel 权限信息
     * @return
     */
    @Override
    public QueryWrapper<T> getCondition(AuthorizeConditionModel conditionModel) {
        QueryWrapper<T> queryWhere = conditionModel.getObj();
        String moduleId = conditionModel.getModuleId(); // 菜单ID
        String tableName = conditionModel.getTableName(); // 字段标识（表名）
        /**
         * 根据moduleId,获取当前登录用户拥有的数据权限配置方案
         */
        AuthorizeVO model = this.getAuthorize(true);
        List<ResourceModel> resourceList =
                model.getResourceList().stream().filter(m -> m.getModuleId().equals(moduleId)).collect(Collectors.toList());
        if (resourceList.size() == 0) {
            return null;
        }
        // 声明变量：解析queryWhere的数据权限方案列表。
        List<ResourceModel> resourceList1 = new ArrayList<>();
        // 标记当前用户是否拥有当前菜单的“全部数据”权限，默认1-没有，0-有。
        int t = 1;
        // 遍历数据权限方案，解析条件规则
        for (ResourceModel item : resourceList) {
            // 判断数据权限方案是否“全部数据”权限，拥有则跳出循环，不再继续。
            if ("alldata".equals(item.getEnCode())) {
                t = 0; // 拥有“全部数据”权限
                break;
            }

            /**
             * 没有“全部数据”权限方案时，解析权限方案。
             * 根据tableName，过滤得到符合当前字段标识（表名）的条件组。
             * 最终对符合moduleId与tableName的数据权限方案和条件组进行解析，拼接成查询条件。
             */
            String conditionJson = item.getConditionJson(); // 条件规则Json
            List<ConditionModel> conditionModelList = JsonUtil.getJsonToList(conditionJson, ConditionModel.class);
            // 遍历条件组列表，过滤得到符合当前字段标识（表名）的条件组
            List<ConditionModel> newConditionModelList = conditionModelList.stream().filter(cm -> {
                if (ObjectUtil.isNotEmpty(cm) && CollectionUtil.isNotEmpty(cm.getGroups())) {
                    // 遍历条件列表，过滤得到符合当前字段标识（表名）的条件
                    List<ConditionModel.ConditionItemModel> groups = cm.getGroups();
                    List<ConditionModel.ConditionItemModel> newGroups = groups.stream().filter(group -> {
                        if (StringUtils.isEmpty(group.getBindTable())) {
                            // 判断条件绑定的表名 为空，则认为是公共性的字段，直接返回true
                            return true;
                        } else if (group.getBindTable().equalsIgnoreCase(tableName)) {
                            // 判断条件绑定的表名 与 当前表名是否一致，一致则返回true 选用为条件
                            return true;
                        }
                        return false;
                    }).collect(Collectors.toList());
                    if (CollectionUtil.isNotEmpty(newGroups)) {
                        cm.setGroups(newGroups);
                        return true; // 经过过滤得到符合当前字段标识（表名）的条件集合不为空，则返回true 条件组有效
                    }
                }
                return false; // 经过过滤得到符合当前字段标识（表名）的条件集合为空，则返回false 条件组无效
            }).collect(Collectors.toList());

            // 数据权限方案，存在有效条件组，则将有效条件组添加到解析数据权限方案列表
            if (CollectionUtil.isEmpty(newConditionModelList)) {
                continue;
            }
            item.setConditionJson(JsonUtil.getObjectToString(newConditionModelList));
            resourceList1.add(item);
        }

        if (t == 1 && resourceList1.size() > 0) {
            /**
             * 对符合moduleId与tableName的数据权限方案和条件组进行解析，拼接成查询条件。
             * 多个方案时 方案与方案之间默认OR关系
             * (                    方案                      OR                      方案                      )
             * ((      分组        matchLogic       分组     ) OR (      分组        matchLogic       分组      ))
             * (((条件 logic 条件) matchLogic (条件 logic 条件) OR ((条件 logic 条件) matchLogic (条件 logic 条件)))
             */
            queryWhere.and(tw -> {
                // 遍历数据权限方案，解析条件规则
                for (ResourceModel item : resourceList1) {
                    String matchLogic = StringUtils.isEmpty(item.getMatchLogic()) ? And.getSymbol() :
                            item.getMatchLogic();
                    List<ConditionModel> conditionModelList = JsonUtil.getJsonToList(item.getConditionJson(),
                            ConditionModel.class);
                    Consumer<QueryWrapper<T>> conditionModelConsumer = cmqw -> {
                        // 遍历条件组列表
                        for (int i = 0; i < conditionModelList.size(); i++) {
                            ConditionModel conditionItem = conditionModelList.get(i);
                            // 遍历条件列表
                            Consumer<QueryWrapper<T>> conditionItemConsumer = this.getConditionItem(conditionItem);
                            if (ObjectUtil.isNotEmpty(conditionItemConsumer)) {
                                if (And.getSymbol().equalsIgnoreCase(matchLogic)) {
                                    cmqw.and(conditionItemConsumer);
                                } else {
                                    // or 关系
                                    cmqw.or(conditionItemConsumer);
                                }
                            }
                        }
                    };
                    tw.or(conditionModelConsumer);
                }
            });
        }
        return queryWhere;
    }

    /**
     * 获取分组条件
     *
     * @param conditionItem
     * @return
     */
    private Consumer<QueryWrapper<T>> getConditionItem(ConditionModel conditionItem) {
        if (Objects.isNull(conditionItem) || CollectionUtil.isEmpty(conditionItem.getGroups())) {
            return null;
        }
        Consumer<QueryWrapper<T>> queryWrapperConsumer = tw -> {
            for (int k = 0; k < conditionItem.getGroups().size(); k++) {
                ConditionModel.ConditionItemModel fieldItem = conditionItem.getGroups().get(k);
                ConditionStrategy strategy;
                if ("datetime".equals(fieldItem.getType())) {
                    strategy = conditionStrategyFactory.getConditionStrategy(1);
                } else if ("text".equals(fieldItem.getType())) {
                    strategy = conditionStrategyFactory.getConditionStrategy(5);
                } else if ("int、bigint、double".contains(fieldItem.getType())) {
                    strategy = conditionStrategyFactory.getConditionStrategy(3);
                } else if ("string、varchar".contains(fieldItem.getType())) {
                    strategy = conditionStrategyFactory.getConditionStrategy(4);
                } else {
                    // 默认策略
                    strategy = conditionStrategyFactory.getConditionStrategy(2);
                }
                strategy.generateCondition(tw, conditionItem.getLogic(), fieldItem);
            }
        };
        return queryWrapperConsumer;
    }

    /**
     * 获取条件过滤
     *
     * @param userInfo  用户信息
     * @param moduleId  功能模块Id
     * @param tableName 主表名
     * @return
     */
    @Override
    public String getConditionSql(UserInfo userInfo, String moduleId, String tableName) {
        return StringUtils.replaceIgnoreCase(this.getConditionSql(moduleId, tableName), "WHERE", " AND") + " ";
    }

    public String getConditionSql(String moduleId, String tableName) {
        AuthorizeConditionModel conditionModel = new AuthorizeConditionModel(new QueryWrapper<>(), moduleId, tableName);
        QueryWrapper<T> queryWhere = this.getCondition(conditionModel);
        String sqlSegment = this.getSql(queryWhere);
        return sqlSegment;
    }

    public <T> String getSql(QueryWrapper<T> queryWrapper) {
        String customSqlSegment = queryWrapper.getCustomSqlSegment();
        for (Map.Entry<String, Object> entry : queryWrapper.getParamNameValuePairs().entrySet()) {
            if (!ObjectUtil.isEmpty(entry) && !ObjectUtil.isEmpty(entry.getKey()) && !ObjectUtil.isEmpty(entry.getValue()))
                customSqlSegment = customSqlSegment.replace("#{ew.paramNameValuePairs." + entry.getKey() + "}",
                        formatParamValue(entry.getValue()));
        }
        return customSqlSegment;
    }

    public String formatParamValue(Object arg) {
        String paramValue;
        if (arg instanceof String) {
            paramValue = "'" + arg.toString().replace("'", "\\'") + "'";
        } else if (arg instanceof Date) {
            paramValue = "'" + DateUtil.format((Date) arg, "yyyy-MM-dd HH:mm:ss") + "'";
        } else if (arg != null) {
            paramValue = arg.toString();
        } else {
            paramValue = null;
        }
        return paramValue;
    }

    @Override
    public AuthorizeDataReturnVO getAuthorizeDataReturnVO(AuthorizeVO authorizeModel, List<AuthorizeEntity> list,
                                                          DataValuesQuery dataValuesQuery, String objectId) {
        AuthorizeDataReturnVO dataReturnVO = new AuthorizeDataReturnVO();
        if (!StringUtils.isEmpty(dataValuesQuery.getType())) {
            switch (dataValuesQuery.getType()) {
                case "module":
                    List<ModuleEntity> moduleList = moduleService.getList().stream().filter(
                            m -> "1".equals(String.valueOf(m.getEnabledMark()))
                    ).collect(Collectors.toList());
                    dataReturnVO = this.module(moduleList, list, authorizeModel);
                    return dataReturnVO;
                case "button":
                    List<ModuleEntity> moduleList1 = moduleService.getList().stream().filter(
                            m -> "1".equals(String.valueOf(m.getEnabledMark()))
                    ).collect(Collectors.toList());
                    //挑选出的list
                    List<ModuleEntity> selectList1 = new ArrayList<>();
                    if (!StringUtils.isEmpty(dataValuesQuery.getModuleIds())) {
                        List<String> moduleId1 = Arrays.asList(dataValuesQuery.getModuleIds().split(","));
                        selectList1 =
                                moduleList1.stream().filter(t -> moduleId1.contains(t.getId())).collect(Collectors.toList());
                    }
                    List<ModuleButtonEntity> moduleButtonList = buttonService.getList().stream().filter(
                            m -> "1".equals(String.valueOf(m.getEnabledMark()))
                    ).collect(Collectors.toList());
                    dataReturnVO = this.moduleButton(selectList1, moduleButtonList, list, authorizeModel);
                    return dataReturnVO;
                case "column":
                    List<ModuleEntity> moduleList2 = moduleService.getList().stream().filter(
                            m -> "1".equals(String.valueOf(m.getEnabledMark()))
                    ).collect(Collectors.toList());
                    //挑选出的list
                    List<ModuleEntity> selectList2 = new ArrayList<>();
                    if (!StringUtils.isEmpty(dataValuesQuery.getModuleIds())) {
                        List<String> moduleId2 = Arrays.asList(dataValuesQuery.getModuleIds().split(","));
                        selectList2 =
                                moduleList2.stream().filter(t -> moduleId2.contains(t.getId())).collect(Collectors.toList());
                    }
                    List<ModuleColumnEntity> moduleColumnList = columnService.getList().stream().filter(
                            m -> "1".equals(String.valueOf(m.getEnabledMark()))
                    ).collect(Collectors.toList());
                    dataReturnVO = this.moduleColumn(selectList2, moduleColumnList, list, authorizeModel);
                    return dataReturnVO;
                case "resource":
                    List<ModuleEntity> moduleList3 = moduleService.getList().stream().filter(
                            m -> "1".equals(String.valueOf(m.getEnabledMark()))
                    ).collect(Collectors.toList());
                    //挑选出的list
                    List<ModuleEntity> selectList3 = new ArrayList<>();
                    if (!StringUtils.isEmpty(dataValuesQuery.getModuleIds())) {
                        List<String> moduleId3 = Arrays.asList(dataValuesQuery.getModuleIds().split(","));
                        selectList3 =
                                moduleList3.stream().filter(t -> moduleId3.contains(t.getId())).collect(Collectors.toList());
                    }
                    List<ModuleDataAuthorizeSchemeEntity> moduleDataSchemeList =
                            schemeService.getList().stream().filter(
                                    m -> "1".equals(String.valueOf(m.getEnabledMark()))
                            ).collect(Collectors.toList());
                    dataReturnVO = this.resourceData(selectList3, moduleDataSchemeList, list, authorizeModel);
                    return dataReturnVO;
                case "form":
                    List<ModuleEntity> moduleList4 = moduleService.getList().stream().filter(
                            m -> "1".equals(String.valueOf(m.getEnabledMark()))
                    ).collect(Collectors.toList());
                    //挑选出的list
                    List<ModuleEntity> selectList4 = new ArrayList<>();
                    if (!StringUtils.isEmpty(dataValuesQuery.getModuleIds())) {
                        List<String> moduleId4 = Arrays.asList(dataValuesQuery.getModuleIds().split(","));
                        selectList4 =
                                moduleList4.stream().filter(t -> moduleId4.contains(t.getId())).collect(Collectors.toList());
                    }
                    List<ModuleFormEntity> moduleFormList = formService.getList().stream().filter(
                            m -> "1".equals(String.valueOf(m.getEnabledMark()))
                    ).collect(Collectors.toList());
                    dataReturnVO = this.moduleForm(selectList4, moduleFormList, list, authorizeModel);
                    return dataReturnVO;
                case "userList":
                    List<UserEntity> userEntityList = userService.getUserListByRoleId(objectId);
                    dataReturnVO = this.roleUserList(userEntityList);
                    return dataReturnVO;
                default:
            }
        }
        return dataReturnVO;
    }

    /**
     * 功能权限
     *
     * @param moduleListAll  所有功能
     * @param authorizeList  已有权限
     * @param authorizeModel 权限集合
     * @return
     */
    @Override
    public AuthorizeDataReturnVO module(List<ModuleEntity> moduleListAll, List<AuthorizeEntity> authorizeList,
                                        AuthorizeVO authorizeModel) {
        AuthorizeDataReturnVO vo = new AuthorizeDataReturnVO();
        List<ModuleModel> moduleList = authorizeModel.getModuleList();
        List<String> moduleModeId = moduleListAll.stream().map(t -> t.getId()).collect(Collectors.toList());
        List<String> ids =
                authorizeList.stream().filter(t -> "module".equals(t.getItemType())).map(t -> t.getItemId()).collect(Collectors.toList());
        moduleModeId.addAll(ids);
        List<AuthorizeDataModel> treeList = JsonUtil.getJsonToList(moduleList, AuthorizeDataModel.class);
        List<SumTree<AuthorizeDataModel>> trees = TreeDotUtils.convertListToTreeDot(treeList, "-1");
        List<AuthorizeDataReturnModel> data = JsonUtil.getJsonToList(trees, AuthorizeDataReturnModel.class);
        vo.setAll(moduleModeId);
        vo.setIds(ids);
        vo.setList(data);
        return vo;
    }

    /**
     * 按钮权限
     *
     * @param moduleList     功能
     * @param moduleButton   按钮
     * @param authorizeList  已有权限
     * @param authorizeModel 权限集合
     * @return
     */
    @Override
    public AuthorizeDataReturnVO moduleButton(List<ModuleEntity> moduleList, List<ModuleButtonEntity> moduleButton,
                                              List<AuthorizeEntity> authorizeList, AuthorizeVO authorizeModel) {
        List<AuthorizeModel> treeList = new ArrayList<>();
        //appid
        List<String> appId =
                moduleList.stream().filter(t -> "App".equals(t.getCategory())|| t.getType()==0).map(t -> t.getId()).collect(Collectors.toList());
        List<ButtonModel> buttonList = authorizeModel.getButtonList();
        List<String> moduleModeId = moduleList.stream().map(t -> t.getId()).collect(Collectors.toList());
        List<String> buttonModeId = buttonList.stream().map(t -> t.getModuleId()).collect(Collectors.toList());
        moduleModeId.addAll(buttonModeId);
        List<String> ids =
                authorizeList.stream().filter(t -> "button".equals(t.getItemType())).map(t -> t.getItemId()).collect(Collectors.toList());
        //获取按钮的菜单id
        for (ButtonModel buttonModel : buttonList) {
            AuthorizeModel treeModel = new AuthorizeModel();
            treeModel.setId(buttonModel.getId());
            treeModel.setFullName(buttonModel.getFullName());
            treeModel.setParentId(buttonModel.getModuleId());
            treeModel.setIcon(buttonModel.getIcon());
            ModuleEntity info = moduleService.getInfo(buttonModel.getModuleId());
            treeModel.setCategory(ObjectUtil.isNotNull(info)?info.getCategory():"");

            treeList.add(treeModel);
        }
        List<ModuleEntity> list =
                moduleList.stream().filter(t -> buttonModeId.contains(t.getId())).collect(Collectors.toList());
        List<AuthorizeModel> moduleListAll = JsonUtil.getJsonToList(ListToTreeUtil.treeWhere(list, moduleList),
                AuthorizeModel.class);
        treeList.addAll(moduleListAll);
        treeList =
                treeList.stream().sorted(Comparator.comparing(AuthorizeModel::getSortCode)).collect(Collectors.toList());
        List<SumTree<AuthorizeModel>> trees = TreeDotUtils.convertListToTreeDot(treeList, "-1");
        List<AuthorizeDataReturnModel> data = JsonUtil.getJsonToList(trees, AuthorizeDataReturnModel.class);

        AuthorizeDataReturnVO vo = new AuthorizeDataReturnVO();
        vo.setAll(moduleModeId);
        vo.setIds(ids);
        vo.setList(data);
        return vo;
    }

    /**
     * 列表权限
     *
     * @param moduleList       功能
     * @param moduleColumnList 列表
     * @param authorizeList    已有权限
     * @param authorizeModel   权限集合
     * @return
     */
    @Override
    public AuthorizeDataReturnVO moduleColumn(List<ModuleEntity> moduleList,
                                              List<ModuleColumnEntity> moduleColumnList,
                                              List<AuthorizeEntity> authorizeList, AuthorizeVO authorizeModel) {
        List<AuthorizeModel> treeList = new ArrayList<>();
        //appid
        List<String> appId =
                moduleList.stream().filter(t -> "App".equals(t.getCategory())|| t.getType()==0).map(t -> t.getId()).collect(Collectors.toList());
        List<ColumnModel> columnList = authorizeModel.getColumnList();
        List<String> moduleModeId = moduleList.stream().map(t -> t.getId()).collect(Collectors.toList());
        List<String> columnModeId = columnList.stream().map(t -> t.getModuleId()).collect(Collectors.toList());
        moduleModeId.addAll(columnModeId);
        List<String> ids =
                authorizeList.stream().filter(t -> "column".equals(t.getItemType())).map(t -> t.getItemId()).collect(Collectors.toList());
        //获取按钮的菜单id
        for (ColumnModel columnModel : columnList) {
            AuthorizeModel treeModel = new AuthorizeModel();
            treeModel.setId(columnModel.getId());
            treeModel.setFullName(columnModel.getFullName());
            treeModel.setParentId(columnModel.getModuleId());
            ModuleEntity info = moduleService.getInfo(columnModel.getModuleId());
            treeModel.setCategory(ObjectUtil.isNotNull(info)?info.getCategory():"");
            treeModel.setIcon("fa fa-tags column");
            treeList.add(treeModel);
        }
        List<ModuleEntity> list =
                moduleList.stream().filter(t -> columnModeId.contains(t.getId())).collect(Collectors.toList());
        List<AuthorizeModel> moduleListAll = JsonUtil.getJsonToList(ListToTreeUtil.treeWhere(list, moduleList),
                AuthorizeModel.class);
        treeList.addAll(moduleListAll);
        treeList =
                treeList.stream().sorted(Comparator.comparing(AuthorizeModel::getSortCode)).collect(Collectors.toList());
        List<SumTree<AuthorizeModel>> trees = TreeDotUtils.convertListToTreeDot(treeList, "-1");
        List<AuthorizeDataReturnModel> data = JsonUtil.getJsonToList(trees, AuthorizeDataReturnModel.class);
        AuthorizeDataReturnVO vo = new AuthorizeDataReturnVO();
        vo.setAll(moduleModeId);
        vo.setIds(ids);
        vo.setList(data);
        return vo;
    }

    /**
     * 表单权限
     *
     * @param moduleList     功能
     * @param moduleFormList 表单列表
     * @param authorizeList  已有权限
     * @param authorizeModel 权限集合
     * @return
     */
    @Override
    public AuthorizeDataReturnVO moduleForm(List<ModuleEntity> moduleList, List<ModuleFormEntity> moduleFormList,
                                            List<AuthorizeEntity> authorizeList, AuthorizeVO authorizeModel) {
        List<AuthorizeModel> treeList = new ArrayList<>();
        //appid
        List<String> appId =
                moduleList.stream().filter(t -> "App".equals(t.getCategory())|| t.getType()==0).map(t -> t.getId()).collect(Collectors.toList());
        List<ModuleFormModel> formList = authorizeModel.getFormsList();
        List<String> moduleModeId = moduleList.stream().map(t -> t.getId()).collect(Collectors.toList());
        List<String> formModeId = formList.stream().map(t -> t.getModuleId()).collect(Collectors.toList());
        moduleModeId.addAll(formModeId);
        List<String> ids =
                authorizeList.stream().filter(t -> "form".equals(t.getItemType())).map(t -> t.getItemId()).collect(Collectors.toList());
        //获取按钮的菜单id
        for (ModuleFormModel columnModel : formList) {
            AuthorizeModel treeModel = new AuthorizeModel();
            treeModel.setId(columnModel.getId());
            treeModel.setFullName(columnModel.getFullName());
            treeModel.setParentId(columnModel.getModuleId());
            ModuleEntity info = moduleService.getInfo(columnModel.getModuleId());
            treeModel.setCategory(ObjectUtil.isNotNull(info)?info.getCategory():"");

            treeModel.setIcon("fa fa-binoculars resource");
            treeList.add(treeModel);
        }
        List<ModuleEntity> list =
                moduleList.stream().filter(t -> formModeId.contains(t.getId())).collect(Collectors.toList());
        List<AuthorizeModel> moduleListAll = JsonUtil.getJsonToList(ListToTreeUtil.treeWhere(list, moduleList),
                AuthorizeModel.class);
        treeList.addAll(moduleListAll);
        treeList =
                treeList.stream().sorted(Comparator.comparing(AuthorizeModel::getSortCode)).collect(Collectors.toList());
        List<SumTree<AuthorizeModel>> trees = TreeDotUtils.convertListToTreeDot(treeList, "-1");
        List<AuthorizeDataReturnModel> data = JsonUtil.getJsonToList(trees, AuthorizeDataReturnModel.class);
        AuthorizeDataReturnVO vo = new AuthorizeDataReturnVO();
        vo.setAll(moduleModeId);
        vo.setIds(ids);
        vo.setList(data);
        return vo;
    }

    /**
     * 数据权限
     *
     * @param moduleList           功能
     * @param moduleDataSchemeList 数据方案
     * @param authorizeList        已有权限
     * @param authorizeModel       权限集合
     * @return
     */
    @Override
    public AuthorizeDataReturnVO resourceData(List<ModuleEntity> moduleList,
                                              List<ModuleDataAuthorizeSchemeEntity> moduleDataSchemeList,
                                              List<AuthorizeEntity> authorizeList, AuthorizeVO authorizeModel) {
        List<AuthorizeModel> treeList = new ArrayList<>();
        //appid
        List<String> appId =
                moduleList.stream().filter(t -> "App".equals(t.getCategory())|| t.getType()==0).map(t -> t.getId()).collect(Collectors.toList());
        List<ResourceModel> resourceList = authorizeModel.getResourceList();
        List<String> moduleModeId = moduleList.stream().map(t -> t.getId()).collect(Collectors.toList());
        List<String> resourceModeId = resourceList.stream().map(t -> t.getModuleId()).collect(Collectors.toList());
        moduleModeId.addAll(resourceModeId);
        List<String> ids =
                authorizeList.stream().filter(t -> "resource".equals(t.getItemType())).map(t -> t.getItemId()).collect(Collectors.toList());
        //获取按钮的菜单id
        for (ResourceModel resourceModel : resourceList) {
            AuthorizeModel treeModel = new AuthorizeModel();
            treeModel.setId(resourceModel.getId());
            treeModel.setFullName(resourceModel.getFullName());
            treeModel.setParentId(resourceModel.getModuleId());
            ModuleEntity info = moduleService.getInfo(resourceModel.getModuleId());
            treeModel.setCategory(ObjectUtil.isNotNull(info)?info.getCategory():"");

            treeModel.setIcon("fa fa-binoculars resource");
            treeList.add(treeModel);
        }
        List<ModuleEntity> list =
                moduleList.stream().filter(t -> resourceModeId.contains(t.getId())).collect(Collectors.toList());
        List<AuthorizeModel> moduleListAll = JsonUtil.getJsonToList(ListToTreeUtil.treeWhere(list, moduleList),
                AuthorizeModel.class);
        treeList.addAll(moduleListAll);
        treeList =
                treeList.stream().sorted(Comparator.comparing(AuthorizeModel::getSortCode)).collect(Collectors.toList());
        List<SumTree<AuthorizeModel>> trees = TreeDotUtils.convertListToTreeDot(treeList, "-1");
        List<AuthorizeDataReturnModel> data = JsonUtil.getJsonToList(trees, AuthorizeDataReturnModel.class);
        AuthorizeDataReturnVO vo = new AuthorizeDataReturnVO();
        vo.setAll(moduleModeId);
        vo.setIds(ids);
        vo.setList(data);
        return vo;
    }

    /**
     * 用户列表数据转换
     *
     * @param userEntityList 用户列表
     * @return
     */
    private AuthorizeDataReturnVO roleUserList(List<UserEntity> userEntityList) {
        AuthorizeDataReturnVO dataReturnVO = new AuthorizeDataReturnVO();
        List<AuthorizeDataReturnModel> userList = new ArrayList<>();
        for (UserEntity userEntity : userEntityList) {
            AuthorizeDataReturnModel authorizeDataReturnModel = new AuthorizeDataReturnModel();
            authorizeDataReturnModel.setId(userEntity.getId());
            authorizeDataReturnModel.setFullName(userEntity.getRealName());
            userList.add(authorizeDataReturnModel);
        }
        dataReturnVO.setList(userList);
        return dataReturnVO;
    }
}
