package com.bringspring.system.print.controller;

import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.bringspring.common.base.ActionResult;
import com.bringspring.common.base.vo.DownloadVO;
import com.bringspring.common.base.vo.ListVO;
import com.bringspring.common.base.vo.PageListVO;
import com.bringspring.common.base.vo.PaginationVO;
import com.bringspring.common.config.ConfigValueUtil;
import com.bringspring.common.constant.MsgCode;
import com.bringspring.common.exception.DataException;
import com.bringspring.common.util.FileUtil;
import com.bringspring.common.util.JsonUtil;
import com.bringspring.common.util.RandomUtil;
import com.bringspring.common.util.XSSEscape;
import com.bringspring.common.util.enums.ModuleTypeEnum;
import com.bringspring.common.util.file.FileExport;
import com.bringspring.system.base.entity.DictionaryDataEntity;
import com.bringspring.system.base.entity.DictionaryTypeEntity;
import com.bringspring.system.base.service.DictionaryDataService;
import com.bringspring.system.base.service.DictionaryTypeService;
import com.bringspring.system.permission.entity.UserEntity;
import com.bringspring.system.permission.service.UserService;
import com.bringspring.system.print.entity.OperatorRecordEntity;
import com.bringspring.system.print.entity.PrintDevEntity;
import com.bringspring.system.print.model.PaginationPrint;
import com.bringspring.system.print.model.dto.PrintDevFormDTO;
import com.bringspring.system.print.model.query.PrintDevDataQuery;
import com.bringspring.system.print.model.query.PrintDevFieldsQuery;
import com.bringspring.system.print.model.vo.PrintDevListVO;
import com.bringspring.system.print.model.vo.PrintDevVO;
import com.bringspring.system.print.service.IPrintDevService;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.MediaType;
import org.springframework.web.bind.annotation.*;
import org.springframework.web.multipart.MultipartFile;

import javax.validation.Valid;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 打印模板 -控制器
 *
 * @author RKKJ开发平台组 YY
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2017年9月30日
 */
@Api(tags = "打印模板", value = "print")
@RestController
@RequestMapping("/api/system/printDev")
public class PrintDevController {

    @Autowired
    private IPrintDevService iPrintDevService;
    @Autowired
    private FileExport fileExport;
    @Autowired
    private ConfigValueUtil configValueUtil;
    @Autowired
    private DictionaryDataService dictionaryDataService;
    @Autowired
    private DictionaryTypeService dictionaryTypeService;
    @Autowired
    private UserService userService;

    /**============增删改==============**/

    /**
     * 新增打印模板对象
     *
     * @param printDevForm 打印模板数据传输对象
     * @return 执行结果标识
     */
    @ApiOperation("新增")
    @PostMapping
    public ActionResult<PrintDevFormDTO> create(@RequestBody @Valid PrintDevFormDTO printDevForm) {
        //重名验证
        QueryWrapper<PrintDevEntity> query0 = new QueryWrapper<>();
        query0.lambda().eq(PrintDevEntity::getFullName, printDevForm.getFullName());
        if (iPrintDevService.getOne(query0) != null) {
            return ActionResult.fail(MsgCode.EXIST003.get());
        }
        QueryWrapper<PrintDevEntity> query1 = new QueryWrapper<>();
        //编码验证
        query1.lambda().eq(PrintDevEntity::getEnCode, printDevForm.getEnCode());
        if (iPrintDevService.getOne(query1) != null) {
            return ActionResult.fail(MsgCode.EXIST002.get());
        }
        PrintDevEntity printDevEntity = JsonUtil.getJsonToBean(printDevForm, PrintDevEntity.class);
        printDevEntity.setId(RandomUtil.uuId());
        iPrintDevService.save(printDevEntity);
        return ActionResult.success(MsgCode.SU001.get());
    }

    /**
     * 复制打印模板
     *
     * @param id 打印模板id
     * @return 执行结果标识
     */
    @ApiOperation("复制")
    @PostMapping("/{id}/Actions/Copy")
    public ActionResult<PageListVO<PrintDevEntity>> copy(@PathVariable String id) {
        int randomNum = (int)(Math.random() * 100000);
        PrintDevEntity entity = iPrintDevService.getById(id);
        entity.setEnCode(entity.getEnCode() + "." + randomNum);
        entity.setId(RandomUtil.uuId());
        entity.setEnabledMark(0);
        entity.setFullName(entity.getFullName() + ".副本" + randomNum);
        entity.setLastModifyTime(null);
        entity.setLastModifyUserId(null);
        entity.setId(RandomUtil.uuId());
        PrintDevEntity entityBean = JsonUtil.getJsonToBean(entity, PrintDevEntity.class);
        iPrintDevService.save(entityBean);
        return ActionResult.success(MsgCode.SU007.get());
    }

    /**
     * 删除打印模板
     *
     * @param id 打印模板id
     * @return ignore
     */
    @ApiOperation("删除")
    @DeleteMapping("/{id}")
    public ActionResult<PrintDevFormDTO> delete(@PathVariable String id) {
        //对象存在判断
        if (iPrintDevService.getById(id) != null) {
            iPrintDevService.removeById(id);
            return ActionResult.success(MsgCode.SU003.get());
        } else {
            return ActionResult.fail(MsgCode.FA003.get());
        }
    }

    /**
     * 删除打印模板
     *
     * @param id           打印模板id
     * @param printDevForm 打印模板数据传输对象
     * @return 执行结果标识
     */
    @ApiOperation("编辑")
    @PutMapping("/{id}")
    public ActionResult<PrintDevFormDTO> update(@PathVariable String id, @RequestBody @Valid PrintDevFormDTO printDevForm) {
        //重名验证
        //编码验证
        //对象存在判断
        PrintDevEntity printDevEntity = JsonUtil.getJsonToBean(printDevForm, PrintDevEntity.class);
        printDevEntity.setId(id);
        iPrintDevService.updateById(printDevEntity);
        return ActionResult.success(MsgCode.SU004.get());
    }

    /**
     * 修改打印模板可用状态
     *
     * @param id 打印模板id
     * @return 执行结果标识
     */
    @ApiOperation("修改状态")
    @PutMapping("/{id}/Actions/State")
    public ActionResult<PageListVO<PrintDevEntity>> state(@PathVariable String id) {
        PrintDevEntity entity = iPrintDevService.getById(id);
        if (entity != null) {
            if ("1".equals(entity.getEnabledMark().toString())) {
                entity.setEnabledMark(0);
            } else {
                entity.setEnabledMark(1);
            }
            iPrintDevService.updateById(entity);
            return ActionResult.success(MsgCode.SU004.get());
        }
        return ActionResult.fail(MsgCode.FA002.get());
    }

    /*============查询==============*/

    /**
     * 查看单个模板详情
     *
     * @param id 打印模板id
     * @return 单个模板对象
     */
    @ApiOperation("预览")
    @GetMapping("/{id}")
    public ActionResult<PrintDevEntity> info(@PathVariable String id) {
        return ActionResult.success(iPrintDevService.getById(id));
    }

    /**
     * 查看所有模板
     *
     * @return 所有模板集合
     */
    @ApiOperation("列表")
    @GetMapping
    public ActionResult list(PaginationPrint paginationPrint) {
        List<PrintDevEntity> list = iPrintDevService.getList(paginationPrint);
        List<String> userId = list.stream().map(t -> t.getCreatorUserId()).collect(Collectors.toList());
        List<String> lastUserId = list.stream().map(t -> t.getLastModifyUserId()).collect(Collectors.toList());
        lastUserId.removeAll(Collections.singleton(null));
        List<UserEntity> userEntities = userService.getUserName(userId);
        List<UserEntity> lastUserIdEntities = userService.getUserName(lastUserId);
        DictionaryTypeEntity typeEntity = dictionaryTypeService.getInfoByEnCode("printDev");
        List<DictionaryDataEntity> typeList = dictionaryDataService.getList(typeEntity.getId());
        List<PrintDevListVO> listVOS = new ArrayList<>();
        for (PrintDevEntity entity : list) {
            PrintDevListVO vo = JsonUtil.getJsonToBean(entity, PrintDevListVO.class);
            DictionaryDataEntity dataEntity = typeList.stream().filter(t -> t.getEnCode().equals(entity.getCategory())).findFirst().orElse(null);
            if (dataEntity != null) {
                vo.setCategory(dataEntity.getFullName());
            } else {
                vo.setCategory("");
            }
            //创建者
            UserEntity creatorUser = userEntities.stream().filter(t -> t.getId().equals(entity.getCreatorUserId())).findFirst().orElse(null);
            vo.setCreatorUserId(creatorUser != null ? creatorUser.getRealName() : entity.getCreatorUserId());
            //修改人
            UserEntity lastModifyUser = lastUserIdEntities.stream().filter(t -> t.getId().equals(entity.getLastModifyUserId())).findFirst().orElse(null);
            vo.setLastModifyUserId(lastModifyUser != null ? lastModifyUser.getRealName(): entity.getLastModifyUserId());
            listVOS.add(vo);
        }
        PaginationVO paginationVO = JsonUtil.getJsonToBean(paginationPrint, PaginationVO.class);
        return ActionResult.page(listVOS , paginationVO);
    }

    /**
     * 下拉列表
     *
     * @return 返回列表数据
     */
    @ApiOperation("下拉列表")
    @GetMapping("/Selector")
    public ActionResult<ListVO<PrintDevVO>> selectorList(Integer type){
        ListVO<PrintDevVO> vo = new ListVO<>();
        try{
            vo.setList(iPrintDevService.getTreeModel(type));
            return ActionResult.success(vo);
        }catch (Exception e){
            return ActionResult.fail(e.getMessage());
        }
    }

    /**
     * 根据sql获取内容
     * @param printDevSqlDataQuery 打印模板查询对象
     * @return 打印模板对应内容
     */
    @ApiOperation("Sql数据获取")
    @GetMapping("/Data")
    public ActionResult<Map<String, Object>> getFieldData(PrintDevDataQuery printDevSqlDataQuery) {
        String id = XSSEscape.escape(printDevSqlDataQuery.getId());
        String formId = XSSEscape.escape(printDevSqlDataQuery.getFormId());
        PrintDevEntity entity = iPrintDevService.getById(id);
        if(entity == null){
            return ActionResult.fail(MsgCode.PRI001.get());
        }
        Map<String, Object> printDataMap;
        try {
            printDataMap = iPrintDevService.getDataBySql(
                    entity.getDbLinkId(),
                    entity.getSqlTemplate().replaceAll("@formId", "'" + formId + "'"),
                    2);
        } catch (Exception e) {
            return ActionResult.fail(e.getMessage());
        }
        List<Map<String, Object>> headTableList = (List<Map<String, Object>>) printDataMap.get("headTable");
        printDataMap.remove("headTable");
        for (Map map : headTableList) {
            printDataMap.putAll(map);
        }
        Map<String, Object> dataMap = new HashMap<>(16);
        dataMap.put("printData", printDataMap);
        dataMap.put("printTemplate", entity.getPrintTemplate());
        List<OperatorRecordEntity> operatorRecordList = iPrintDevService.getFlowTaskOperatorRecordList(formId);
        dataMap.put("operatorRecordList", operatorRecordList);
        return ActionResult.success(dataMap);
    }

    /**
     * 获取打印模块字段信息
     *
     * @param printDevFieldsQuery 打印模板查询对象
     * @return 字段信息数据
     */
    @ApiOperation("Sql字段获取")
    @PostMapping("/Fields")
    public ActionResult<Map<String, Object>> getFields(@RequestBody PrintDevFieldsQuery printDevFieldsQuery) {
        String dbLinkId = XSSEscape.escape(printDevFieldsQuery.getDbLinkId());
        try {
            return ActionResult.success(
                    iPrintDevService.getDataBySql(dbLinkId,
                            printDevFieldsQuery.getSqlTemplate().replaceAll("@formId", " null "), 1));
        } catch (Exception e) {
            return ActionResult.fail(e.getMessage());
        }
    }

    /*==========行为============*/

    /**
     * 导出打印模板备份json
     *
     * @param id 打印模板id
     */
    @ApiOperation("导出")
    @GetMapping("/{id}/Actions/Export")
    public ActionResult<DownloadVO> export(@PathVariable String id) {
        PrintDevEntity entity = iPrintDevService.getById(id);
        //导出文件
        DownloadVO downloadVO = fileExport.exportFile(entity, configValueUtil.getTemporaryFilePath(), entity.getFullName(), ModuleTypeEnum.SYSTEM_PRINT.getTableName());
        return ActionResult.success(downloadVO);
    }

    /**
     * 导入打印模板备份json
     *
     * @param multipartFile 备份json文件
     * @return 执行结果标识
     */
    @ApiOperation("导入")
    @PostMapping(value = "/Actions/ImportData", consumes = MediaType.MULTIPART_FORM_DATA_VALUE)
    public ActionResult<PageListVO<PrintDevEntity>> importData(@RequestPart("file") MultipartFile multipartFile) throws DataException {
        //判断是否为.json结尾
        if (FileUtil.existsSuffix(multipartFile, ModuleTypeEnum.SYSTEM_PRINT.getTableName())) {
            return ActionResult.fail(MsgCode.IMP002.get());
        }
        //读取文件内容
        String fileContent = FileUtil.getFileContent(multipartFile, configValueUtil.getTemporaryFilePath());
        try {
            PrintDevEntity entity = JsonUtil.getJsonToBean(fileContent, PrintDevEntity.class);
            //id为空切名称不存在时
            if (iPrintDevService.getById(entity.getId()) == null &&
                    !iPrintDevService.checkNameExist(entity.getFullName(), entity.getId())) {
                iPrintDevService.save(entity);
                return ActionResult.success(MsgCode.IMP001.get());
            }
        } catch (Exception e) {
            throw new DataException(MsgCode.IMP004.get());
        }
        return ActionResult.fail(MsgCode.IMP003.get());
    }

}
