package com.bringspring.system.msgcenter.service.strategy.impl;

import cn.hutool.core.collection.CollectionUtil;
import cn.hutool.core.util.ObjectUtil;
import com.alibaba.fastjson.JSONObject;
import com.bringspring.common.base.UserInfo;
import com.bringspring.common.util.StringUtils;
import com.bringspring.system.msgcenter.entity.McMsgAccountEntity;
import com.bringspring.system.msgcenter.entity.McTaskMsgContentEntity;
import com.bringspring.system.msgcenter.entity.McTaskMsgReceiveEntity;
import com.bringspring.system.msgcenter.enums.ChannelTypeEnum;
import com.bringspring.system.msgcenter.enums.EnabledMarkEnum;
import com.bringspring.system.msgcenter.model.mcmsgsendtemplate.McMsgSendTemplateModel;
import com.bringspring.system.msgcenter.service.McMsgAccountService;
import com.bringspring.system.msgcenter.service.McTaskMsgReceiveService;
import com.bringspring.system.msgcenter.service.context.SendMessageContext;
import com.bringspring.system.msgcenter.service.strategy.AbstractMessageSender;
import com.bringspring.system.msgcenter.util.WebHookUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

/**
 * 消息中心-webhook下发工具类
 */
@Component
@Slf4j
public class MsgToWebhookService extends AbstractMessageSender {

    @Override
    protected int getBatchSize() {
        return 1000;
    }

    @Override
    public ChannelTypeEnum getChannelType() {
        return ChannelTypeEnum.WEBHOOK;
    }


    @Override
    protected void validateParameters(SendMessageContext context) {
        validateTemplate(context);
        validateAccount(context);
    }

    @Override
    protected void doSend(SendMessageContext context) {
        if (CollectionUtil.isNotEmpty(context.getValidList())) {
            List<McTaskMsgReceiveEntity> receiveList = context.getValidList();
            // 预处理：消息内容
            McTaskMsgContentEntity taskMsgContent = context.getTaskMsgContent();
            // 预处理：应用账号配置
            McMsgAccountEntity account = context.getMsgAccount();

            String approveType = String.valueOf(account.getApproveType());
            String webhookType = String.valueOf(account.getWebhookType());
            String webhookAddress = account.getWebhookAddress();
            String content = taskMsgContent.getContent();

            List<McTaskMsgReceiveEntity> validList = new ArrayList<>();
            List<McTaskMsgReceiveEntity> invalidList = new ArrayList<>();
            JSONObject retJson = new JSONObject();
            switch (webhookType) {
                case "1":
                    //钉钉
                    if ("1".equals(approveType)) {
                        retJson = WebHookUtil.sendDDMessage(webhookAddress, content);
                    } else if ("2".equals(approveType)) {
                        retJson = WebHookUtil.sendDingDing(webhookAddress, account.getBearer(), content);
                    }
                    break;
                case "2":
                    if ("1".equals(approveType)) {
                        retJson = WebHookUtil.callWeChatBot(webhookAddress, content);
                    }
                    break;
                default:
                    break;
            }

            /**
             * 处理发送结果
             */
            if ("ok".equals(retJson.getString("errmsg"))) {
                // 成功
                receiveList.get(0).setEnabledMark(EnabledMarkEnum.SUCCESS.getCode());
                receiveList.get(0).setDescription(JSONObject.toJSONString(retJson));
                validList.addAll(receiveList);
            } else {
                // 失败
                receiveList.get(0).setEnabledMark(EnabledMarkEnum.FAIL.getCode());
                receiveList.get(0).setDescription(JSONObject.toJSONString(retJson));
                invalidList.addAll(receiveList);
            }
            context.getInvalidList().addAll(invalidList);
            context.setValidList(validList);
        }
    }


    @Override
    protected void sendBatchNonSystemUser(boolean isAgain, McMsgSendTemplateModel sendTemplate,
                                          McTaskMsgContentEntity taskMsgContent,
                                          List<McTaskMsgReceiveEntity> receiveListS, UserInfo currentUserInfo) {
        if (ObjectUtil.isEmpty(taskMsgContent) || CollectionUtil.isEmpty(receiveListS)) {
            log.error("~·~·~·~调用了 webhook消息【非系统用户】下发，但taskMsgContent为空 或 receiveList为空~·~·~·~");
        }
        sendBatch(isAgain, sendTemplate, taskMsgContent, receiveListS, currentUserInfo);
    }

    /**
     * 发送webhook
     *
     * @param isAgain        是否重发
     * @param taskMsgContent 业务消息内容
     * @param receiveListS   业务消息接收人
     */
    @Override
    protected void sendBatch(boolean isAgain,
                             McMsgSendTemplateModel sendTemplate,
                             McTaskMsgContentEntity taskMsgContent,
                             List<McTaskMsgReceiveEntity> receiveListS,
                             UserInfo currentUserInfo) {
        if (ObjectUtil.isEmpty(taskMsgContent) || CollectionUtil.isEmpty(receiveListS)) {
            log.error("~·~·~·~调用了 webhook消息下发，但taskMsgContent为空 或 receiveList为空~·~·~·~");
        }
        for (McTaskMsgReceiveEntity receive : receiveListS) {
            if (StringUtils.isNotEmpty(receive.getReceiveUserId())) {
                // 应用授权账号信息
                McMsgAccountEntity msgAccount = mcMsgAccountService.getInfo(receive.getReceiveUserId());
                if (ObjectUtil.isNotNull(msgAccount)) {
                    String approveType = String.valueOf(msgAccount.getApproveType());
                    String webhookType = String.valueOf(msgAccount.getWebhookType());
                    String webhookAddress = msgAccount.getWebhookAddress();
                    String content = taskMsgContent.getContent();
                    JSONObject retJson = new JSONObject();
                    switch (webhookType) {
                        case "1":
                            //钉钉
                            if ("1".equals(approveType)) {
                                retJson = WebHookUtil.sendDDMessage(webhookAddress, content);
                            } else if ("2".equals(approveType)) {
                                retJson = WebHookUtil.sendDingDing(webhookAddress, msgAccount.getBearer(), content);
                            }
                            break;
                        case "2":
                            if ("1".equals(approveType)) {
                                retJson = WebHookUtil.callWeChatBot(webhookAddress, content);
                            }
                            break;
                        default:
                            break;
                    }

                    /**
                     * 处理发送结果
                     */
                    if ("ok".equals(retJson.getString("errmsg"))) {
                        // 成功
                        String errMsg = (isAgain ? "重发成功：" : "下发成功：") + JSONObject.toJSONString(retJson);
                        this.settingResult(isAgain, receive, 1, errMsg, currentUserInfo);
                    } else {
                        // 失败
                        String errMsg = (isAgain ? "重发失败：" : "下发失败：") + JSONObject.toJSONString(retJson);
                        this.settingResult(isAgain, receive, 2, errMsg, currentUserInfo);
                    }
                } else {
                    // 无效的接收人
                    String errMsg = (isAgain ? "重发失败：" : "下发失败：") + "无效的接收人，未指定有效的webhook配置";
                    this.settingResult(isAgain, receive, 2, errMsg, currentUserInfo);
                }
            } else {
                // 空的接收人
                String errMsg = (isAgain ? "重发失败：" : "下发失败：") + "空的接收人，未指定webhook配置";
                this.settingResult(isAgain, receive, 2, errMsg, currentUserInfo);
            }
        }
        mcTaskMsgReceiveService.updateBatchById(receiveListS);
    }

    private void settingResult(boolean isAgain, McTaskMsgReceiveEntity receive, Integer enabledMark, String remark,
                               UserInfo currentUserInfo) {
        if (!isAgain) {
            receive.setEnabledMark(enabledMark); // 渠道消息下发状态，0待下发、1下/重发成功、2下/重发失败
            receive.setSendTime(new Date());
            receive.setSendUserId(currentUserInfo.getUserId());
            receive.setDescription(remark);
            receive.setAutoAgainNum(0);
        } else {
            receive.setEnabledMark(enabledMark); // 渠道消息下发状态，0待下发、1下/重发成功、2下/重发失败
            receive.setAgainTime(new Date());
            receive.setAgainUserId(currentUserInfo.getUserId());
            receive.setDescription(remark);
            if (ObjectUtil.isEmpty(receive.getAutoAgainNum())) {
                receive.setAutoAgainNum(0);
            } else {
                receive.setAutoAgainNum(receive.getAutoAgainNum() + 1);
            }
        }
    }
}