package com.bringspring.system.base.util.jobutil;

import lombok.extern.slf4j.Slf4j;
import org.quartz.*;
import org.quartz.impl.StdSchedulerFactory;

import java.util.*;

/**
 * @author RKKJ开发平台组
 * @version V1.0.0
 * @copyright 荣科科技股份有限公司
 * @date 2021/3/12 15:31
 */
@Slf4j
public class JobManager {

    private SchedulerFactory schedulerFactory = new StdSchedulerFactory();

    // 触发器命名前缀
    private final String TRIGGER_PREFIX = "tri_";
    // 多触发器分隔符
    public static final String CRON_DELIMITER = ";";

    /**
     * 新增任务 - 支持单个和多个触发器
     */
    public void addJob(String jobName, String jobGroupName, String cron, JobDataMap jobDataMap, Date startDate, Date endDate,
                       Class<? extends Job> jobClass) {
        this.addJobInternal(jobName, jobGroupName, jobClass, cron, jobDataMap, startDate, endDate);
    }

    /**
     * 修改任务 - 支持多触发器更新
     */
    public void updateJob(String jobName, String jobGroupName, String cron, JobDataMap jobDataMap, Date startDate, Date endDate) {
        this.updateJobInternal(jobName, jobGroupName, cron, jobDataMap, startDate, endDate);
    }

    /**
     * 移除任务 - 移除所有关联的触发器
     */
    public void removeJob(String jobName, String jobGroupName) {
        this.removeJobWithAllTriggers(jobName, jobGroupName);
    }

    /**
     * 执行一次任务
     */
    public void startJobNow(String jobName, String jobGroupName, JobDataMap jobDataMap, Class<? extends Job> jobClass) {
        try {
            Scheduler sched = schedulerFactory.getScheduler();

            String triggerName = generateUniqueTriggerName(jobName, jobGroupName, "once");

            JobDetail jobDetail;
            if (jobDataMap != null) {
                jobDetail = JobBuilder.newJob(jobClass).withIdentity(jobName, jobGroupName).setJobData(jobDataMap).build();
            } else {
                jobDetail = JobBuilder.newJob(jobClass).withIdentity(jobName, jobGroupName).build();
            }

            Trigger trigger = TriggerBuilder.newTrigger()
                    .withIdentity(triggerName, jobGroupName)
                    .startNow()
                    .build();

            sched.scheduleJob(jobDetail, trigger);

            if (!sched.isShutdown()) {
                sched.start();
                log.info("立即执行一次任务成功: {}.{}", jobGroupName, jobName);
            }
        } catch (Exception e) {
            log.error("立即执行一次任务失败: {}.{} - {}", jobGroupName, jobName, e.getMessage());
        }
    }

    /**
     * 添加任务 - 支持多个Cron表达式（内部实现）
     */
    private void addJobInternal(String jobName, String jobGroupName, Class<? extends Job> jobClass,
                                String cron, JobDataMap jobDataMap, Date startDate, Date endDate) {
        try {
            Scheduler sched = schedulerFactory.getScheduler();

            JobKey jobKey = new JobKey(jobName, jobGroupName);
            if (sched.checkExists(jobKey)) {
                log.warn("任务已存在: {}.{}", jobGroupName, jobName);
                return;
            }

            JobDetail jobDetail = createJobDetail(jobName, jobGroupName, jobClass, jobDataMap);

            Set<Trigger> triggers = parseCronExpressions(jobName, jobGroupName, cron, jobDataMap, startDate, endDate);

            if (triggers.isEmpty()) {
                log.error("未找到有效的Cron表达式: {}", cron);
                return;
            }

            // 使用Set来避免类型问题
            sched.scheduleJob(jobDetail, triggers, true);

            if (!sched.isShutdown()) {
                sched.start();
                log.info("添加定时任务成功: {}.{}, 触发器数量: {}", jobGroupName, jobName, triggers.size());
            }
        } catch (Exception e) {
            log.error("添加定时任务失败: {}.{} - {}", jobGroupName, jobName, e.getMessage());
        }
    }

    /**
     * 移除任务及其所有触发器
     */
    private void removeJobWithAllTriggers(String jobName, String jobGroupName) {
        try {
            Scheduler sched = schedulerFactory.getScheduler();
            JobKey jobKey = new JobKey(jobName, jobGroupName);

            if (!sched.checkExists(jobKey)) {
                log.warn("任务不存在: {}.{}", jobGroupName, jobName);
                return;
            }

            List<? extends Trigger> triggers = sched.getTriggersOfJob(jobKey);
            for (Trigger trigger : triggers) {
                sched.pauseTrigger(trigger.getKey());
            }

            boolean deleted = sched.deleteJob(jobKey);

            if (deleted) {
                log.info("移除任务成功: {}.{}, 移除触发器数量: {}", jobGroupName, jobName, triggers.size());
            }
        } catch (Exception e) {
            log.error("移除定时任务失败: {}.{} - {}", jobGroupName, jobName, e.getMessage());
        }
    }

    /**
     * 修改任务 - 重新创建所有触发器（内部实现）
     */
    private void updateJobInternal(String jobName, String jobGroupName, String cron,
                                   JobDataMap jobDataMap, Date startDate, Date endDate) {
        try {
            Scheduler sched = schedulerFactory.getScheduler();
            JobKey jobKey = new JobKey(jobName, jobGroupName);

            if (!sched.checkExists(jobKey)) {
                log.warn("任务不存在，无法更新: {}.{}", jobGroupName, jobName);
                return;
            }

            // 获取现有触发器
            List<? extends Trigger> oldTriggers = sched.getTriggersOfJob(jobKey);

            // 移除所有旧触发器
            Set<TriggerKey> triggerKeys = new HashSet<>();
            for (Trigger trigger : oldTriggers) {
                triggerKeys.add(trigger.getKey());
                sched.unscheduleJob(trigger.getKey());
            }

            // 创建新触发器
            Set<Trigger> newTriggers = parseCronExpressions(jobName, jobGroupName, cron, jobDataMap, startDate, endDate);

            // 重新关联触发器
            for (Trigger trigger : newTriggers) {
                sched.scheduleJob(trigger);
            }

            log.info("更新任务成功: {}.{}, 旧触发器: {}, 新触发器: {}",
                    jobGroupName, jobName, oldTriggers.size(), newTriggers.size());

        } catch (Exception e) {
            log.error("修改定时任务失败: {}.{} - {}", jobGroupName, jobName, e.getMessage());
        }
    }

    /**
     * 解析Cron表达式 - 支持单个和多个
     */
    private Set<Trigger> parseCronExpressions(String jobName, String jobGroupName, String cron,
                                              JobDataMap jobDataMap, Date startDate, Date endDate) {
        Set<Trigger> triggers = new HashSet<>();

        if (cron == null || cron.trim().isEmpty()) {
            return triggers;
        }

        String[] cronExpressions = cron.split(CRON_DELIMITER);

        for (int i = 0; i < cronExpressions.length; i++) {
            String singleCron = cronExpressions[i].trim();
            if (singleCron.isEmpty()) continue;

            try {
                String triggerName = generateUniqueTriggerName(jobName, jobGroupName, String.valueOf(i));

                // 使用明确的TriggerBuilder类型，避免泛型问题
                TriggerBuilder<? extends Trigger> triggerBuilder = TriggerBuilder.newTrigger()
                        .withIdentity(triggerName, jobGroupName)
                        .withSchedule(CronScheduleBuilder.cronSchedule(singleCron));

                if (startDate != null) {
                    triggerBuilder.startAt(startDate);
                } else {
                    triggerBuilder.startNow();
                }

                if (endDate != null) {
                    triggerBuilder.endAt(endDate);
                }

                if (jobDataMap != null) {
                    triggerBuilder.usingJobData(new JobDataMap(jobDataMap));
                }

                triggers.add(triggerBuilder.build());

            } catch (Exception e) {
                log.error("解析Cron表达式失败: {} - {}", singleCron, e.getMessage());
            }
        }

        return triggers;
    }

    /**
     * 创建JobDetail
     */
    private JobDetail createJobDetail(String jobName, String jobGroupName,
                                      Class<? extends Job> jobClass, JobDataMap jobDataMap) {
        JobBuilder jobBuilder = JobBuilder.newJob(jobClass)
                .withIdentity(jobName, jobGroupName);

        if (jobDataMap != null) {
            jobBuilder.setJobData(jobDataMap);
        }

        return jobBuilder.build();
    }

    /**
     * 生成唯一的触发器名称
     */
    private String generateUniqueTriggerName(String jobName, String jobGroupName, String suffix) {
        return TRIGGER_PREFIX + jobName + "_" + jobGroupName + "_" + suffix + "_" + System.currentTimeMillis();
    }

    /**
     * 获取任务的所有触发器信息
     */
    public List<String> getJobTriggersInfo(String jobName, String jobGroupName) {
        List<String> triggerInfos = new ArrayList<>();
        try {
            Scheduler sched = schedulerFactory.getScheduler();
            JobKey jobKey = new JobKey(jobName, jobGroupName);

            if (sched.checkExists(jobKey)) {
                List<? extends Trigger> triggers = sched.getTriggersOfJob(jobKey);
                for (Trigger trigger : triggers) {
                    if (trigger instanceof CronTrigger) {
                        CronTrigger cronTrigger = (CronTrigger) trigger;
                        triggerInfos.add(String.format("触发器: %s, Cron: %s",
                                trigger.getKey(), cronTrigger.getCronExpression()));
                    } else {
                        triggerInfos.add(String.format("触发器: %s, 类型: %s",
                                trigger.getKey(), trigger.getClass().getSimpleName()));
                    }
                }
            }
        } catch (Exception e) {
            log.error("获取任务触发器信息失败: {}.{} - {}", jobGroupName, jobName, e.getMessage());
        }
        return triggerInfos;
    }

    /**
     * 判断任务是否存在
     */
    public boolean isJobExists(String jobName, String jobGroupName) {
        try {
            Scheduler sched = schedulerFactory.getScheduler();
            JobKey jobKey = new JobKey(jobName, jobGroupName);
            return sched.checkExists(jobKey);
        } catch (Exception e) {
            log.error("判断任务是否存在失败: {}.{} - {}", jobGroupName, jobName, e.getMessage());
            return false;
        }
    }

    /**
     * 启动所有定时任务
     */
    public void startJobs() {
        try {
            Scheduler sched = schedulerFactory.getScheduler();
            sched.start();
            log.info("启动所有定时任务成功");
        } catch (Exception e) {
            log.error("启动所有定时任务失败: {}", e.getMessage());
            throw new RuntimeException(e);
        }
    }

    /**
     * 关闭所有定时任务
     */
    public void shutdownJobs() {
        try {
            Scheduler sched = schedulerFactory.getScheduler();
            if (!sched.isShutdown()) {
                sched.shutdown();
                log.info("关闭所有定时任务成功");
            }
        } catch (Exception e) {
            log.error("关闭所有定时任务失败: {}", e.getMessage());
            throw new RuntimeException(e);
        }
    }

    // 保留原始的单触发器方法，用于向后兼容

    /**
     * 移除任务（原始单触发器版本）
     */
    private void removeJob(String jobName, String jobGroupName, String triggerName, String triggerGroupName) {
        try {
            Scheduler sched = schedulerFactory.getScheduler();
            JobKey jobKey = JobKey.jobKey(jobName, jobGroupName);
            JobDetail jobDetail = sched.getJobDetail(jobKey);
            if (null != jobDetail) {
                TriggerKey triggerKey = TriggerKey.triggerKey(triggerName, triggerGroupName);
                sched.pauseTrigger(triggerKey);
                sched.unscheduleJob(triggerKey);
                sched.deleteJob(jobKey);
            }
        } catch (Exception e) {
            log.error("移除定时任务失败:{}", e.getMessage());
        }
    }

    /**
     * 修改任务（原始单触发器版本）
     */
    private void updateJob(String jobName, String jobGroupName, String triggerName, String triggerGroupName,
                           String cron, JobDataMap jobDataMap, Date startDate, Date endDate) {
        try {
            Scheduler sched = schedulerFactory.getScheduler();
            TriggerKey triggerKey = TriggerKey.triggerKey(triggerName, triggerGroupName);
            Trigger trigger = sched.getTrigger(triggerKey);
            if (trigger == null) {
                return;
            }

            // 检查是否是CronTrigger
            if (trigger instanceof CronTrigger) {
                CronTrigger cronTrigger = (CronTrigger) trigger;
                String oldTime = cronTrigger.getCronExpression();
                if (!oldTime.equalsIgnoreCase(cron)) {
                    TriggerBuilder<Trigger> triggerBuilder = TriggerBuilder.newTrigger();
                    if (jobDataMap != null) {
                        triggerBuilder.withIdentity(triggerName, triggerGroupName).usingJobData(jobDataMap);
                    } else {
                        triggerBuilder.withIdentity(triggerName, triggerGroupName);
                    }
                    triggerBuilder.startNow();
                    triggerBuilder.withSchedule(CronScheduleBuilder.cronSchedule(cron));
                    if (endDate != null) {
                        triggerBuilder.endAt(endDate);
                    }
                    Trigger newTrigger = triggerBuilder.build();
                    sched.rescheduleJob(triggerKey, newTrigger);
                }
            }
        } catch (Exception e) {
            log.error("修改定时任务失败:{}", e.getMessage());
        }
    }

    /**
     * 判断触发器是否存在（原始方法）
     */
    private boolean isTriKey(String jobName, String jobGroupName) {
        boolean flag = false;
        try {
            Scheduler sched = schedulerFactory.getScheduler();
            TriggerKey triggerKey = TriggerKey.triggerKey(TRIGGER_PREFIX + jobName, TRIGGER_PREFIX + jobGroupName);
            Trigger trigger = sched.getTrigger(triggerKey);
            if (trigger != null) {
                flag = true;
            }
        } catch (Exception e) {
            log.error("判断任务调度触发器是否存在:{}", e.getMessage());
        }
        return flag;
    }
}