package com.bringspring.system.msgcenter.service.chain;

import cn.hutool.core.collection.CollectionUtil;
import com.bringspring.common.util.StringUtils;
import com.bringspring.system.message.entity.SynThirdInfoEntity;
import com.bringspring.system.msgcenter.entity.McTaskMsgReceiveEntity;
import com.bringspring.system.msgcenter.enums.EnabledMarkEnum;
import com.bringspring.system.msgcenter.enums.MsgCenterCode;
import com.bringspring.system.msgcenter.service.context.FilterResult;
import com.bringspring.system.msgcenter.service.context.SendMessageContext;
import org.springframework.stereotype.Component;

import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * 业务消息接收人是否有效过滤器
 * 第三方信息校验过滤器（ThirdInfoFilter后进一步对绑定应用第三方信息有效的过滤，要求ThirdObjectId不为空）
 * <p>
 * 此处节点针对第三方信息表：是系统用户绑定的渠道应用是否有第三方应用ID。
 * 注意：此处是对消息中心接收人记录过滤后的扩展，即接收人拥有系统用户且给用户第三方应用发送消息的情况。
 * 适用渠道：企业微信、钉钉、微信公众号、微信小程序等适用此处策略。
 */
@Component
public class ThirdObjectIdFilter implements FilterProcessor {

    @Override
    public FilterResult process(SendMessageContext context) {
        /**
         * 在上一节点（ThirdInfoFilter给上下文添加了第三方信息）后，此处可以直接获取上下文中的第三方信息
         * 注意：责任链上一个节点已经将无效的记录过滤掉了，所以此处获取context.getValidList()
         */
        Map<Boolean, List<McTaskMsgReceiveEntity>> partitioned = context.getValidList().stream()
                .collect(Collectors.partitioningBy(r -> {
                    SynThirdInfoEntity info = context.getThirdInfoMap().get(r.getReceiveUserId());
                    return info != null && StringUtils.isNotEmpty(info.getThirdObjectId());
                }));

        List<McTaskMsgReceiveEntity> invalidList = partitioned.get(false);
        if (CollectionUtil.isNotEmpty(invalidList)) {
            // 分离有效和无效记录，并标注原因
            invalidList.stream().forEach(e -> {
                e.setEnabledMark(EnabledMarkEnum.FAIL.getCode());
                e.setDescription(MsgCenterCode.THIRD_PARTY_ID_EMPTY.getDesc() + "（" + e.getReceiveUserId() + "）");
            });
        }

        return new FilterResult(partitioned.get(true), invalidList);
    }
}
